/*
* Copyright (c) 2005-2007 Compuware Corporation and others.
* All rights reserved. This program and the accompanying
* materials are made available under the terms of the
* Eclipse Public License v1.0 which accompanies this
* distribution, and is available at:
* http://www.eclipse.org/legal/epl-v10.html
*
* Contributors:
*     Compuware Corporation - initial API and implementation
*
*/
package org.eclipse.cosmos.me.management.annotations;

import static java.lang.annotation.RetentionPolicy.RUNTIME;
import static java.lang.annotation.ElementType.FIELD;
import static java.lang.annotation.ElementType.METHOD;

import java.lang.annotation.Retention;
import java.lang.annotation.Target;

/**
 * Annotation used to indicate a metric. This annotation may be 
 * applied to any field with ManagedProperty applied, or to any
 * method that had ManagedPropertyGetter applied. 
 * 
 */
@Target({FIELD,METHOD})
@Retention(RUNTIME)
public @interface ManagedMetric {
	
	/**
	 *	Enumeration used to specify the value of changeType() in the annotation.
	 *
	 * 	COUNTER - the value of the metric is a monotonically increasing integer. 
	 *  Such a metric value increases by increments of �1� when a situational event occurs to the resource.
	 *
	 *	GUAGE - changes of the value of the metric are not constrained in the way 
	 *  changes to Counter metrics are constrained.

	 *  UNKNOWN - the change behavior for the value of the metric is not known or cannot be described
	 *  
	 */
	enum CHANGE_TYPE {COUNTER, GAUGE, UNKNOWN};
	
	/**
	 * Enumeration used to specify the value of timeScope() in the annotation.
	 * 
	 * INTERVAL - the value of a metric is collected over some time interval.
	 * 
	 * POINT_IN_TIME - the value of a metric is counted, collected, 
	 * or measured at a single instant in time.
	 * 
	 * SINCE_RESET - the value of the metric is collected since the last reset of a resource, or since the manageable 
	 * resource started collecting data for a metric.
	 *
	 */
	enum TIME_SCOPE {INTERVAL,POINT_IN_TIME,SINCE_RESET};
	
	
	/**
	 * Enumeration used to specify the value of gatheringTime() in the annotation.
	 * 
	 * ON_CHANGE - the value of a metric is updated whenever a change occurs to the quantity measured.
	 * 
	 * PERIODIC - the value of a metric is updated on a regularly scheduled basis.
	 * 
	 * ON_DEMAND - the value of a metric is updated when processing a request for the metric value.
	 * 
	 * UNKNOWN - it is unknown when the value of a metric is updated.
	 * 
	 */
	enum GATHERING_TIME {ON_CHANGE,PERIODIC,ON_DEMAND,UNKNOWN};
	
	/** 
	 * The namespace to be used for the operation. If no namespace is provided, 
	 * and the operation is declared as a member of a ManagedResource,
	 * then the ManagedResource annotation's namespace will be used. If no namespace
	 * is provided, and the operation is operation is declared as a member of a 
	 * ManagedResourceCapability, then the ManagedResourceCapability's namespace
	 * will be used.
	 */
	String namespace() default "";

	/**
	 * The local name for the metric group.
	 */
	String group() default "";
	
	/**
	 * The calculation interval for the metric, specified in the format of an XML schema duration.
	 * Required if timeScope is set to INTERVAL. Prohibited if the timeScope value is POINT_IN_TIME or
	 * SINCE_RESET. 
	 */
	String calulationInterval() default "";
	
	/**
	 * The change type of the metric. See CHANGE_TYPE for a discussion of valid values.
	 */
	CHANGE_TYPE changeType() default CHANGE_TYPE.UNKNOWN;

	/**
	 * The time scope of the metric. See TIME_SCOPE for a discussion of valid values. 
	 */
	TIME_SCOPE timeScope() default TIME_SCOPE.POINT_IN_TIME;
	
	/**
	 * The gathering time of the metric. See GATHERING_TIME for a discussion of valid values. 
	 */
	GATHERING_TIME gatheringTime() default GATHERING_TIME.UNKNOWN;
}
