/*******************************************************************************
 * Copyright (c) 2005, 2007 Compuware Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Compuware Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.me.management.common.util;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;

import org.eclipse.cosmos.me.management.annotations.ManagedResourceCapability;

public class BindingUtil {
	
	private static Annotation[] EMPTY_ANNOTATION_LIST = {};
	
	public static void extractFields(Class clazz, HashMap<String, Field> map){
		Field fields[] = clazz.getDeclaredFields();
		for(Field field: fields){
			if(map.get(field.getName()) == null)
				map.put(field.getName(), field);
		}
	}
	
	
	//Collection annotation methods
	public static void getAnnotationsFromHierarchy(Class clazz, HashMap<Class, Annotation> map){
		if(clazz == null) return;
		Annotation[] annotations = clazz.getAnnotations();
		for(Annotation annotation: annotations){
			if(map.get(annotation.annotationType()) == null)
				map.put(annotation.annotationType(), annotation);
		}
		getAnnotationsFromHierarchy(clazz.getSuperclass(), map);
	}
	
	public static void getAnnotationsFromInterface(Class clazz, HashMap<Class, Annotation> map){
		if(clazz == null) return;
		if(clazz.isInterface()){
			getAnnotationsFromHierarchy(clazz, map);
		}else{
			Class[] interfazes = clazz.getInterfaces();
			for(Class interfaze : interfazes){
				getAnnotationsFromHierarchy(clazz, map);		
			}
			getAnnotationsFromInterface(clazz.getSuperclass(), map);
		}
		return;
	}

		
	public static Annotation[] getAnnotations(Class clazz){
		HashMap<Class, Annotation> map = new HashMap<Class, Annotation>();
		getAnnotationsFromHierarchy(clazz, map);
		getAnnotationsFromInterface(clazz, map);
		if(map.isEmpty())
			return EMPTY_ANNOTATION_LIST;
		else 
			return map.values().toArray(EMPTY_ANNOTATION_LIST);
	}
	
	
	public static Annotation[] getAnnotations(Method method){
		HashMap<Class, Annotation> map = new HashMap<Class, Annotation>();
		Annotation[] annotations = method.getAnnotations();
		for(Annotation annotation: annotations){
			if(map.get(annotation.annotationType()) == null)
				map.put(annotation.annotationType(), annotation);
		}
		getMethodAnnotationsFromHierarchy(method, method.getDeclaringClass(), map);
		getMethodAnnotationsFromInterface(method, method.getDeclaringClass(), map);
		if(map.isEmpty())
			return EMPTY_ANNOTATION_LIST;
		else 
			return map.values().toArray(EMPTY_ANNOTATION_LIST);
	}
	
	public static Annotation[][] getParameterAnnotations(Method method){
		HashMap<Class, Annotation> maps[] = new HashMap[method.getParameterTypes().length];
		Annotation[][] annotationSet = method.getParameterAnnotations();
		for(int i= 0; i < annotationSet.length;i++ ){
			Annotation[] annotations = annotationSet[i];
			HashMap<Class, Annotation> map = maps[i] = new HashMap<Class, Annotation>();
			for(Annotation annotation: annotations){
				if(map.get(annotation.annotationType()) == null)
					map.put(annotation.annotationType(), annotation);
			}
		}
		getMethodParameterAnnotationsFromHierarchy(method, method.getDeclaringClass(), maps);
		getMethodParameterAnnotationsFromInterface(method, method.getDeclaringClass(), maps);
		
		ArrayList<Annotation[]> list = new ArrayList<Annotation[]>();
		for(HashMap<Class, Annotation> map: maps){
			list.add(map.values().toArray(new Annotation[]{}));
		}
		return list.toArray(new Annotation[][]{});
	}
	
	public static void getMethodAnnotationsFromHierarchy(Method method, Class clazz, HashMap<Class, Annotation> map){
		Method superMethod = getMethodFromSuper(method, clazz);
		Annotation ret = null;
		if(superMethod != null){
			Annotation[] annotations = superMethod.getAnnotations();
			for(Annotation annotation: annotations){
				if(map.get(annotation.annotationType()) == null)
					map.put(annotation.annotationType(), annotation);
			}
			getMethodAnnotationsFromHierarchy(method, superMethod.getDeclaringClass().getSuperclass(), map);
		}
	}

	public static void getMethodAnnotationsFromInterface(Method method, Class clazz, HashMap<Class, Annotation> map){
		if(clazz == null) return;
		if(clazz.isInterface()){
			Method iMethod;
			try {
				iMethod = clazz.getMethod(method.getName(), method.getParameterTypes());
				Annotation[] annotations = iMethod.getAnnotations();
				for(Annotation annotation: annotations){
					if(map.get(annotation.annotationType()) == null)
						map.put(annotation.annotationType(), annotation);
				}
				if(clazz.getAnnotation(ManagedResourceCapability.class) != null){
					Annotation annotation = clazz.getAnnotation(ManagedResourceCapability.class);
					if(map.get(annotation.annotationType()) == null)
						map.put(annotation.annotationType(), annotation);
				}
			} catch (SecurityException e) {
			} catch (NoSuchMethodException e) {
			}
		}
		Class[] interfazes = clazz.getInterfaces();
		for(Class interfaze : interfazes){
			getMethodAnnotationsFromInterface(method, interfaze, map);
		}
	}
	

	public static void getMethodParameterAnnotationsFromHierarchy(Method method, Class clazz, HashMap<Class, Annotation> maps[]){
		Method superMethod = getMethodFromSuper(method, clazz);
		Annotation ret = null;
		if(superMethod != null){
			Annotation[][] annotationSet = superMethod.getParameterAnnotations();
			for(int i= 0; i < annotationSet.length;i++ ){
				Annotation[] annotations = annotationSet[i];
				HashMap<Class, Annotation> map = maps[i] = new HashMap<Class, Annotation>();
				for(Annotation annotation: annotations){
					if(map.get(annotation.annotationType()) == null)
						map.put(annotation.annotationType(), annotation);
				}
			}
			getMethodParameterAnnotationsFromHierarchy(method, superMethod.getDeclaringClass().getSuperclass(), maps);
		}
	}

	public static void getMethodParameterAnnotationsFromInterface(Method method, Class clazz, HashMap<Class, Annotation> maps[]){
		if(clazz == null) return;
		if(clazz.isInterface()){
			Method iMethod;
			try {
				iMethod = clazz.getMethod(method.getName(), method.getParameterTypes());
				Annotation[][] annotationSet = iMethod.getParameterAnnotations();
				for(int i= 0; i < annotationSet.length;i++ ){
					Annotation[] annotations = annotationSet[i];
					HashMap<Class, Annotation> map = maps[i] = new HashMap<Class, Annotation>();
					for(Annotation annotation: annotations){
						if(map.get(annotation.annotationType()) == null)
							map.put(annotation.annotationType(), annotation);
					}
				}
			} catch (SecurityException e) {
			} catch (NoSuchMethodException e) {
			}
		}
		Class[] interfazes = clazz.getInterfaces();
		for(Class interfaze : interfazes){
			getMethodParameterAnnotationsFromInterface(method, interfaze, maps);
		}
	}
	
	
	public static Annotation[] getAnnotations(Field field){
		HashMap<Class, Annotation> map = new HashMap<Class, Annotation>();
		return EMPTY_ANNOTATION_LIST;
	}

	public static Annotation getAnnotation(Class annotation, Class clazz){
		Annotation ret = clazz.getAnnotation(annotation);
		if(ret != null) return ret;
		ret = getAnnotationFromHierarchy(annotation, clazz);
		if(ret != null) return ret;
		return getAnnotationFromInterface(annotation, clazz);
	}
	
	
	//Specific annotation methods
	public static Annotation getAnnotationFromInterface(Class annotation, Class clazz){
		if(clazz == null) return null;
		Annotation ret = clazz.getAnnotation(annotation);
		if(ret != null) return ret;
		Class[] interfazes = clazz.getInterfaces();
		for(Class interfaze : interfazes){
			ret = getAnnotationFromInterface(annotation, interfaze);
			if(ret != null) return ret;
		}
		return null;
	}
	
	public static Annotation getAnnotationFromHierarchy(Class annotation, Class clazz){
		if(clazz == null) return null;
		if(clazz.getAnnotation(annotation) != null) return clazz.getAnnotation(annotation);
		else {
			return getAnnotationFromHierarchy(annotation, clazz.getSuperclass());
		}
	}

	private static Method getMethodFromSuper(Method method, Class superClazz){
		if(superClazz == null) return null;
		try {
			return superClazz.getMethod(method.getName(), method.getParameterTypes());
		} catch (SecurityException e) {
		} catch (NoSuchMethodException e) {
			return getMethodFromSuper(method, superClazz.getSuperclass());
		}
		return null;
	}
	
	public static Annotation getMethodAnnotation(Class annotation, Method method){
		//check for annotation on immeditat method
		Annotation ret = method.getAnnotation(annotation);
		if(ret != null) return ret;
		ret = getMethodAnnotationFromHierarchy(method, annotation, method.getDeclaringClass().getSuperclass());
		if(ret != null) return ret;
		ret = getMethodAnnotationFromInterface(method, annotation, method.getDeclaringClass());
		return ret;
	}
	
	public static Annotation getMethodAnnotationFromHierarchy(Method method, Class annotation, Class clazz){
		Method superMethod = getMethodFromSuper(method, clazz);
		Annotation ret = null;
		if(superMethod != null){
			ret = superMethod.getAnnotation(annotation);
			if(ret == null){
				return getMethodAnnotationFromHierarchy(method, annotation, superMethod.getDeclaringClass().getSuperclass());
			}
		}
		return ret;
	}

	public static Annotation getMethodAnnotationFromInterface(Method method, Class annotation, Class clazz){
		if(clazz == null) return null;
		Annotation ret = null;
		if(clazz.isInterface()){
			Method iMethod;
			try {
				iMethod = clazz.getMethod(method.getName(), method.getParameterTypes());
				if(annotation.equals(ManagedResourceCapability.class)){
					ret = clazz.getAnnotation(annotation);
				}else{
					ret = iMethod.getAnnotation(annotation);
				}
				if(ret != null) return ret;
			} catch (SecurityException e) {
			} catch (NoSuchMethodException e) {
			}
		}
		Class[] interfazes = clazz.getInterfaces();
		for(Class interfaze : interfazes){
			ret = getMethodAnnotationFromInterface(method, annotation, interfaze);
			if(ret != null) return ret;
		}
		return null;
		
	}
	
	public static boolean equivalentMethod(Method iMethod, Method mMethod){
		boolean ok = true;
		if(!iMethod.getName().equals(mMethod.getName())) return false;
		if(iMethod.getReturnType().equals(mMethod.getReturnType())){
			Class[] iParms = iMethod.getParameterTypes();
			Class[] mParms = mMethod.getParameterTypes();
			if(iParms.length == mParms.length){
				int i = 0;
				for(Class pClass :iParms){
					if(pClass.equals(mParms[i++]))continue;
					ok = false;
					break;
				}
			}
		}
		return ok;
	}

}
