/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.dc.dataManager.impl;

import java.io.StringWriter;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Properties;

import org.apache.muse.core.Capability;
import org.apache.muse.util.uuid.RandomUuidFactory;
import org.apache.muse.util.xml.XmlUtils;
import org.apache.muse.ws.addressing.EndpointReference;
import org.apache.muse.ws.dm.muws.Description;
import org.apache.muse.ws.dm.muws.MuwsConstants;
import org.apache.muse.ws.resource.WsResource;
import org.apache.muse.ws.resource.basefaults.BaseFault;
import org.eclipse.cosmos.dc.broker.client.BrokerClient;
import org.eclipse.cosmos.dc.cmdbf.services.metadata.transform.QueryMetadataTransformer;
import org.eclipse.cosmos.dc.cmdbf.services.metadata.transform.input.artifacts.IQueryServiceMetadata;
import org.eclipse.cosmos.dc.dataManager.api.DataManagerPolicyAssertion;
import org.eclipse.cosmos.dc.dataManager.api.IDataManager;
import org.eclipse.cosmos.dc.dataManager.exception.DataManagerException;
import org.eclipse.cosmos.me.management.annotations.ComposableManagedCapabilitySet;
import org.eclipse.cosmos.me.management.annotations.ManagedFrameworkAutowire;
import org.eclipse.cosmos.me.management.annotations.ManagedProperty;
import org.eclipse.cosmos.me.management.binding.Binding;
import org.eclipse.cosmos.me.management.provisional.wsdm.capabilities.impl.DescriptionImpl;
import org.eclipse.cosmos.me.management.provisional.wsdm.PolicyAssertion;
import org.eclipse.cosmos.me.management.provisional.wsdm.PolicyAssertionProvider;
import org.eclipse.cosmos.dc.management.domain.client.ManagementDomainClient;

/**
 * The super class of data manager implementations.  
 * It provides implementation of data manager capabilities.  
 * 
 */
@ComposableManagedCapabilitySet(set={DescriptionImpl.class})
public abstract class AbstractDataManager implements IDataManager, PolicyAssertionProvider {

	@ManagedProperty(resourceID=true)
	private String identifier = "undefined";
	
	// for getting access to the WSDM resource (bug 209244)
	@ManagedFrameworkAutowire(name="WSDM")
    private Binding wsdmBinding;

    public void setWsdmBinding(Binding wsdmBinding){
            this.wsdmBinding = wsdmBinding;
    }

	/**
	 * Configuration entries for the data manager, loaded from a config file.
	 */
	protected Properties config = null;
	
	private boolean initialized = false;
	
	// Management domain EPR
	private EndpointReference domainEpr = null;
	
	// Data manager display name
	private String displayName = null;
	
	// Data manager description
	private String description = null;
	
	// Data manager EPR
	private EndpointReference dataManagerEPR = null;

	/**
	 * Gets the identifier of the data manager.  
	 * This value will become the resource ID of the resource.   
	 * @return
	 */
	public String getIdentifier() {
		if (config != null) {
			identifier = config.getProperty(CONFIG_DATA_MANAGER_RESOURCE_ID);
		} else {
			// Generate an ID if unspecified
			identifier = RandomUuidFactory.getInstance().createUUID();
		}

		return identifier;
	}
	
	/**
	 * Gets the display name of the data manager.
	 * 
	 * @return
	 */
	public String getDisplayName() {
		if (displayName == null && config != null) {
			displayName = config.getProperty(CONFIG_DATA_MANAGER_DISPLAY_NAME);
		}
		return displayName;
	}
	
	/**
	 * Gets the description of the data manager.
	 * 
	 * @return
	 */
	public String getDescription() {
		if (description == null && config != null) {
			description = config.getProperty(CONFIG_DATA_MANAGER_DESCRIPTION);
		}
		return description;
	}
	
	/**
	 * Called by bundle activator.  
	 */
	public void setConfigProperties(Properties config) {
		this.config = config; 
	}
	
	public void setManagementDomainEPR(EndpointReference domainEPR) {
		this.domainEpr = domainEPR;
	}
	
	protected String getConfigProperty(String key) {
		return (config != null) ? config.getProperty(key) : null;
	}
	
	/**
	 * Initialize values by reading configuration properties passed in by
	 * using setConfigProperties method.
	 * 
	 * @throws DataManagerException
	 */
	private void initialize() throws DataManagerException {
		if (initialized == false) {
	        
	        // get WSDM resource object of this capability
			Object resourceObject = wsdmBinding.getBindingForObject(this);
			WsResource resource = null;
	        if (resourceObject != null && resourceObject instanceof WsResource) {
	        	resource = (WsResource)resourceObject;
	        } else {
	        	throw new DataManagerException("Unable to access WsResource object of data manager " + this.getClass().getSimpleName() );
	        }

	    	// Get my own EPR
	    	dataManagerEPR = resource.getEndpointReference();
	    	
			// TODO: Log entry
			System.out.println( "Initializing " + this.getClass().getSimpleName() );
	    	System.out.println( " at EPR: " );
	    	System.out.println( dataManagerEPR.toString() );

	    	// Set resource caption (display name) and description. 
	    	// Get values from config file. 
	    	Capability c = resource.getCapability(MuwsConstants.DESCRIPTION_URI);
	    	Description d = (Description)c;

	    	displayName = config.getProperty(CONFIG_DATA_MANAGER_DISPLAY_NAME);
			description = config.getProperty(CONFIG_DATA_MANAGER_DESCRIPTION);
			try {
				d.setCaption(new String[] {displayName});
				d.setDescription(new String[] {description});    	    	
 			} catch (BaseFault e) {
				throw new DataManagerException("Error occurred during initialization: " + e.getMessage() );
			}
			
			// TODO: Log entry
			System.out.println( "Initialization completed for " + displayName );
	    	System.out.println( " description: " + description );
		}
	}
	
	public void register() throws DataManagerException {	
		// do initialization if not already done. 
		initialize();
		
		// TODO: Log entry
		System.out.println( "Registering " + this.getClass().getSimpleName() );
		System.out.println( " using domain: " );
		System.out.println( domainEpr.toString() );
		System.out.println( " with COSMOS broker...");
		
        // Get Broker EPR from management domain
        String[] brokerEPRs = null;
        try {
			ManagementDomainClient client = new ManagementDomainClient(domainEpr);
			brokerEPRs = client.getAllServices();
		} catch (Exception e) {
			throw new DataManagerException("Error occurred while getting broker EPR from management domain.", e);
		}
		
		if (brokerEPRs.length == 0) {
			throw new DataManagerException("Data manager is not registered because no brokers registered at the management domain. ");
		}
		
		// register data manager with broker(s)
		for (int i=0; i<brokerEPRs.length; i++) {
			try {
		        EndpointReference brokerEPR = new EndpointReference(XmlUtils.createDocument(brokerEPRs[i]).getDocumentElement());
				BrokerClient brokerClient = new BrokerClient(brokerEPR);
				brokerClient.add(dataManagerEPR, null);
				
				// TODO: Log entry
		    	System.out.println( "DataManager '" + displayName + "' successfully registered with Broker:" );
		    	System.out.println( brokerEPR.toString() );
			} catch (Exception e) {
				throw new DataManagerException("Error occurred during registration with data broker.", e);
			}
		}
	}

	public void deregister() throws DataManagerException {
		System.out.println("deregister MDR from broker");
	}
	

	/**
	 * Gets the policy assertions of the Data Manager. 
	 *   
	 * Override this method to add your policy assertions (ie: for MDR metadata).
	 * The Contribution Manager will include them in your Data Manager WSDL
	 *  
	 * @return Collection<PolicyAssertion>
	 */
	public Collection<PolicyAssertion> getPolicyAssertions() {
		
		ArrayList<PolicyAssertion> policyAssertionList = new ArrayList<PolicyAssertion>();

		String metadataFilename = getConfigProperty("METADATA_FILE");
		
		if ( metadataFilename != null ){
	
			try {
				URL metadataURL = Thread.currentThread().getContextClassLoader().getResource( metadataFilename );
				
				// read metadata from file
				IQueryServiceMetadata queryServiceMetadata = QueryMetadataTransformer.transform(metadataURL.openStream());
				
				StringWriter sw = new StringWriter();
				queryServiceMetadata.toXML( sw, 0 );
				String metadata = sw.toString();
				
				// create a DataManagerPolicyAssertion containing the metadata
				PolicyAssertion metadataPolicyAssertion = new DataManagerPolicyAssertion( "SupportedMetadata", metadata, null);
				
				// add it to the array of policy assertions
				policyAssertionList.add(metadataPolicyAssertion);
				
			} catch (Exception e) {
				e.printStackTrace();
				System.out.println( "getPolicyAssertions Error with metadata " + metadataFilename + " : " + e.getMessage() );
			}
		}
		
		return policyAssertionList;
	}
	
}
