/*******************************************************************************
 * Copyright (c) 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.example.mdr.registration.internal.core;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;

import org.eclipse.cosmos.dc.cmdbf.services.common.INotificationHandlerFactory;
import org.eclipse.cosmos.dc.cmdbf.services.query.service.IQueryHandlerFactory;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IGraphElement;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IInstanceId;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IRelationship;
import org.eclipse.cosmos.dc.mdr.registration.AbstractFederatingCMDB;
import org.eclipse.cosmos.dc.mdr.registration.IFederatingCMDBServerConstants;
import org.eclipse.cosmos.example.mdr.registration.internal.query.CMDBQueryHandlerFactory;
import org.eclipse.cosmos.example.mdr.registration.internal.service.CMDBNotificationHandlerFactory;
import org.eclipse.cosmos.me.management.annotations.ManagedResource;

/**
 * The core class implementing the registration and query services 
 * of this federating CMDB.  The registration service works by only
 * accepting record types that are included in a configuration file.  
 * All registered items are kept in memory until deregistered.  The query
 * operation will read in the registered items/relationship and return 
 * a set matching the query constraint. 
 *
 * @author Ali Mehregani
 */
@ManagedResource
public class SampleFederatingCMDB extends AbstractFederatingCMDB
{
	/**
	 * A map of the items indexed by MDR id and local id
	 */
	private Map<String, Map<String, IGraphElement>> itemsMap;
	
	/**
	 * A map of the relationships indexed by MDR id and local id
	 */
	private Map<String, Map<String, IGraphElement>> relationshipsMap;
	
	/**
	 * A map of the relationships indexed by source instance id
	 */
	private Map<IInstanceId, List<IRelationship>> relationshipsMapBySource;
	
	
	/**
	 * Default constructor - used to perform initializations
	 */
	public SampleFederatingCMDB()
	{
		itemsMap = new Hashtable<String, Map<String, IGraphElement>>();
		relationshipsMap = new Hashtable<String, Map<String, IGraphElement>>();
		relationshipsMapBySource = new Hashtable<IInstanceId, List<IRelationship>>();
		
		Map<String, Object> init = new Hashtable<String, Object>();
		init.put(ISampleCMDBConstants.DATA_PROVIDER, this);
		
		getQueryOperation().initialize(init);
		getRegistrationOperation().initialize(init);
		getDeregistrationOperation().initialize(init);
	}
	
	
	/**
	 * @see org.eclipse.cosmos.dc.mdr.registration.AbstractFederatingCMDB#getNotificationHandlerFactory()
	 */
	@Override
	public INotificationHandlerFactory getNotificationHandlerFactory()
	{
		return CMDBNotificationHandlerFactory.getInstance();
	}

	
	/**
	 * @see org.eclipse.cosmos.dc.mdr.impl.AbstractMdr#getQueryHandlerFactory()
	 */
	@Override
	public IQueryHandlerFactory getQueryHandlerFactory()
	{
		return CMDBQueryHandlerFactory.getInstance();
	}

	
	/**
	 * Returns the item map indexed by MDR id and local id
	 * 
	 * @return The item map
	 */
	public Map<String, Map<String, IGraphElement>> retrieveItemsMap()
	{
		return itemsMap;
	}


	/**
	 * Returns the relationship map indexed by MDR id and 
	 * local id
	 * 
	 * @return The relationship map
	 */
	public Map<String, Map<String, IGraphElement>> retrieveRelationshipsMap()
	{		
		return relationshipsMap;
	}

	
	/**
	 * Returns the relationship map indexed by source instance
	 * id
	 * 
	 * @return The relationship map
	 */
	public Map<IInstanceId, List<IRelationship>> retrieveRelationshipBySource()
	{	
		return relationshipsMapBySource;
	}

	
	/**
	 * Add an element to this repository.  This methods has no
	 * effect if there is already an item/relationship added
	 * to the repository with the same MDR/local id
	 * 
	 * @param element The element to be added
	 * @param isItem True to indicate the element is an item; false
	 * otherwise
	 */
	public void addElement(IGraphElement element, boolean isItem)
	{
		IInstanceId[] instanceIds = element.getInstanceIds();
		for (int i = 0; i < instanceIds.length; i++)
		{
			// Update the itemsMap or relationshipsMap
			String mdrId = instanceIds[i].getMdrId().toString();
			String localId = instanceIds[i].getLocalId().toString();
			
			Map<String, Map<String, IGraphElement>> elements = isItem ? itemsMap : relationshipsMap;
			Map<String, IGraphElement> mdrElements = elements.get(mdrId);
			if (mdrElements == null)
			{
				mdrElements = new Hashtable<String, IGraphElement>();
				elements.put(mdrId, mdrElements);
			}
			IGraphElement graphElement = mdrElements.get(localId);
			if (graphElement == null)
			{
				mdrElements.put(localId, element);
			}
			
			// We need to also update the relationshipsMapBySource if
			// the element is a relationship
			if (isItem)
			{
				continue;
			}
			
			IRelationship relationship = (IRelationship)element;
			IInstanceId sourceInstanceId = relationship.getSourceId();
			List<IRelationship> storedRelationships = relationshipsMapBySource.get(sourceInstanceId);
			if (storedRelationships == null)
			{
				storedRelationships = new ArrayList<IRelationship>();
				relationshipsMapBySource.put(sourceInstanceId, storedRelationships);				
			}
			if (!storedRelationships.contains(relationship))
			{
				storedRelationships.add(relationship);
			}
		}
	}

	
	/**
	 * Removes an element from this repository.  This method has
	 * no effect if the element cannot be found
	 * 
	 * @param element The element to be removed
	 * @param isItem True to indicate the element is an item; false
	 * otherwise
	 */
	public void removeElement(IGraphElement element, boolean isItem)
	{
		IInstanceId[] instanceIds = element.getInstanceIds();
		for (int i = 0; i < instanceIds.length; i++)
		{
			String mdrId = instanceIds[i].getMdrId().toString();
			String localId = instanceIds[i].getLocalId().toString();
			Map<String, Map<String, IGraphElement>> elements = isItem ? itemsMap : relationshipsMap;
			
			// MDR id = '*'
			if (IFederatingCMDBServerConstants.ALL_ENTITIES.equals(mdrId))
			{
				elements.clear();
				if (!isItem)
				{
					relationshipsMapBySource.clear();
				}
			}
			// Otherwise MDR id is a constant
			else
			{
				Map<String, IGraphElement> mdrElements = elements.get(mdrId);
				if (mdrElements == null)
				{
					continue;
				}
				
				IGraphElement[] removedElements;

				// Local id = '*' 
				if (IFederatingCMDBServerConstants.ALL_ENTITIES.equals(localId))
				{
					removedElements = mdrElements.values().toArray(new IGraphElement[mdrElements.size()]);
					elements.remove(mdrId);					
				}
				// Otherwise local id is a constant
				else
				{					
					IGraphElement elementToBeRemoved = mdrElements.get(localId);
					removedElements = new IGraphElement[]{elementToBeRemoved};
					mdrElements.remove(localId);
					if (mdrElements.isEmpty())
					{
						elements.remove(mdrId);
					}					
				}
				
				// Update the relationships map indexed by source
				if (!isItem)
				{
					for (int j = 0; j < removedElements.length; j++)
					{
						IInstanceId sourceInsntanceId = ((IRelationship)removedElements[i]).getSourceId();
						relationshipsMapBySource.remove(sourceInsntanceId);						
					}					
				}
			}
			
			
		}
	}
}
