/*******************************************************************************
 * Copyright (c) 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.example.mdr.registration.internal.core;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.eclipse.cosmos.dc.cmdbf.services.query.transform.response.artifacts.IEdges;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.response.artifacts.INodes;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.response.artifacts.IQueryServiceElementCollection;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.response.artifacts.QueryOutputArtifactFactory;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IAdditionalRecordType;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IGraphElement;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IInstanceId;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IItem;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IRecord;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IRelationship;
import org.eclipse.cosmos.dc.mdr.registration.IFederatingCMDBServerConstants;

/**
 * Contains common utility methods used by multiple classes
 * 
 * @author Ali Mehregani
 */
public class SampleFederatingCMDBUtil
{
	/**
	 * Constants used to indicate the types of attributes
	 * associated with an graph element
	 */
	private static final int RECORDS = 0x00;
	private static final int ADDITIONAL_RECORDS = 0x01;
	private static final int INSTANCE_IDS = 0x02;
	
	
	public static IQueryServiceElementCollection handleInstanceIdConstraint(SampleFederatingCMDB federatingCMDB, IQueryServiceElementCollection context, IInstanceId[] instanceId, boolean isItem)
	{		
		Map<String, Map<String, IGraphElement>> graphElements = isItem ? federatingCMDB.retrieveItemsMap() : federatingCMDB.retrieveRelationshipsMap();	
		List<IGraphElement> currentResult = context.isStartingContext() ? null : createGraphElementList(context.getElements());		
		
		
		// For each instance id
		for (int i = 0; i < instanceId.length; i++)
		{
			// Build the context if it hasn't already been built
			if (currentResult == null)
			{
				currentResult = SampleFederatingCMDBUtil.findMatchedGraphElements(instanceId[i], graphElements);
			}
			// Otherwise the instance ids are applied using an AND logic
			else
			{
				List<IGraphElement> newList = new ArrayList<IGraphElement>();
				
				// Walk through the current result and eliminate the ones that don't have 
				// the current instance id
				for (int j = 0, currentResultCount = currentResult.size(); j < currentResultCount; j++)
				{
					IInstanceId[] instanceIds = currentResult.get(j).getInstanceIds();
					if (contains(instanceId[i], instanceIds))
					{
						newList.add(currentResult.get(j));
					}
				}
				currentResult = newList;
				
				
//				currentResult = findMutualSet (
//									currentResult,
//									SampleFederatingCMDBUtil.findMatchedGraphElements(instanceId[i], graphElements), isItem);
			}
			
			if (currentResult.isEmpty())
			{
				break;
			}
		}
		
		IQueryServiceElementCollection newContext = isItem ? 
					QueryOutputArtifactFactory.getInstance().createNodes(context.getId()) : 
					QueryOutputArtifactFactory.getInstance().createEdges(context.getId());
		// Add the result to the context
		for (int i = 0, itemCount = currentResult.size(); i < itemCount; i++)
		{
			if (isItem)
			{
				((INodes)newContext).addItem((IItem)currentResult.get(i));
			}
			else
			{
				((IEdges)newContext).addRelationship((IRelationship)currentResult.get(i));
			}			
		}
		
		return newContext;
	}
	
	
	private static boolean contains(IInstanceId instanceId, IInstanceId[] instanceIds)
	{
		String mdrId = instanceId.getMdrId().toString();
		String localId = instanceId.getLocalId().toString();
		for (int i = 0; i < instanceIds.length; i++)
		{
			if (matched(mdrId, instanceIds[i].getMdrId().toString()) &&
				matched(localId, instanceIds[i].getLocalId().toString()))
			{
				return true;
			}
		}
		return false;
	}


	private static boolean matched(String str1, String str2)
	{
		if (IFederatingCMDBServerConstants.ALL_ENTITIES.equals(str1))
		{
			return true;
		}
		return str1.equals(str2);
	}


	/**
	 * The following method returns true if one of the two conditions
	 * below is satisfied:
	 * <ul>
	 * 	<li> matchAny = false AND the elements in array1 and array2 are
	 *  	 the same but possibly in different order
	 *  </li>
	 *  <li> matchAny = true AND there is one element in array1 that is 
	 *       the same as one element in array2.
	 *  </li>
	 * </ul>
	 * If neither one of the two conditions are met, then false will be
	 * returned
	 * 
	 * @param array1 The first array
	 * @param array2 The second array
	 * @param matchAny Indicates subset equality vs. entire equality
	 * @return true if array1 is equal to array2 based on the description
	 * provided above; false otherwise
	 */
	public static boolean isEqual(Object[] array1, Object[] array2, boolean matchAny)
	{		
		if (!matchAny && array1.length != array2.length)
		{
			return false;
		}
		
		for (int i = 0; i < array1.length; i++)
		{
			boolean found = false;
			for (int j = 0; j < array2.length; j++)
			{
				if (array1[i].equals(array2[j]))
				{
					found = true;
					break;
				}
			}
			
			if (!matchAny && !found)
			{
				return false;
			}
			else if (matchAny && found)
			{
				return true;
			}
		}

		return !matchAny;
	}
	
	public static List<IGraphElement> createGraphElementList(IGraphElement[] elements)
	{
		List<IGraphElement> list = new ArrayList<IGraphElement>();
		for (int i = 0; i < elements.length; i++)
		{
			list.add(elements[i]);
		}
		return list;
	}


	/**
	 * Find and return the mutual set between list1 and list2
	 *  
	 * @param list1 The first list
	 * @param list2 The second list
	 * @return The mutual set between list1 and list2
	 */
	public static List<IGraphElement> findMutualSet(List<IGraphElement> list1, List<IGraphElement> list2, boolean isItem)
	{
		List<IGraphElement> result = new ArrayList<IGraphElement>();
		for (int i = 0, list1Count = list1.size(); i < list1Count; i++)
		{
			IGraphElement graphElement1 = list1.get(i);
			for (int j = 0, list2Count = list2.size(); j < list2Count; j++)
			{	
				IGraphElement graphElement2 = list2.get(j);
				if (graphElement1.equals(graphElement2))
				{					
					result.add(clone(graphElement1, graphElement2, isItem));
				}
			}
		}
		
		return result;		
	}

	
	/**
	 * Returns true iff graphElement1 is equal to graphElement1.  Only the records
	 * and additional record type of the two graph elements are compared.
	 * 
	 * @param graphElement1 The first graph element
	 * @param graphElement2 The second graph element
	 * @return true if graphElement1 is equal to graphElement2; false otherwise
	 */
	public static boolean isEqual(IGraphElement graphElement1, IGraphElement graphElement2)
	{
		return 	SampleFederatingCMDBUtil.isEqual(graphElement1.getRecords(), graphElement2.getRecords(), false) &&
				SampleFederatingCMDBUtil.isEqual(graphElement1.getAdditionalRecordTypes(), graphElement2.getAdditionalRecordTypes(), false);
	}


	/**
	 * Constructs an graph element from the two graph elements passed in by
	 * cloning the records and additional record types of the first
	 * graph element and adding the instance ids of both graph elements
	 * 
	 * @param graphElement1 The first graph element
	 * @param graphElement2 The second graph element
	 * @return An graph element that is a clone of the two graph elements passed in
	 */
	public static IGraphElement clone(IGraphElement graphElement1, IGraphElement graphElement2, boolean isItem)
	{
		IGraphElement graphElement = isItem ? QueryOutputArtifactFactory.getInstance().createItem() : QueryOutputArtifactFactory.getInstance().createRelationship();
		addAttribute(graphElement, RECORDS, graphElement1.getRecords());
		addAttribute(graphElement, ADDITIONAL_RECORDS, graphElement1.getAdditionalRecordTypes());
		
		IInstanceId[] instanceIds1 = graphElement1.getInstanceIds();
		IInstanceId[] instanceIds2 = graphElement2.getInstanceIds();
		
		List<IInstanceId> finalInstanceIdList = new ArrayList<IInstanceId>();
		for (int i = 0; i < instanceIds1.length; i++)
		{
			finalInstanceIdList.add(instanceIds1[i]);
		}
		
		for (int i = 0; i < instanceIds2.length; i++)
		{
			if (!finalInstanceIdList.contains(instanceIds2[i]))
			{
				finalInstanceIdList.add(instanceIds2[i]);
			}
		}
		
		addAttribute(graphElement, INSTANCE_IDS, finalInstanceIdList.toArray(new IInstanceId[finalInstanceIdList.size()]));
		
		return graphElement;
	}


	/**
	 * A generic helper method used to adding attributes to the 
	 * graph element passed in.
	 * 
	 * @param graphElement The graph element that the attributes will be added to
	 * @param attributeType The type of the attribute being added (possible
	 * fields are indicated by the following constants):
	 * <ul>
	 * 	<li> {@link #ADDITIONAL_RECORDS} </li>
	 *  <li> {@link #RECORDS} </li>
	 *  <li> {@link #INSTANCE_IDS} </li>
	 * </ul> 
	 * @param attributes The actual attributes being added
	 */
	public static void addAttribute(IGraphElement graphElement, int attributeType, Object[] attributes)
	{
		if (attributes == null)
		{
			return;
		}
		
		for (int i = 0; i < attributes.length; i++)
		{
			switch (attributeType)
			{
				case RECORDS:
					graphElement.addRecord((IRecord)attributes[i]);
					break;
				case ADDITIONAL_RECORDS:
					graphElement.addAdditionalRecordType((IAdditionalRecordType)attributes[i]);
					break;
				case INSTANCE_IDS:
					graphElement.addInstanceId((IInstanceId)attributes[i]);
					break;					
			}			
		}
	}
	
	
	/**
	 * Given an instance id and a map that's indexed by MDR id and local id,
	 * find the graph elements matching the instance id's values.  Asterisk ('*') is 
	 * used to indicate 'any' for the values of the MDR id/local id.
	 * 
	 * @param instanceId The instance id
	 * @param graphElements A map of the graph elements indexed by MDR id and local id
	 * @return Graph elements matching the instance id
	 */
	public static List<IGraphElement> findMatchedGraphElements(IInstanceId instanceId, Map<String, Map<String, IGraphElement>> graphElements)
	{
		String mdrId = instanceId.getMdrId().toString();
		String localId = instanceId.getLocalId().toString();
		Map<String, IGraphElement>[] matchingMDRs = findMatchingMDRs(mdrId, graphElements);
		List<IGraphElement> result = new ArrayList<IGraphElement>();
		
		// For each matching MDRs
		for (int j = 0; j < matchingMDRs.length; j++)
		{
			// Find matching graph elements			
			result.addAll(findMatchingGraphElements(matchingMDRs[j], localId));
		}			
		
		return result;
	}
	
	
	/**
	 * Finds and returns entries in the indexed graph element map that match
	 * the mdrId passed in.  Asterisk ('*') can be used to indicate
	 * any MDR id.
	 * 
	 * @param mdrId The MDR id
	 * @param graphElements A set of graph elements indexed by MDR id and local id
	 * @param clear If set, clears the elements as they are discovered 
	 * @return Matched graph elements
	 */
	@SuppressWarnings("unchecked")
	public static Map<String, IGraphElement>[] findMatchingMDRs(String mdrId, Map<String, Map<String, IGraphElement>> graphElements)
	{
		if (IFederatingCMDBServerConstants.ALL_ENTITIES.equals(mdrId))
		{	
			return graphElements.values().toArray(new Map[graphElements.size()]);
		}
		
		Map<String, IGraphElement> matchedMDRs = graphElements.get(mdrId);
		return matchedMDRs == null ? new Map[0] : new Map[]{matchedMDRs};	
	}

	
	/**
	 * Finds and returns the graph elements matching the local id passed
	 * in. Asterisk ('*') can be used to represent any local id.
	 *
	 * @param matchedMDRs A set of graph elements indexed by local ID
	 * @param localId The local id to look for
	 * @param clear If set, clears the elements as they are discovered 
	 * @return Matched graph elements
	 */
	public static List<IGraphElement> findMatchingGraphElements(Map<String, IGraphElement> matchedMDRs, String localId)
	{
		List<IGraphElement> matchedGraphElements = new ArrayList<IGraphElement>();
		if (IFederatingCMDBServerConstants.ALL_ENTITIES.equals(localId))
		{
			matchedGraphElements.addAll(matchedMDRs.values());
		}
		else
		{
			IGraphElement element = matchedMDRs.get(localId);
			if (element != null)
			{
				matchedGraphElements.add(element);
			}
		}
		return matchedGraphElements;
	}
}
