/**********************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.smlif.actions;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.filesystem.IFileStore;
import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.cosmos.rm.internal.smlif.SMLPlugin;
import org.eclipse.cosmos.rm.internal.smlif.common.SMLCommonUtil;
import org.eclipse.cosmos.rm.internal.validation.common.ISMLConstants;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidationMessages;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidatorUtil;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.osgi.util.NLS;
import org.eclipse.ui.IObjectActionDelegate;
import org.eclipse.ui.IWorkbenchPart;

/**
 * An abstract class that all other object action delegates are expected
 * to extend.
 * 
 * @author Ali Mehregani
 */
public abstract class AbstractAction implements IObjectActionDelegate
{
	/**
	 * The current selection
	 */
	private IStructuredSelection selection;
	
	/**
	 * The project name that will be used as the root context 
	 */
	private String projectName;
	
	/**
	 * List of the sml-if documents selected
	 */
	private List<String> smlifDocuments;
	
	/**
	 * The selected resources
	 */
	private StringBuffer smlModelUnitPaths;

	/**
	 * The root directory that should be used
	 */
	private IContainer rootDirectory;
	
	
	/** 
	 * @see org.eclipse.ui.IObjectActionDelegate#setActivePart(org.eclipse.jface.action.IAction, org.eclipse.ui.IWorkbenchPart)
	 */
	public void setActivePart(IAction action, IWorkbenchPart targetPart)
	{
	}

	
	/**
	 * @see org.eclipse.ui.IActionDelegate#selectionChanged(org.eclipse.jface.action.IAction, org.eclipse.jface.viewers.ISelection)
	 */
	public void selectionChanged(IAction action, ISelection selection)
	{
		this.selection = selection instanceof IStructuredSelection ? 
				(IStructuredSelection)selection : null;			
	}


	/**
	 * @return the selection
	 */
	public IStructuredSelection getSelection()
	{
		return selection;
	}
	
	protected boolean noValidSelection()
	{
		// Make sure something is selected
		if (selection == null || selection.isEmpty())
		{
			MessageDialog.openError(SMLPlugin.getDefault().getWorkbench().getActiveWorkbenchWindow().getShell(), 
					SMLValidationMessages.errorNoSelectionTitle, SMLValidationMessages.errorNoSelection);
			return true;
		}
		
		// Make sure that the resources are all from one project.  The selection
		// can't span multiple projects
		projectName = null;
		smlModelUnitPaths = new StringBuffer();
		initSmlifDocuments();
		rootDirectory = null;
		for (Iterator<?> selections = ((IStructuredSelection)selection).iterator(); selections.hasNext();)
		{
			Object currentSelection = selections.next();
			IResource currentResource = (IResource) currentSelection;
			if (projectName == null)
			{
				projectName = currentResource.getProject().getName();
			}
			else if (!projectName.equals(currentResource.getProject().getName()))
			{
				MessageDialog.openError(SMLPlugin.getDefault().getWorkbench().getActiveWorkbenchWindow().getShell(), 
						SMLValidationMessages.errorNoSelectionTitle, SMLValidationMessages.errorNoSelection);
				return true;
			}			
			
			if (rootDirectory == null)
			{
				rootDirectory = findParentDirectory(currentResource);
			}
			else if (!contains (rootDirectory, findParentDirectory(currentResource)))				
			{
				rootDirectory = findMutualContainer(rootDirectory, currentResource);
			}
			groupSelections(currentResource);
		}
		
		if (smlModelUnitPaths.length() > 0)
		{
			smlModelUnitPaths.deleteCharAt(smlModelUnitPaths.length() - 1);
		}
		return false;		
	}


	protected void initSmlifDocuments() {
		smlifDocuments = new ArrayList<String>();
	}

	private boolean contains(IContainer root, IContainer parent)
	{
		if (root.equals(parent))
			return true;
				
		while ((parent = parent.getParent()) != null && !root.equals(parent));
		return parent != null;
	}
	
	private IContainer findMutualContainer(IContainer root, IResource resource)
	{
		IContainer mutualContainer = findParentDirectory(resource);
		while (mutualContainer != null && !contains(mutualContainer, root))
		{
			mutualContainer = root.getParent();
		}
		return mutualContainer;
	}

	private IContainer findParentDirectory(IResource resource)
	{
		if (resource instanceof IContainer)
			return (IContainer)resource;
		return resource.getParent();
	}


	private void groupSelections(IResource resource)
	{
		if (resource instanceof IFile)
		{				
			try
			{
				int documentType = SMLValidatorUtil.identifyDocumentType((IFile)resource);
				if (ISMLConstants.TYPE_SMLIF == documentType)
				{
					smlifDocuments.add(resource.getLocation().toString());
				}
				else if (ISMLConstants.TYPE_UNKNOWN != documentType)
				{
					smlModelUnitPaths.append(resource.getLocation().toString()).append(",");
				}	
			} 
			catch (Exception e)
			{
				SMLCommonUtil.openErrorWithDetail(SMLValidationMessages.errorContentTypeTitle, 
					NLS.bind(SMLValidationMessages.errorContentType, ((IFile)resource).getFullPath()), e);
			} 		
		}
		else if (resource instanceof IContainer)
		{			
			try
			{
				IResource[] children = ((IContainer)resource).members();
				for (int i = 0; i < children.length; i++)
				{
					groupSelections (children[i]);
				}
			} catch (CoreException e)
			{				
				SMLPlugin.logError(e);
			}
			
		}

	}


	/**
	 * @return the rootDirectory
	 */
	public String getRootDirectory()
	{
		return rootDirectory == null ? projectName : rootDirectory.getLocation().toString();
	}
	
	public String getSmlModelUnitPaths()
	{
		return (smlModelUnitPaths == null) ? "" : smlModelUnitPaths.toString();
	}
	
	public List<String> getSmlifDocuments()
	{
		return smlifDocuments;
	}

}
