/**********************************************************************
 * Copyright (c) 2006, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.repository.operations;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.cosmos.rm.internal.repository.core.FileSystemSMLRepository;
import org.eclipse.cosmos.rm.internal.repository.core.IFileSystemSMLProperties;
import org.eclipse.cosmos.rm.internal.validation.common.AbstractValidationOutput;
import org.eclipse.cosmos.rm.internal.validation.common.FileOutput;
import org.eclipse.cosmos.rm.internal.validation.common.ISMLConstants;
import org.eclipse.cosmos.rm.internal.validation.common.IValidationConstants;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidationMessages;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidatorUtil;
import org.eclipse.cosmos.rm.internal.validation.common.SystemOutput;
import org.eclipse.cosmos.rm.internal.validation.common.AbstractValidationOutput.ValidationMessage;
import org.eclipse.cosmos.rm.internal.validation.common.AbstractValidationOutput.ValidationMessageFactory;
import org.eclipse.cosmos.rm.internal.validation.core.FoundationBuilder;
import org.eclipse.cosmos.rm.internal.validation.core.ISMLValidator;
import org.eclipse.cosmos.rm.internal.validation.core.ISchematronValidation;
import org.eclipse.cosmos.rm.internal.validation.core.IValidationListener;
import org.eclipse.cosmos.rm.internal.validation.core.IValidator;
import org.eclipse.cosmos.rm.internal.validation.core.IXMLValidator;
import org.eclipse.cosmos.rm.internal.validation.core.ValidationEvent;
import org.eclipse.cosmos.rm.internal.validation.core.ValidationFactory;
import org.eclipse.cosmos.rm.internal.validation.databuilders.DataBuilderRegistry;
import org.eclipse.cosmos.rm.internal.validation.smlvalidators.AcyclicValidator;
import org.eclipse.cosmos.rm.internal.validation.smlvalidators.IdentityConstraintValidator;
import org.eclipse.cosmos.rm.internal.validation.smlvalidators.ReferenceValidator;
import org.eclipse.cosmos.rm.internal.validation.smlvalidators.SchemaValidator;
import org.eclipse.cosmos.rm.internal.validation.smlvalidators.SchematronValidator;
import org.eclipse.cosmos.rm.internal.validation.smlvalidators.TargetValidator;
import org.eclipse.cosmos.rm.provisional.repository.core.ISMLRepository;
import org.eclipse.cosmos.rm.provisional.repository.core.SMLRepositoryFactory;
import org.eclipse.cosmos.rm.provisional.repository.exception.MissingRepositoryException;
import org.eclipse.cosmos.rm.provisional.repository.exception.RepositoryConnectionException;
import org.eclipse.cosmos.rm.provisional.repository.exception.RepositoryOperationException;
import org.eclipse.cosmos.rm.provisional.repository.operations.ISMLOperation;
import org.eclipse.cosmos.rm.provisional.repository.operations.ISMLValidateOperation;


/**
 * This is the starting point for a validation process.  Clients are encouraged to re-use the same instance
 * with the {@link this#initialize(Map)} and/or {@link this#setAttribute(String, String)} methods 
 * to configure an instance based on a specific setting.  
 * 
 * @author Ali Mehregani
 */
public class SMLMainValidator implements IValidationListener
{	
	private static final String OPTION_XML = "-xml";
	private static final String OPTION_SML = "-sml";
	private static final String OPTION_SCHEMATRON = "-schematron";
	private static final String OPTION_FILE = "-output";
	private static final String OPTION_CONTEXT = "-context";
	private static final String OPTION_CONFIGURATION = "-configuration";
	private static final String OPTION_SUPPRESS_WARNINGS = "-sw";
	
	/**
	 * The SML-IF schema location
	 */
	private static final String SML_IF_SCHEMA = "validation-resources/smlif.xsd";
	
	/**
	 * Stores the user specified context directory
	 */
	private static String contextDirectory;
	
	/**
	 * Stores the user specified configuration file
	 */
	private static String configurationFile;
	
	/**
	 * Indicates if warnings should be suppressed
	 */
	private static boolean suppressWarnings;
	private static IValidationOutputFactory validationOutputFactory;
	private static IValidationOutputFactory defaultValidationOutputFactory = new IValidationOutputFactory() {
		public AbstractValidationOutput createEclipseOutput() {
			return new SystemOutput();
		}
	};
	
	/**
	 * The attributes of this validation class
	 */
	private Map<String, Object> attributes;
	
	/**
	 * The progress monitor
	 */
	private IProgressMonitor monitor;
	
	
	/**
	 * The constructor
	 */
	public SMLMainValidator()
	{
		this.attributes = new Hashtable<String, Object>();
		setDefaultAttributes();
	}
	
	
	/**
	 * Sets default values for attributes
	 */
	protected void setDefaultAttributes()
	{
		/* Standalone environment */
		attributes.put(IValidator.ATTRIBUTE_ENV, IValidator.VALUE_ENV_STANDALONE);
		
		/* Default XML validator */
		attributes.put(IValidator.ATTRIBUTE_VALIDATION_XML, IXMLValidator.DEFAULT_VALIDATION_XML_LITERAL);
		
		/* Default SML validators */
		List<String> smlValidators = new ArrayList<String>();
		smlValidators.add(AcyclicValidator.class.getName());
		smlValidators.add(IdentityConstraintValidator.class.getName());
		smlValidators.add(TargetValidator.class.getName());
		smlValidators.add(SchemaValidator.class.getName());
		smlValidators.add(SchematronValidator.class.getName());
		smlValidators.add(ReferenceValidator.class.getName());
		attributes.put(IValidator.ATTRIBUTE_VALIDATION_SML, smlValidators);
		
		/* The SML-IF schema */
		attributes.put(IValidator.ATTRIBUTE_SML_IF_SCHEMA, SML_IF_SCHEMA);

		/* By default, continue processing after an error */
		attributes.put(IValidator.ATTRIBUTE_FAST_FAIL_POLICY, "false");
	}


	/**
	 * The initialize method is used to pass in multiple attributes
	 * at the same time
	 * 
	 * @param attributes The attributes.  The key and value pairs are expected 
	 * to be of type String
	 */
	public void initialize(Map<String, Object> attributes)
	{
		for (Iterator<String> keys = attributes.keySet().iterator(); keys.hasNext();)
		{
			String currentKey = keys.next();
			setAttribute(currentKey, (String)attributes.get(currentKey));
		}
	}


	/**
	 * Set an attribute of this type.  The key must be one of the ATTRIBUTE_* 
	 * constants defined by this class.
	 * 
	 * @param key The attribute key (see the ATTRIBUTE_* constants of this class)
	 * @param value The value of the attribute
	 */
	public void setAttribute(String key, String value)
	{
		attributes.put(key, value);
	}

	
	/**
	 * Start the validation process
	 */
	public boolean start()
	{
		// Clear the data builder registry
		DataBuilderRegistry.instance().clear();

		// Setup a common output stream if none has been setup 
		AbstractValidationOutput validationOutput = (AbstractValidationOutput)attributes.get(IValidator.ATTRIBUTE_OUTPUT);
		if (validationOutput == null)
		{
			validationOutput = createValidationOutput();
			attributes.put(IValidator.ATTRIBUTE_OUTPUT, validationOutput);			
		}
		if (validationOutput != null)
		{
			validationOutput.initialize(attributes);
		}
		
		Exception exception = null;
		monitor = (IProgressMonitor)attributes.get(IValidator.ATTRIBUTE_PROGRESS_MONITOR);
		monitor = monitor == null ? new NullProgressMonitor() : monitor; 
		
		try
		{
			// There are three phases to validate a SML document
			// Phase 1: create data structures that will be used in the sml validation phase.  Note: In this phase W3CSchema language validation will occur.		
			// DatastructureBuilder is always needed.
			FoundationBuilder datastructureBuilder = new FoundationBuilder();
	
			// Phase 2: sml validation phase
			IValidator[] smlValidations = ValidationFactory.createValidator(ISMLValidator.class, attributes.get(IValidator.ATTRIBUTE_VALIDATION_SML));
			
			// Phase 3: schematron validation phase
			// Note: schematron validation is now a builder
			IValidator[] schematronValidation = ValidationFactory.createValidator(ISchematronValidation.class, attributes.get(IValidator.ATTRIBUTE_VALIDATION_SCHEMATRON));
									
			IValidator[] completeSet = new IValidator[1 + smlValidations.length + schematronValidation.length];
			completeSet[0] = datastructureBuilder;
			System.arraycopy(smlValidations, 0, completeSet, 1, smlValidations.length);
			System.arraycopy(schematronValidation, 0, completeSet, 1 + smlValidations.length, schematronValidation.length);
			
			initValidators(completeSet);
			validate(completeSet);
			return validationOutput.getErrorCount() == 0;
		} 
		catch (InstantiationException e)
		{
			exception = e;
		} 
		catch (IllegalAccessException e)
		{
			exception = e;
		} 
		catch (ClassNotFoundException e)
		{
			exception = e;
		}
		finally
		{
			if (exception != null)
			{
				String message = exception.getMessage();
				validationOutput.reportMessage(ValidationMessageFactory.createErrorMessage(ValidationMessage.NO_LINE_NUMBER, SMLValidationMessages.errorValidatorInstantiation + (message == null ? IValidationConstants.EMPTY_STRING : message)));
			}
			validationOutput.close();
		}
		
		return false;
	}	
	
	/**
	 * Run the validators
	 * 
	 * @param validators
	 * @return false if any one validator fails
	 */
	private boolean validate(IValidator[] validators)
	{	
		boolean allValid = true;
		
		// Calculate how many validators must be traversed per one unit of validation work
		int workPerXValidators =  (int)Math.ceil((double)validators.length/IValidationConstants.VALIDATION_WORK);
		int counter = 0;
		int totalWorkReported = 0;
		
		for (int i = 0; i < validators.length; i++)
		{
			if (monitor.isCanceled())
			{
				return allValid;
			}
			
			counter++;
			if (validators[i] != null)
			{
				if (!validators[i].validate()) {
					allValid = false;
					if (fastFailPolicy()) {
						return false;
					}
				}
			}
			
			if (counter == workPerXValidators)
			{
				counter = 0;
				monitor.worked(1);
				totalWorkReported++;
			}
		}		
		
		int remainingWork = IValidationConstants.VALIDATION_WORK - totalWorkReported;
		if (remainingWork > 0)
		{
			monitor.worked(remainingWork);
		}
		return allValid;
	}
	
	private boolean fastFailPolicy() {
		String fastFail = (String) attributes.get(IValidator.ATTRIBUTE_FAST_FAIL_POLICY);
		if (fastFail != null) {
			return Boolean.valueOf(fastFail).booleanValue();
		}
		return false;
	}


	/**
	 * Initialize the validators
	 * @param validators
	 */
	private void initValidators(IValidator[] validators)
	{				
		for (int i = 0; i < validators.length; i++)
		{
			if (validators[i] != null)
			{
				validators[i].initialize(attributes);
				validators[i].addValidationListener(this);
			}
		}		
	}


	/**
	 * Create a validation output based on the environment that the validation 
	 * is being performed in.
	 * 
	 * @return An output stream
	 */
	protected AbstractValidationOutput createValidationOutput()
	{
		String environment = (String)attributes.get(IValidator.ATTRIBUTE_ENV);
		AbstractValidationOutput validationOutput = null;
		/* Eclipse */
		if (IValidator.VALUE_ENV_ECLIPSE.equals(environment))
		{
			validationOutput = createEclipseOutput();
		}
		/* Standalone */
		else if (IValidator.VALUE_ENV_STANDALONE.equals(environment))
		{
			validationOutput =  new FileOutput();
		}
		/* Unspecified - system output by default */
		else
		{
			validationOutput =  new SystemOutput();
		}
		return validationOutput;
	}


	protected AbstractValidationOutput createEclipseOutput() {
		return getEclipseValidationOutputFactory().createEclipseOutput();
	}


	public static IValidationOutputFactory getEclipseValidationOutputFactory() {
		if (validationOutputFactory == null) {
			return defaultValidationOutputFactory;
		}
		return validationOutputFactory;
	}
	
	public static void setEclipseValidationOutputFactory(IValidationOutputFactory factory) {
		validationOutputFactory = factory;
	}


	/**
	 * Returns the attributes
	 * 
	 * @return attributes
	 */
	public Map<String, Object> getAttributes()
	{
		return attributes;
	}
	
	
	/**
	 * The main method - Used to set the appropriate attributes based on the
	 * program arguments.  Expected format:
	 * org.eclipse.cosmos.rm.internal.validation.core.MainValidator ([option][option argument(s)])* <comma separated path(s) to SML-IF/SML document(s)>,
	 * where available option and arguments are:
	 * <ul>
	 * 	<li> -xml <a fully qualified class name> </li>
	 * 	<li> -sml <fully qualified class names separated by a comma> </li>
	 * 	<li> -schematron <a fully qualified class name> </li>
	 *  <li> -output <path to the output file> </li>
	 *  <li> -context <the context directory of the repository>  (the current working
	 *  director will be used as the context directory if this option is not specified)</li>
	 *  <li> -configuration <the path to the configuration file of the repository> (this
	 *  option takes precedence over the context option)</li>
	 * </ul>
	 * 
	 * @param args The program arguments
	 */
	public static void main (String[] args)
	{
		if (args.length <= 0)
		{
			printUsage();
			return;
		}
		
		int lastInx = -1, currentInx = 0;
		Map<String, Object> validationAttributes = new Hashtable<String, Object>();
		try
		{
			while (lastInx != currentInx)
			{
				lastInx = currentInx;
				currentInx = processOption(validationAttributes, args, currentInx);
			}
			
			
			if (currentInx != args.length - 1)
			{
				printUsage();
				return;
			}
			
			validationAttributes.put(IValidator.ATTRIBUTE_ENV, IValidator.VALUE_ENV_STANDALONE);
			if (System.getProperty("org.eclipse.cosmos.rm.repository") == null)
			{
				System.setProperty("org.eclipse.cosmos.rm.repository", FileSystemSMLRepository.class.getName());
			}
			
			ISMLRepository repository = SMLRepositoryFactory.createRepository();
			Map<String, Object> repositoryAttributes = new Hashtable<String, Object>();
			
			// The configuration file takes the highest precedence
			if (configurationFile != null)
			{
				repositoryAttributes.put(FileSystemSMLRepository.ATTRIBUTE_CONFIGURATION_PROPERTY, configurationFile);
			}
			// Use the context directory if the configuration file is not specified
			else if (contextDirectory != null)
			{
				repositoryAttributes.put(IFileSystemSMLProperties.ROOT_DIRECTORY, contextDirectory);
			}
			// Resort to the current working directory if the context directory is not specified
			else
			{
				repositoryAttributes.put(IFileSystemSMLProperties.ROOT_DIRECTORY, System.getProperty("user.dir"));
			}
			
			repository.connect(repositoryAttributes);
			ISMLOperation validationOperation = repository.getOperation(ISMLValidateOperation.ID);
			
			
			StringTokenizer resources = new StringTokenizer(args[currentInx], ",");
			StringBuffer smlModelUnits = new StringBuffer();
			
			if (suppressWarnings)
			{
				validationAttributes.put(IValidator.ATTRIBUTE_SUPPRESS_WARNINGS, Boolean.TRUE);
			}
			
			validationAttributes.put(IValidator.ATTRIBUTE_INPUT_TYPE, IValidator.VALUE_SML_IF);			
			while (resources.hasMoreTokens())
			{
				String currentResource = resources.nextToken();
				int documentType;
				try
				{
					String fullPath = currentResource;
					String contextDirectory = repository.getProperty(IFileSystemSMLProperties.ROOT_DIRECTORY, "");
					if (!currentResource.startsWith(contextDirectory))
					{
						fullPath = contextDirectory + (contextDirectory.endsWith("/") || contextDirectory.endsWith("\\") ? "" : "/") + 
									currentResource;
					}
					documentType = SMLValidatorUtil.identifyDocumentType(new File(fullPath));
					if (documentType == ISMLConstants.TYPE_SMLIF)
					{
						validationAttributes.put(IValidator.ATTRIBUTE_INSTANCE, currentResource);		
						validationOperation.setArguments(new Object[]{validationAttributes});
						validationOperation.run();		
					}
					else if (documentType != ISMLConstants.TYPE_UNKNOWN)
					{
						smlModelUnits.append(currentResource).append(",");
					}
				} 
				catch (IOException e)
				{					
					e.printStackTrace();
				} 
				catch (CoreException e)
				{					
					e.printStackTrace();
				}				
			}
			
			if (smlModelUnits.length() > 0)
			{
				smlModelUnits.deleteCharAt(smlModelUnits.length() - 1);
				validationAttributes.put(IValidator.ATTRIBUTE_INPUT_TYPE, IValidator.VALUE_SML_UNIT);
				validationAttributes.put(IValidator.ATTRIBUTE_INSTANCE, smlModelUnits.toString());
				validationOperation.setArguments(new Object[]{validationAttributes});
				validationOperation.run();		
			}
				
		}
		catch (IllegalArgumentException e)
		{
			printUsage();
		} 
		catch (MissingRepositoryException e)
		{			
			e.printStackTrace();
		} 
		catch (RepositoryConnectionException e)
		{		
			e.printStackTrace();
		} 
		catch (RepositoryOperationException e)
		{
			e.printStackTrace();
		}		
	}


	private static int processOption(Map<String, Object> attributes, String[] args, int inx) throws IllegalArgumentException
	{
		if (inx < 0 || inx >= args.length)
			return inx;
		
		String currentOption = args[inx];
		boolean xml = false;
		
		if ((xml = OPTION_XML.equals(currentOption)) || OPTION_SCHEMATRON.equals(currentOption))
		{	
			/* The next argument is expected to be the XML validator class */
			inx = addAttribute(attributes, args, ++inx, xml ? IValidator.ATTRIBUTE_VALIDATION_XML : IValidator.ATTRIBUTE_VALIDATION_SCHEMATRON);			
		}
		else if (OPTION_SML.equals(currentOption))
		{
			List<String> smlValidators = new ArrayList<String>();
			while (++inx < args.length - 1 && !args[inx].startsWith("-"))
			{
				StringTokenizer st = new StringTokenizer(args[inx]);
				while (st.hasMoreTokens())
				{
					smlValidators.add(st.nextToken());
				}
			}
			
			if (smlValidators.isEmpty())
				throw new IllegalArgumentException();
			
			attributes.put(IValidator.ATTRIBUTE_VALIDATION_SML, smlValidators);
		}
		else if (OPTION_FILE.equals(currentOption))
		{
			inx = addAttribute(attributes, args, ++inx, IValidator.ATTRIBUTE_FILE_OUTPUT);						
		}
		else if (OPTION_CONTEXT.equals(currentOption))
		{
			contextDirectory = args[++inx];
			inx++;
		}
		else if (OPTION_CONFIGURATION.equals(currentOption))
		{
			configurationFile = args[++inx];
			inx++;
		}
		else if (OPTION_SUPPRESS_WARNINGS.equals(currentOption))
		{
			suppressWarnings = true;
			inx++;
		}
		
		return inx;
	}


	private static int addAttribute(Map<String, Object> attributes, String[] args, int inx, String attribute) throws IllegalArgumentException
	{
		if (inx >= args.length || args[inx].startsWith("-"))
			throw new IllegalArgumentException();
		
		attributes.put(attribute, args[inx]);
		return inx + 1;
	}


	private static void printUsage()
	{
		System.err.println(SMLValidationMessages.errorIllegalArguments);
	}


	public void validationErrorOccurred(ValidationEvent event) {
		event.continueOnError = !fastFailPolicy();
	}
}
