/**********************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.artifacts;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;

import org.eclipse.cosmos.rm.internal.validation.common.SMLValidatorUtil;
import org.eclipse.cosmos.rm.internal.validation.databuilders.IdentityConstraintDataBuilder.IdentityConstraint;

/**
 * A data structure which stores information about the identity
 * constraints.  e.g. The declaration and the instances that require
 * applying the identity constraint to.
 * 
 * @author Ali Mehregani
 */
public class IdentityConstraintStructure
{		
	/**
	 * The declarations
	 * KEY = A {@link String} indicating the URI
	 * VALUE = A {@link Map}, where
	 * 	KEY = A {@link String} indicating the local name of the element
	 *  VALUE = An object of type {@link List} where each object contained is of type
	 *  {@link IdentityConstraint}
	 */
	private Map<String, Map<String, List<IdentityConstraint>>> declarations;
			
	/**
	 * The instances of nodes that require an identity constraint check
	 * KEY = A {@link String} indicating the document alias
	 * VALUE = A {@link List} containing elements of type int[], which identify the
	 * indices leading to the node that needs the identity constraint check. 
	 */
	private Map<String, List<int[]>> constrainedInstances;
	
	/**
	 * Constrained instances that don't have an alias
	 * KEY = A {@link String} indicating the index of the orphaned document
	 * VALUE = A {@link List} of elements containing objects of type int[], which identify
	 * the indices leading to the node that needs an identity constraint check.
	 */
	private Map<String, List<int[]>> orphanedConstrainedInstances;
	
	
	public IdentityConstraintStructure()
	{
		declarations = new Hashtable<String, Map<String, List<IdentityConstraint>>>();
		constrainedInstances = new Hashtable<String, List<int[]>>();
		orphanedConstrainedInstances = new Hashtable<String, List<int[]>>();
	}
	
	/**
	 * Adds a declaration to this structure
	 * 
	 * @param uri The uri of the element
	 * @param localName The local name of the element
	 * @param identityDecl The identity declaration
	 */
	@SuppressWarnings("unchecked")
	public void addDeclaration(String uri, String localName, IdentityConstraint identityDecl)
	{
		Map<String, List<IdentityConstraint>> declarationsPerURI = (Map<String, List<IdentityConstraint>>)SMLValidatorUtil.retrieveNestedMap(declarations, uri, true);
		List<IdentityConstraint> listOfConstraints = declarationsPerURI.get(localName);
		if (listOfConstraints == null)
		{
			listOfConstraints = new ArrayList<IdentityConstraint>();
			declarationsPerURI.put(localName, listOfConstraints);
		}
		listOfConstraints.add(identityDecl);
	}

	
	/**
	 * Adds a constrained instance to this structure
	 * 
	 * @param alias The alias of the document
	 * @param indices The indices leading to the node that has an identity constraint
	 */
	public void addConstrainedInstance(String alias, int[] indices)
	{			
		addToNestedList (constrainedInstances, alias, indices);
	}
	
	
	public void addOrphanedConstrainedInstance(int orphanInx, int[] indices)
	{
		addToNestedList (orphanedConstrainedInstances, String.valueOf(orphanInx), indices);
	}
	
	private void addToNestedList(Map<String, List<int[]>> map, String key, int[] value)
	{
		List<int[]> nestedList = (List<int[]>)map.get(key);
		if (nestedList == null)
		{
			nestedList = new ArrayList<int[]>();
			map.put(key, nestedList);
		}
		nestedList.add(value);
	}
	
	/**
	 * Retunrns true iff the element with the namespace, 'uri', and local name,
	 * 'localName' has an associated identity constraint.
	 *  
	 * @param uri The URI of the element
	 * @param localName The local name of the element
	 * @return true if the element has an associated identity constraint; false
	 * otherwise
	 */
	public boolean isConstrained(String uri, String localName)
	{
		return retrieveConstraint(uri, localName) != null;			
	}

	
	/**
	 * Retrieves the constraints based on the uri and the local name of
	 * an element.
	 * 
	 * @param uri The uri of an element
	 * @param localName The local name of an element
	 * @return The constraints associated with the element or null if no constraints
	 * are associated.
	 */
	public IdentityConstraint[] retrieveConstraint (String uri, String localName)
	{
		if (uri == null || localName == null)
			return null;
		
		Map<String, List<IdentityConstraint>> declarationsPerURI = declarations.get(uri);
		List<IdentityConstraint> constraints = declarationsPerURI == null ? null : declarationsPerURI.get(localName);
		return constraints == null ? null : constraints.toArray(new IdentityConstraint[constraints.size()]);
	}
	
	/**
	 * @return the constrainedInstances
	 */
	public Map<String, List<int[]>> getConstrainedInstances()
	{
		return constrainedInstances;
	}

	/**
	 * @return the declarations
	 */
	public Map<String, Map<String, List<IdentityConstraint>>> getDeclarations()
	{
		return declarations;
	}

	/**
	 * @return the orphanedConstrainedInstances
	 */
	public Map<String, List<int[]>> getOrphanedConstrainedInstances()
	{
		return orphanedConstrainedInstances;
	}

	/**
	 * @param constrainedInstances the constrainedInstances to set
	 */
	public void setConstrainedInstances(Map<String, List<int[]>> constrainedInstances)
	{
		this.constrainedInstances = constrainedInstances;
	}

	/**
	 * @param declarations the declarations to set
	 */
	public void setDeclarations(Map<String, Map<String, List<IdentityConstraint>>> declarations)
	{
		this.declarations = declarations;
	}

	/**
	 * @param orphanedConstrainedInstances the orphanedConstrainedInstances to set
	 */
	public void setOrphanedConstrainedInstances(Map<String, List<int[]>> orphanedConstrainedInstances)
	{
		this.orphanedConstrainedInstances = orphanedConstrainedInstances;
	}		
}