/**********************************************************************
 * Copyright (c) 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.databuilders;

import java.util.Map;
import java.util.Stack;

import javax.xml.namespace.QName;

import org.eclipse.cosmos.rm.internal.validation.artifacts.AbstractDeclaration;
import org.eclipse.cosmos.rm.internal.validation.artifacts.ConstraintNode;
import org.eclipse.cosmos.rm.internal.validation.artifacts.ElementLocation;
import org.eclipse.cosmos.rm.internal.validation.artifacts.GroupDeclaration;
import org.eclipse.cosmos.rm.internal.validation.common.ISMLConstants;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidatorUtil;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;

/**
 * An abstract declaration builder, containing common methods
 * used by concrete declaration builders 
 * 
 * @author Ali Mehregani
 */
public abstract class AbstractDeclarationBuilder<T> extends AbstractDataBuilder<T>
{
	/**
	 * Keeps track of the element notifications
	 */
	private Stack<String> elements;
	
	/**
	 * The current Declaration
	 */
	private AbstractDeclaration declaration;
	
	/**
	 * Indicates when the sequence element has been hit
	 */
	private boolean sequenceElementHit;
	
	
	public AbstractDeclarationBuilder()
	{
		elements = new Stack<String>();
	}
	
	/**
	 * Using the value of the attribute with namespace URI = uri and local name = localName.
	 * Attempt to use localName alone if the attribute can't be found with uri.
	 * 
	 * @param attributes The attributes
	 * @param uri The attribute's namespace
	 * @param localName the attribute's local name
	 * @return The value of a the attribute with the uri and local name passed in
	 */
	protected String retrieveValue (Attributes attributes, String uri, String localName)
	{
		String value = attributes.getValue(uri, localName);
		return value == null ? attributes.getValue(localName) : value;
	}
	
	
	/**
	 * Constructs and returns an object representing an 
	 * element declaration
	 * 
	 * @param attributes A set of attributes
	 * @return An object representing an element declaration
	 */
	protected ConstraintNode createElementDeclaration(Attributes attributes)
	{
		Map<String, String> prefixUriMap = getPrefixMap();
		String targetNamespace = getTargetNamespace();
		
		ConstraintNode targetNode = new ConstraintNode();
		QName localElementName = SMLValidatorUtil.toQName(prefixUriMap, targetNamespace, retrieveValue(attributes, ISMLConstants.SCHEMA_URI, ISMLConstants.NAME_ATTRIBUTE));
		if (localElementName != null)
		{
			targetNode.setName(localElementName.getLocalPart());
			targetNode.setUri(localElementName.getNamespaceURI());
		}
		else
		{
			targetNode.setUri(targetNamespace);
		}
		targetNode.setRef(SMLValidatorUtil.toQName(prefixUriMap, targetNamespace, retrieveValue(attributes, ISMLConstants.SCHEMA_URI, ISMLConstants.REF_ATTRIBUTE)));
		targetNode.setTargetElement(SMLValidatorUtil.toQName(prefixUriMap, targetNamespace, attributes.getValue(ISMLConstants.SML_URI, ISMLConstants.TARGETELEMENT_ATTRIBUTE)));
		targetNode.setTargetType(SMLValidatorUtil.toQName(prefixUriMap, targetNamespace, attributes.getValue(ISMLConstants.SML_URI, ISMLConstants.TARGETTYPE_ATTRIBUTE)));
		targetNode.setTargetRequired(attributes.getValue(ISMLConstants.SML_URI, ISMLConstants.TARGETREQUIRED_ATTRIBUTE));
		return targetNode;
	}
	
	
	private GroupDeclaration createGroupDeclaration(Attributes attributes)
	{
		Map<String, String> prefixUriMap = getPrefixMap();
		String targetNamespace = getTargetNamespace();
		
		GroupDeclaration groupDeclaration = new GroupDeclaration();
		groupDeclaration.setQName(SMLValidatorUtil.toQName(prefixUriMap, targetNamespace, retrieveValue(attributes, ISMLConstants.SCHEMA_URI, ISMLConstants.NAME_ATTRIBUTE)));
		groupDeclaration.setRef(SMLValidatorUtil.toQName(prefixUriMap, targetNamespace, retrieveValue(attributes, ISMLConstants.SCHEMA_URI, ISMLConstants.REF_ATTRIBUTE)));
		return groupDeclaration;
	}
	
	
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.AbstractDataBuilder#startElement(java.lang.String, java.lang.String, java.lang.String, org.xml.sax.Attributes)
	 */
	@Override
	public void startElement(String uri, String localName, String name, Attributes attributes) throws SAXException
	{		
		super.startElement(uri, localName, name, attributes);
		
		if (!ISMLConstants.SCHEMA_URI.equals(uri))
		{
			return;
		}		
		
		elements.push(localName);
		
		if (declaration != null)
		{
			// <xs:sequence
			if (ISMLConstants.SEQUENCE_ELEMENT.equals(localName))
			{
				sequenceElementHit = true;
			}
			
			// <xs:element
			else if (sequenceElementHit && ISMLConstants.ELEMENT_ELEMENT.equals(localName))
			{									
				declaration.getElementDeclarations().add(createElementDeclaration(attributes));
			}
			
			// <xs:group
			else if (sequenceElementHit && ISMLConstants.GROUP_ELEMENT.equals(localName))
			{
				declaration.getGroupDeclarations().add(createGroupDeclaration(attributes));
			}
		}
	}
	

	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.AbstractDataBuilder#endElement(java.lang.String, java.lang.String, java.lang.String)
	 */
	@Override
	public void endElement(String uri, String localName, String name) throws SAXException
	{
		super.endElement(uri, localName, name);
		
		if (!ISMLConstants.SCHEMA_URI.equals(uri))
		{
			return;	
		}		
		
		elements.pop();
		if (sequenceElementHit && ISMLConstants.SEQUENCE_ELEMENT.equals(localName))
		{
			sequenceElementHit = false;
		}	
	}
	
	
	/**
	 * Returns the local name of the last element that caused a start
	 * notification and not an end notification
	 * 
	 * @return The local name of the last element that caused a start
	 * notification
	 */
	protected String getLastElement()
	{
		return elements.isEmpty() ? null : elements.peek();
	}
	
	
	protected AbstractDeclaration setCommonFields(Attributes attributes, AbstractDeclaration declaration)
	{
		String typeName = retrieveValue(attributes, ISMLConstants.SCHEMA_URI, ISMLConstants.NAME_ATTRIBUTE);
		String typeUri = super.getTargetNamespace();
		
		if (typeName != null && typeUri != null)
		{				
			declaration.setQName(new QName(typeUri, typeName));
			declaration.setLocation(new ElementLocation(null, locator.getLineNumber(), locator.getColumnNumber()));
			return declaration;
		}
		return null;
	}

	
	/**
	 * @return the declaration
	 */
	protected AbstractDeclaration getDeclaration()
	{
		return declaration;
	}
	

	/**
	 * @param declaration the declaration to set
	 */
	protected void setDeclaration(AbstractDeclaration declaration)
	{
		this.declaration = declaration;
	}
}
