/**********************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.databuilders;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

/**
 * The data builder registery is a singleton class which allows different data
 * builders to register themselves prior to the start of the validation process.
 * 
 * @author sleeloy
 * @author Ali Mehregani
 */
public class DataBuilderRegistry
{
	/**
	 * Stores the data structure builders 
	 */
	protected Map<String, IDataBuilder<?>> dataStructureBuilders = new HashMap<String, IDataBuilder<?>>();

	/**
	 * The instance of this singleton class
	 */
	private static DataBuilderRegistry INSTANCE;

	
	/**
	 * Retrieves the instance of this singleton class
	 * 
	 * @return The static instance of this class 
	 */
	public static DataBuilderRegistry instance()
	{
		if (INSTANCE == null)
		{
			INSTANCE = new DataBuilderRegistry();
		}
		return INSTANCE;

	}

	
	/**
	 * Register data structure builder, 'dataBuilder', with id: 'id'.
	 *  
	 * @param id The id of the data builder
	 * @param dataBuilder The data builder
	 */
	public void registerDataStructureBuilder(String id, IDataBuilder<?> dataBuilder)
	{
		dataStructureBuilders.put(id, dataBuilder);
	}

	
	/**
	 * Remove the data structure with the passed in id
	 * 
	 * @param id The id of the data structure builder to remove
	 */
	public void unRegisterDataStructureBuilder(String id)
	{
		dataStructureBuilders.remove(id);
	}

	
	/**
	 * Returns the data structure builders registered
	 * 
	 * @return The registered data structure builders
	 */
	public Collection<IDataBuilder<?>> getDataStructureBuilders()
	{
		return dataStructureBuilders.values();
	}

	
	/**
	 * Returns the data structure builder with the passed in id
	 * 
	 * @param id The id of a data structure builder
	 * @return The data structure builder corresponding to 'id'
	 */
	public IDataBuilder<?> getDataStructureBuilder(String id)
	{
		return (IDataBuilder<?>) dataStructureBuilders.get(id);
	}
	
	
	/**
	 * Clears the registry 
	 */
	public void clear()
	{
		dataStructureBuilders.clear();
	}
}
