/**********************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.databuilders;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.cosmos.rm.internal.validation.SMLActivator;
import org.eclipse.cosmos.rm.internal.validation.artifacts.RuleBinding;
import org.eclipse.cosmos.rm.internal.validation.artifacts.RuleBindings;
import org.eclipse.cosmos.rm.internal.validation.artifacts.Schematron;
import org.eclipse.cosmos.rm.internal.validation.artifacts.Schematron.Pattern;
import org.eclipse.cosmos.rm.internal.validation.artifacts.Schematron.Rule;
import org.eclipse.cosmos.rm.internal.validation.common.ISMLConstants;
import org.eclipse.cosmos.rm.internal.validation.common.XMLInternalUtility;
import org.eclipse.cosmos.rm.internal.validation.common.AbstractValidationOutput.ValidationMessage;
import org.eclipse.cosmos.rm.internal.validation.core.IFoundationBuilder;
import org.eclipse.cosmos.rm.internal.validation.reference.SMLExtensionFunctions;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
/**
 * This builder will extract the schematron from the definition document and create a
 * map of namespace, element and schematron fragments.  There are a few assumptions
 * that are made:
 * <ol>
 * 	<li> If a schematron rule is contained in an schema element, then it must
 * 		 be contained within an element or element type declaration
 * 	</li>
 * 	<li> A schematron rule associated with an element or element type declaration
 * 		 must be contained in a schema element that defines a target namespace.
 * 	</li>
 * 	<li> A schematron rule that is associated with an element or element type declaration
 * 		 is always applied.
 * 	</li>
 * 	<li> In the presence of the ruleBindings element, a definition document that only contains 
 * 		 a schematron rule is applied to a document only if it is bound to the rule based
 * 		 on the content of the ruleBindings element.
 * 	</li>
 * 	<li> In the absense of the ruleBindings element, a definition document that only contains 
 * 		 a schematron rule is applied to ALL instance documents. 
 * 	</li>
 * </ol>
 * 		
 * @author sleeloy
 * @author Ali Mehregani
 */
public class ElementSchematronCacheBuilder extends AbstractDataBuilder<RuleBindings>
{
	/**
	 * The ID of this data builder
	 */
	public static final String ID = SMLActivator.PLUGIN_ID + ".ElementSchematronCacheBuilder"; //$NON-NLS-1$

	
	/**
	 * The data structure of this data builder
	 */
	private RuleBindings dataStructure;
	
	
	/**
	 * Stores the schematron rule
	 */
	protected StringBuffer buffer;
	
	/**
	 * Stores an element name declaration
	 */
	protected String name;
	
	/**
	 * Stores the name of an element type declaration
	 */
	protected String type;
	
	/**
	 * Stores the target namespace of the current schema being processed
	 */
	protected String targetNamespace;
	
	private Map<String, String> schemaAttributes;
	private boolean schematronFound = false;
	private Map<String, String> prefixLookup;
	private List<RuleBinding> ruleBindings = new ArrayList<RuleBinding>();
	private RuleBinding currentRuleBinding = null;
	private boolean aliasFound = false;
	private boolean aliasesFound = false;
	private List<String> aliases;
	private boolean instance = false;
		
	/**
	 * The patterns of the current schematron being processed	
	 */
	private List<Pattern> patterns;
	
	/**
	 * The current pattern
	 */
	private Pattern pattern;
	
	/**
	 * The current rule
	 */
	private Rule rule;

	/**
	 * Indicates that the rule element has been hit
	 */
	private boolean ruleElementHit;
	
	/**
	 * Indicates that the pattern element has been hit
	 */
	private boolean patternElementHit;

	/**
	 * Indicates that a ruleAlias element has been encountered
	 */
	private boolean ruleAliasFound = false;


	/**
	 * Indicates that a documentAlias element has been encountered
	 */
	private boolean documentAliasFound = false;
	
	public ElementSchematronCacheBuilder()
	{
		dataStructure = new RuleBindings();
		patterns = new ArrayList<Pattern>();
		addEvent(IFoundationBuilder.EVENT_CHARACTER);
	}
	
	public void endElement(String uri, String localName, String qName) throws SAXException 
	{
		if (ISMLConstants.INSTANCES_ELEMENT.equals(localName))
		{
			instance = false;
			return;
		}

		if (ISMLConstants.DOCUMENTALIAS_ELEMENT.equals(localName)) {
			documentAliasFound = false;
			return;
		}

		if (ISMLConstants.RULEALIAS_ELEMENT.equals(localName)) {
			ruleAliasFound = false;
			return;
		}
		
		if (ISMLConstants.RULEBINDING_ELEMENT.equals(localName))
		{
			if (currentRuleBinding.isValid()) {
				ruleBindings.add(currentRuleBinding);
			}
			currentRuleBinding = null;
			return;
		}

		if (ISMLConstants.SCHEMATRON_URI.equals(uri))
		{							
			XMLInternalUtility.addElement(getBuffer(), 0, qName, true, true);
			if (ISMLConstants.RULE_ELEMENT.equals(localName))
			{
				pattern.addRule(rule);
				ruleElementHit = false;
				rule = null;
			}
			else if (ISMLConstants.PATTERN_ELEMENT.equals(localName))
			{
				patterns.add(pattern);
				patternElementHit = false;
				pattern = null;
			}			
			else if (schematronFound && ISMLConstants.SCHEMA_ELEMENT.equals(localName))
			{
				int lineNumber = (locator == null) ? ValidationMessage.NO_LINE_NUMBER : locator.getLineNumber();
				Schematron schematronRule = new Schematron(buffer, patterns, prefixLookup, lineNumber);

				// associate schematron to element name if found
				if (name != null)
				{
					dataStructure.bindRule(targetNamespace, name, schematronRule);
				}
				
				// associate schematron to element type if found
				else if (type != null)
				{
					dataStructure.bindRule(targetNamespace, type+ISMLConstants.TYPEDELIM, schematronRule);
				}
							
				// associate schematron to aliases if found
				else if (aliases != null && ruleBindings.size() > 0)
				{
					Iterator<RuleBinding> iterRules = ruleBindings.iterator();
					while (iterRules.hasNext())
					{
						RuleBinding binding = iterRules.next();
						String docAlias = binding.getAlias();
						if (docAlias == null) {
							// rule is to be bound to all documents in SML-IF model
							dataStructure.addGloballyBoundRule(schematronRule);
							continue;
						}
						Object ruleAlias = binding.getRule();
						if (aliases.contains(ruleAlias))
						{
							dataStructure.bindRule(docAlias, schematronRule);							
						}
					}
				}				
				// No rule binding defined. This is a standalone schematron rule defined
				// under a definition document.  These are to be ignored.
				else if (targetNamespace == null && ruleBindings.size() <= 0)
				{
				}

				prefixLookup = new HashMap<String, String>();
				buffer = null;
				name = null;
				schematronFound = false;
				aliases = null;
				patterns.clear();
			}			
		}
		
		// Schema URI
		if (ISMLConstants.SCHEMA_URI.equals(uri))
		{
			// Schema element
			if (ISMLConstants.SCHEMA_ELEMENT.equals(localName))
			{
				schemaAttributes = null;
				targetNamespace = null;
			}
			// Element declaration
			else if (ISMLConstants.ELEMENT_ELEMENT.equals(localName))
			{
				name = null;
			}
			// Complex type element
			else if (ISMLConstants.COMPLEXTYPE_ELEMENT.equals(localName))
			{
				type = null;
			}
		}
		
		// SML-IF URI
		else if (ISMLConstants.SMLIF_URI.equals(uri))
		{
			if (ISMLConstants.ALIASES_ELEMENT.equals(localName))
			{
				aliasesFound = false;
			}
			else if (aliasFound && ISMLConstants.ALIAS_ELEMENT.equals(localName))
			{
				aliasFound = false;
			}
		}
	}

	public void startElement(String uri, String localName, String qName, Attributes attributes) throws SAXException 
	{
		
		if (ISMLConstants.INSTANCES_ELEMENT.equals(localName))
		{
			instance = true;
			return;
		}
		
		if (ISMLConstants.SCHEMA_URI.equals(uri))
		{
			if (ISMLConstants.SCHEMA_ELEMENT.equals(localName)){
				schematronFound = false;			
				targetNamespace = attributes.getValue(ISMLConstants.TARGET_NAMESPACE_ATTRIBUTE);
				//need to keep track of namespace to apply to the sch:schema since the schematron will be extracted in its own document
				schemaAttributes = new HashMap<String, String>();
				for (int x = 0; x < attributes.getLength(); x++){
					String key = attributes.getQName(x);
					if (key.startsWith(ISMLConstants.XML_NS_ATTRIBUTE)){
						schemaAttributes.put(attributes.getQName(x), attributes.getValue(x));
					}
				}
			}
			else if (ISMLConstants.ELEMENT_ELEMENT.equals(localName))
			{
				name = attributes.getValue(ISMLConstants.NAME_ATTRIBUTE);			
			}
			else if (ISMLConstants.COMPLEXTYPE_ELEMENT.equals(localName))
			{
				type = attributes.getValue(ISMLConstants.NAME_ATTRIBUTE);
			}			
		}
		else if (ISMLConstants.SMLIF_URI.equals(uri))
		{
			if (ISMLConstants.RULEBINDING_ELEMENT.equals(localName))
			{
				currentRuleBinding = new RuleBinding();
			}
			else if (ISMLConstants.RULEALIAS_ELEMENT.equals(localName))
			{
				ruleAliasFound = true;
			}
			else if (ISMLConstants.DOCUMENTALIAS_ELEMENT.equals(localName))
			{
				documentAliasFound  = true;
			}
			else if (ISMLConstants.ALIASES_ELEMENT.equals(localName))
			{
				aliasesFound = true;
				aliases = new ArrayList<String>();
			}
			else if ((aliasesFound) && (ISMLConstants.ALIAS_ELEMENT.equals(localName)))
			{
				aliasFound = true;
			}
		}
		else if (ISMLConstants.SCHEMATRON_URI.equals(uri))
		{
			if ((ISMLConstants.SCHEMA_ELEMENT.equals(localName)))
			{
				schematronFound = true;		
				buffer = new StringBuffer();
			}
		}
		
		
		if (buffer != null)
		{
			if (ISMLConstants.SCHEMATRON_URI.equals(uri))
			{				
				if (ISMLConstants.RULE_ELEMENT.equals(localName))
				{
					ruleElementHit = true;
					rule = new Rule();
					if (locator != null) {
						rule.setLineNumber(locator.getLineNumber());
					}
					StringBuffer ruleFragment = rule.getFragment();
					ruleFragment.append(ISMLConstants.OPEN_ANGLE_BRACKET).append(qName).append(printRuleAttributes(attributes)).append(ISMLConstants.CLOSE_ANGLE_BRACKET);
					return;
				}
				else if (ISMLConstants.PATTERN_ELEMENT.equals(localName))
				{
					patternElementHit = true;
					pattern = new Pattern();					
				}
				else
				{
					if (ISMLConstants.NS_ELEMENT.equals(localName)){
						if (prefixLookup == null){
							prefixLookup = new HashMap<String, String>();
						}
						prefixLookup.put(attributes.getValue(ISMLConstants.PREFIX_ATTRIBUTE), attributes.getValue(ISMLConstants.URI_ATTRIBUTE).trim());
					}					
				}
				
				getBuffer().append(ISMLConstants.OPEN_ANGLE_BRACKET).append(qName).append(printReplaceSMLFNAttributes(attributes)).append(ISMLConstants.CLOSE_ANGLE_BRACKET);
			}
			else if (ISMLConstants.ELEMENT_ELEMENT.equals(localName) && ISMLConstants.SCHEMA_URI.equals(uri))
			{
				buffer.append(ISMLConstants.OPEN_ANGLE_BRACKET).append(qName).append(printSchNamespaceAttributes(attributes)).append(ISMLConstants.CLOSE_ANGLE_BRACKET);				
			}
			else
			{			
				getBuffer().append(ISMLConstants.OPEN_ANGLE_BRACKET).append(qName).append(printAttributes(attributes)).append(ISMLConstants.CLOSE_ANGLE_BRACKET); 
			}
		}
	}
	
	private StringBuffer getBuffer()
	{
		StringBuffer buff = buffer;
		if (ruleElementHit)
		{
			buff = rule.getFragment();
		}
		else if (patternElementHit)
		{
			buff = pattern.getFragment();
		}
		return buff;
	}


	protected StringBuffer printRuleAttributes(Attributes attributes)
	{
		StringBuffer buffer = new StringBuffer();
		for (int x = 0; x<attributes.getLength(); x++){
			String attributeValue = attributes.getValue(x);
			if (ISMLConstants.CONTEXT_ATTRIBUTE.equals(attributes.getLocalName(x)))
			{			
				rule.setContext(attributeValue);
								
				if (attributeValue.indexOf(ISMLConstants.DEREF_FN) > -1)
				{
					buffer.append(ISMLConstants.SINGLE_SPACE).append(attributes.getQName(x)).append(ISMLConstants.EQUAL_SIGN).append(ISMLConstants.DOUBLE_QUOTE).append("/*[1]").append(ISMLConstants.DOUBLE_QUOTE);
					rule.setMatchContainsDeref(true);
					continue;
				}				
			}
			buffer.append(ISMLConstants.SINGLE_SPACE).append(attributes.getQName(x)).append(ISMLConstants.EQUAL_SIGN).append(ISMLConstants.DOUBLE_QUOTE).append(attributeValue).append(ISMLConstants.DOUBLE_QUOTE);			
		}
		return buffer;
		
	}	
	
	protected StringBuffer printReplaceSMLFNAttributes(Attributes attributes){
		StringBuffer buffer = new StringBuffer();
		for (int x = 0; x<attributes.getLength(); x++){
			if ((ISMLConstants.URI_ATTRIBUTE.equals(attributes.getLocalName(x))) && (ISMLConstants.SML_FN_URI.equals(attributes.getValue(x)))) {
				buffer.append(ISMLConstants.SINGLE_SPACE).append(attributes.getQName(x)).append(ISMLConstants.EQUAL_SIGN).append(ISMLConstants.DOUBLE_QUOTE).append(SMLExtensionFunctions.class.getName()).append(ISMLConstants.DOUBLE_QUOTE);
			} else {
				buffer.append(ISMLConstants.SINGLE_SPACE).append(attributes.getQName(x)).append(ISMLConstants.EQUAL_SIGN).append(ISMLConstants.DOUBLE_QUOTE).append(attributes.getValue(x)).append(ISMLConstants.DOUBLE_QUOTE);
			}				 
		}
		return buffer;
		
	}	
	protected StringBuffer printSchNamespaceAttributes(Attributes attributes){
		StringBuffer buffer = new StringBuffer();
		for (int x = 0; x<attributes.getLength(); x++){
			buffer.append(ISMLConstants.SINGLE_SPACE).append(attributes.getQName(x)).append(ISMLConstants.EQUAL_SIGN).append(ISMLConstants.DOUBLE_QUOTE).append(attributes.getValue(x)).append(ISMLConstants.DOUBLE_QUOTE);				 
			schemaAttributes.remove(attributes.getQName(x));
		}
		Iterator<String> iter = schemaAttributes.keySet().iterator();
		while (iter.hasNext()){
			String key = iter.next();
			buffer.append(ISMLConstants.SINGLE_SPACE).append(key).append(ISMLConstants.EQUAL_SIGN).append(ISMLConstants.DOUBLE_QUOTE).append(schemaAttributes.get(key)).append(ISMLConstants.DOUBLE_QUOTE);				 
		}
		return buffer;
		
	}
	
	
	protected StringBuffer printAttributes(Attributes attributes){
		StringBuffer buffer = new StringBuffer();
		for (int x = 0; x<attributes.getLength(); x++){
			buffer.append(ISMLConstants.SINGLE_SPACE).append(attributes.getQName(x)).append(ISMLConstants.EQUAL_SIGN).append(ISMLConstants.DOUBLE_QUOTE).append(attributes.getValue(x)).append(ISMLConstants.DOUBLE_QUOTE);				 
		}
		return buffer;
		
	}

	public RuleBindings getDataStructure() 
	{
		return dataStructure;
	}

	
	/**
	 * @see org.xml.sax.helpers.DefaultHandler#characters(char[], int, int)
	 */
	public void characters(char[] ch, int start, int length) throws SAXException
	{
		if (instance) 
		{
			return;
		}
				
		if (ruleElementHit && rule != null) 
		{
			StringBuffer buffer = rule.getFragment();			
			buffer.append(new String(ch, start, length));
		}
		
		if (ruleAliasFound && (currentRuleBinding != null)) 
		{
			String currentRule = currentRuleBinding.getRule();
			if (currentRule == null) {
				currentRule = "";
			}
			currentRuleBinding.setRule(currentRule + new String(ch, start, length));
		}
		
		if (documentAliasFound && (currentRuleBinding != null)) {
			String currentDocumentAlias = currentRuleBinding.getAlias();
			if (currentDocumentAlias == null) {
				currentDocumentAlias = "";
			}
			currentRuleBinding.setAlias(currentDocumentAlias + new String(ch, start, length));
		}

		if (aliasFound) 
		{
			aliases.add(new String(ch, start, length));
		}
	}

	public byte getPhase() {
		return ISMLConstants.UNKNOWN_PHASE;
	}
}
