/**********************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.databuilders;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import javax.xml.namespace.NamespaceContext;
import javax.xml.namespace.QName;

import org.eclipse.cosmos.rm.internal.validation.SMLActivator;
import org.eclipse.cosmos.rm.internal.validation.artifacts.ConstraintNode;
import org.eclipse.cosmos.rm.internal.validation.artifacts.IdentityConstraintStructure;
import org.eclipse.cosmos.rm.internal.validation.common.ISMLConstants;
import org.eclipse.cosmos.rm.internal.validation.common.IValidationConstants;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidationMessages;
import org.eclipse.cosmos.rm.internal.validation.common.AbstractValidationOutput.ValidationMessageFactory;
import org.eclipse.cosmos.rm.internal.validation.core.IFoundationBuilder;
import org.eclipse.osgi.util.NLS;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;

/**
 * This builder will construct a data structure that will be used
 * as part of the identity constraint (key, keyref, and unique) 
 * validation.  The data structure returned by this builder is of type
 * {@link IdentityConstraintStructure}
 * 
 * @author Ali Mehregani
 */
public class IdentityConstraintDataBuilder extends AbstractDataBuilder<IdentityConstraintStructure>
{
	/**
	 * The ID of this builder
	 */
	public static final String ID = SMLActivator.PLUGIN_ID + ".IdentityConstraintDataBuilder";
	
	/**
	 * The identity constraint structure
	 */
	private IdentityConstraintStructure identityStruct;
	
	/**
	 * The current identity constraint declaration
	 */
	private IdentityConstraint identityDecl;
	
	/**
	 * Indicates that the 'element' element is hit
	 */
	private boolean elementHit;
	
	/**
	 * Indicates that the 'annotation' element is hit
	 */
	private boolean annotationHit;
	
	/**
	 * Inidcates that the 'appinfo' element is hit
	 */
	private boolean appInfoHit;
	
	/**
	 * The current index of the document node
	 */
	private LinkedList<Integer> documentNodeInx;
	
	/**
	 * Indicates that the 'document' element has been hit
	 */
	private boolean documentElementHit;
	
	/**
	 * Indicates that the 'data' element is hit
	 */
	private boolean dataElementHit;

	/**
	 * The current aliases of a document
	 */
	private List<String> currentAliases;
	
	/**
	 * The current alias
	 */
	private String currentAlias;

	/**
	 * Indicates that an alias element has been hit
	 */
	private boolean aliasElementHit;
	
	/**
	 * The current index of the child element being processed
	 */
	private int currentInx;
	
	/**
	 * The current depth of the XML structure
	 */
	private int currentDepth;

	/**
	 * The uri of the element being processed
	 */
	private String currentURI;

	/**
	 * The local name of the element being processed
	 */
	private String currentElementName;
	
	/**
	 * The current orphan document index
	 */
	private int orphanInx;
	
	/**
	 * Keeps track of the constraints
	 */
	private IdentityConstraintMap identityConstraintMap;
	
	/**
	 * Set when a constraint is referenced in an element 
	 * declaration
	 */
	private boolean referencedConstraint;
	
	public IdentityConstraintDataBuilder()
	{
		identityStruct = new IdentityConstraintStructure();
		documentNodeInx = new LinkedList<Integer>();
		currentDepth = -1;
		orphanInx = 0;
		addEvent(IFoundationBuilder.EVENT_CHARACTER);
		identityConstraintMap = new IdentityConstraintMap();
	}
	
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.AbstractDataBuilder#startElement(java.lang.String, java.lang.String, java.lang.String, org.xml.sax.Attributes)
	 */
	public void startElement(String uri, String localName, String qName, Attributes attributes) throws SAXException
	{		
		super.startElement(uri, localName, qName, attributes);
		
		byte currentPhase = getCurrentPhase();
		switch (currentPhase)
		{
			case ISMLConstants.DEFINITIONS_PHASE:
				handleDefinitionStartElement(uri, localName, attributes);							
				break;
			case ISMLConstants.INSTANCES_PHASE:
				handleInstanceStartElement(uri, localName, attributes);
				break;
			default: 
				break;
		}		
	}
	
	public void endElement(String uri, String localName, String qName) throws SAXException
	{
		super.endElement(uri, localName, qName);
		byte currentPhase = getCurrentPhase();
		switch (currentPhase)
		{
			case ISMLConstants.DEFINITIONS_PHASE:
				handleDefinitionCloseElement(uri, localName);							
				break;
			case ISMLConstants.INSTANCES_PHASE:
				handleInstanceCloseElement(uri, localName);
				break;
			default: 
				break;
		}		
	}

	private boolean validField(String field)
	{
		return field != null && field.length() > 0;
	}



	@SuppressWarnings("unchecked")
	private void handleDefinitionStartElement(String uri, String localName, Attributes attributes)
	{
		if (ISMLConstants.SCHEMA_URI.equals(uri))
		{
			if (!elementHit && ISMLConstants.ELEMENT_ELEMENT.equals(localName))
			{
				elementHit = true;
				currentURI = getTargetNamespace();
				currentElementName = attributes.getValue(ISMLConstants.NAME_ATTRIBUTE);				
				currentElementName = currentElementName == null ? attributes.getValue(ISMLConstants.SCHEMA_URI, ISMLConstants.NAME_ATTRIBUTE) : currentElementName;
				
				// Check to see if the element has a substitution group
				String substitutionGroup = attributes.getValue(ISMLConstants.SUBSTITUTIONGROUP_ATTRIBUTE);
				ConstraintNode substitutionGroupNode = validField(substitutionGroup) ? retrieveElementDeclaration(substitutionGroup) : null;
				while (substitutionGroupNode != null)
				{				
					IdentityConstraint[] identityConstraints = substitutionGroupNode == null ? null : identityStruct.retrieveConstraint(substitutionGroupNode.getUri(), substitutionGroupNode.getName());
					if (identityConstraints != null)
					{						
						// Add the identities of the substitution group node
						for (int i = 0; i < identityConstraints.length; i++)
						{
							identityStruct.addDeclaration(currentURI, currentElementName, identityConstraints[i]);							
						}								
					}
					
					substitutionGroupNode = substitutionGroupNode.getSubstitutionGroup();
				}
			}
			else if (elementHit && ISMLConstants.ANNOTATION_ELEMENT.equals(localName))
			{
				annotationHit = true;
			}
			else if (annotationHit && ISMLConstants.APP_INFO_ELEMENT.equals(localName))
			{
				appInfoHit = true;
			}			
			return;
		}
		
		
		boolean isKey = ISMLConstants.KEY_ELEMENT.equals(localName);
		boolean isKeyRef = !isKey && ISMLConstants.KEY_REF_ELEMENT.equals(localName);
		boolean isUnique = !isKeyRef && !isKey && ISMLConstants.UNIQUE_ELEMENT.equals(localName);
		boolean constraintDefined = isKey || isKeyRef || isUnique;
		
		if (appInfoHit && ISMLConstants.SML_URI.equals(uri))			
		{
			
			if (constraintDefined)
			{
				// Get the name of the key 
				String name = retrieveAttribute(attributes, ISMLConstants.SML_URI, ISMLConstants.NAME_ATTRIBUTE);
				
				// Is this a reference?
				String ref = retrieveAttribute(attributes, ISMLConstants.SML_URI, ISMLConstants.REF_ATTRIBUTE);
				if (validField(ref))
				{
					// Display an error message if a name attribute is preset
					if (validField(name))
					{
						getMessageOutputter().reportMessage(ValidationMessageFactory.createErrorMessage(
								locator.getLineNumber(), SMLValidationMessages.identityInvalidNameAttribute));
						return;
					}
					
					// Try to resolve the identity constraint reference
					IdentityConstraint constraint = identityConstraintMap.get(ref);
					if (constraint == null)
					{
						getMessageOutputter().reportMessage(ValidationMessageFactory.createErrorMessage(
								locator.getLineNumber(), NLS.bind(SMLValidationMessages.identityMissingConstraint, ref)));
						return;
					}
					else
					{
						// Make sure the referenced constraint has a consistent type					
						if (constraint.getType() != computeType(isKey, isKeyRef, isUnique))
						{
							getMessageOutputter().reportMessage(ValidationMessageFactory.createErrorMessage(
									locator.getLineNumber(), NLS.bind(SMLValidationMessages.identityBadReferenceType, ref)));
							return;
						}
						
						referencedConstraint = true;
						identityDecl = constraint;
					}						
				}
				else
				{
					if (!validField(name))
					{
						return;
					}
					 
					identityDecl = new IdentityConstraint(name);	
					final Map<String, String> clonedPrefixMap = (Map<String, String>)((Hashtable)super.getPrefixMap()).clone();
					 
					identityDecl.setNamespaceContext(new NamespaceContext(){

						public String getNamespaceURI(String prefix)
						{		
							String uri = (String)clonedPrefixMap.get(prefix);
							return uri == null ? IValidationConstants.EMPTY_STRING : uri;
						}

						public String getPrefix(String arg0)
						{
							// Doesn't need to be implemented 
							return null;
						}

						public Iterator<?> getPrefixes(String arg0)
						{
							// Doesn't need to be implemented 
							return null;
						}});
					 
					identityDecl.setType(computeType(isKey, isKeyRef, isUnique));
					if (isKeyRef)
					{
						identityDecl.setReference(retrieveAttribute(attributes, ISMLConstants.SML_URI, ISMLConstants.REFER_ATTRIBUTE));
					}
				}				
			}
			else if (identityDecl != null)
			{
				if (referencedConstraint)
				{
					// Child elements are not permitted when constraint is a reference
					getMessageOutputter().reportMessage(ValidationMessageFactory.createErrorMessage(
							locator.getLineNumber(), SMLValidationMessages.identityInvalidElements));
					identityDecl = null;
					return;
				}
				else if (ISMLConstants.SELECTOR_ELEMENT.equals(localName))
				{					 
					String selector = attributes.getValue(ISMLConstants.XPATH_ATTRIBUTE);
					selector = selector == null ? attributes.getValue(ISMLConstants.SML_URI, ISMLConstants.XPATH_ATTRIBUTE) : selector;
					identityDecl.setSelector(selector);
				}
				else if (ISMLConstants.FIELD_ELEMENT.equals(localName))
				{
					String field = attributes.getValue(ISMLConstants.XPATH_ATTRIBUTE);
					field = field == null ? attributes.getValue(ISMLConstants.SML_URI, ISMLConstants.XPATH_ATTRIBUTE) : field;
					identityDecl.addField(field);
				}
			}
		}		
		
		// The constraint is defined in a location where it's not suppose to appear
		else if (constraintDefined)
		{
			getMessageOutputter().reportMessage(ValidationMessageFactory.createWarningMessage(
							locator.getLineNumber(), 
							SMLValidationMessages.identityBadDeclaration));
		}
			
	}

	/**
	 * Retrieve the element declaration with the qualified
	 * name passed in
	 * 
	 * @param qName A qualified name
	 * @return A constraint node representing the element declaration
	 */
	private ConstraintNode retrieveElementDeclaration(String qName)
	{
		String[] uriLocalName = getUriLocalName(qName);
		return DataBuilderUtil.retrieveElement(new QName(uriLocalName[0], uriLocalName[1]));
	}

	private byte computeType(boolean isKey, boolean isKeyRef, boolean isUnique)
	{
		return (isKey ? IdentityConstraint.KEY_TYPE : (isKeyRef ? IdentityConstraint.KEY_REF_TYPE : IdentityConstraint.UNIQUE_TYPE));
	}

	private String retrieveAttribute(Attributes attributes, String uri, String localName)
	{
		String value = attributes.getValue(localName);
		return value == null ? attributes.getValue(uri, localName) : value;
	}

	private void handleDefinitionCloseElement(String uri, String localName)
	{
		if (ISMLConstants.SML_URI.equals(uri))
		{
			if (appInfoHit && identityDecl != null)
			{
				boolean isKey = ISMLConstants.KEY_ELEMENT.equals(localName);
				boolean isKeyref = ISMLConstants.KEY_REF_ELEMENT.equals(localName);
				boolean isUnique = ISMLConstants.UNIQUE_ELEMENT.equals(localName);
				
				if (isKey || isKeyref || isUnique)
				{
					// The selector and the field are required parts of an identity constraint declaration
					if (validField(identityDecl.getSelector()) && identityDecl.getFields().size() > 0)
					{
						// Names of identity constraints are required to be unique 
						if (identityConstraintMap.get(identityDecl.getName()) != null)
						{
							getMessageOutputter().reportMessage(ValidationMessageFactory.createErrorMessage(
									locator.getLineNumber(), SMLValidationMessages.identityDuplicateName));							
						}
						else
						{						
							identityStruct.addDeclaration(currentURI, currentElementName, identityDecl);
							identityConstraintMap.add(identityDecl);
						}
						
						referencedConstraint = false;
					}
					identityDecl = null;
				}
			}
			else if (appInfoHit && ISMLConstants.APP_INFO_ELEMENT.equals(localName))
			{
				appInfoHit = false;
			}
			else if (annotationHit && ISMLConstants.ANNOTATION_ELEMENT.equals(localName))
			{
				annotationHit = false;
			}
		}
		else if (ISMLConstants.SCHEMA_URI.equals(uri))
		{
			if (elementHit && ISMLConstants.ELEMENT_ELEMENT.equals(localName))
			{
				elementHit = false;
			}
		}
		
	}

	private void handleInstanceStartElement(String uri, String localName, Attributes attributes)
	{
		if (ISMLConstants.SMLIF_URI.equals(uri))			
		{
			if (!documentElementHit && ISMLConstants.DOCUMENT_ELEMENT.equals(localName))
			{
				documentElementHit = true;
				if (currentAliases != null)
					currentAliases.clear();
			}
			else if (documentElementHit && ISMLConstants.DATA_ELEMENT.equals(localName))
			{
				dataElementHit = true;
			}
			else if (documentElementHit && ISMLConstants.ALIAS_ELEMENT.equals(localName))
			{
				aliasElementHit = true;				
			}
		}
		else if (dataElementHit)
		{
			currentDepth++;
			if (documentNodeInx.size() - 1 < currentDepth)
			{
				currentInx = 0;
			}
			else
			{
				currentInx = ((Integer)documentNodeInx.get(currentDepth)).intValue() + 1;
			}
			
			documentNodeInx.add(new Integer(currentInx));
			if (identityStruct.isConstrained(uri, localName))
			{
				int[] primitiveIndices = new int[documentNodeInx.size()]; 
				for (int i = 0; i < primitiveIndices.length; i++)
				{
					primitiveIndices[i] = ((Integer)documentNodeInx.get(i)).intValue();
				}
				
				if (currentAliases == null || currentAliases.isEmpty())
				{
					identityStruct.addOrphanedConstrainedInstance(orphanInx++, primitiveIndices);
				}
				else
				{
					for (int i = 0, aliasCount = currentAliases.size(); i < aliasCount; i++)
					{
						identityStruct.addConstrainedInstance((String)currentAliases.get(i), primitiveIndices);
					}					
				}
			}
		}
	}

	
	private void handleInstanceCloseElement(String uri, String localName)
	{
		if (ISMLConstants.SMLIF_URI.equals(uri))
		{
			if (dataElementHit && ISMLConstants.DATA_ELEMENT.equals(localName))
			{
				dataElementHit = false;
			}
			else if (documentElementHit && ISMLConstants.DOCUMENT_ELEMENT.equals(localName))
			{
				documentElementHit = false;
				currentDepth = -1;
				documentNodeInx = new LinkedList<Integer>();
			}
			else if (aliasElementHit && ISMLConstants.ALIAS_ELEMENT.equals(localName))
			{
				aliasElementHit = false;
				if (currentAlias != null)
				{
					if (currentAliases == null)
						currentAliases = new ArrayList<String>();
					currentAliases.add(currentAlias);
				}
				currentAlias = null;
			}
		}
		else if (dataElementHit)
		{
			currentDepth--;
			if (documentNodeInx.size() - currentDepth >= 3)
				documentNodeInx.removeLast();			
		}		
	}
	
	/**
	 * @see org.xml.sax.helpers.DefaultHandler#characters(char[], int, int)
	 */
	public void characters(char[] ch, int start, int length) throws SAXException
	{
		if (aliasElementHit)
		{
			currentAlias = currentAlias == null ? new String(ch, start, length) : currentAlias + new String(ch, start, length);
		}
	}
	
	
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.IDataBuilder#getDataStructure()
	 */
	public IdentityConstraintStructure getDataStructure()
	{
		return identityStruct;
	}

	
	/**
	 * Represents an identity constraint declaration
	 * 
	 * @author Ali Mehregani
	 */
	public static class IdentityConstraint
	{
		/**
		 * Indicates the key identity constraint
		 */
		public static final byte KEY_TYPE = 0x00;
		
		/**
		 * Indicates the key ref identity constraint
		 */
		public static final byte KEY_REF_TYPE = 0x01;
		
		/**
		 * Indicates the unique identity constraint
		 */
		public static final byte UNIQUE_TYPE = 0x02;
		
	
		/**
		 * The type of this declaration
		 */
		private byte type;
		
		/**
		 * The name of this declaration
		 */
		private String name;
		
		/**
		 * The fields associated with this constraint
		 */
		private List<String> fields;
		
		/**
		 * The xpath expression indicating the selector of this constraint
		 */
		private String selector;
		
		/**
		 * The namespace context
		 */
		private NamespaceContext namespaceContext; 
		
		/**
		 * The referenced constraint
		 */
		private String reference;
		
		
		/**
		 * Constructor
		 * 
		 * @param name The name of the declaration
		 */
		public IdentityConstraint(String name)
		{
			this.name = name;
			this.fields = new ArrayList<String>();
		}

		
		public byte getType()
		{
			return type;
		}
		
		public void setType(byte type)
		{
			this.type = type;
		}

		public List<String> getFields()
		{			
			return fields;
		}

		public String getSelector()
		{
			return selector;
		}

		public void setFields(List<String> fields)
		{
			this.fields = fields;
		}

		public void addField (String field)
		{
			fields.add(field);
		}
		
		public void setSelector(String selector)
		{
			this.selector = selector;
		}
		
		public String getName()
		{
			return name;
		}
	
		public void setName(String name)
		{
			this.name = name;
		}

		/**
		 * @return the namespaceContext
		 */
		public NamespaceContext getNamespaceContext()
		{
			return namespaceContext;
		}

		/**
		 * @param namespaceContext the namespaceContext to set
		 */
		public void setNamespaceContext(NamespaceContext namespaceContext)
		{
			this.namespaceContext = namespaceContext;
		}

		/**
		 * @return the reference
		 */
		public String getReference()
		{
			return reference;
		}

		/**
		 * @param reference the reference to set
		 */
		public void setReference(String reference)
		{
			this.reference = reference;
		}			
	}
	
	private String[] getUriLocalName(String name)
	{
		Map<String, String> prefixMap = IdentityConstraintDataBuilder.this.getPrefixMap();
		String[] tokenizedName = name.split(":");
		return tokenizedName.length == 2 ? 
				new String[]{prefixMap.get(tokenizedName[0]), tokenizedName[1]} : 
				new String[]{IdentityConstraintDataBuilder.this.getTargetNamespace(), tokenizedName[0]};
	}
	
	private class IdentityConstraintMap
	{
		private Map<String, Map<String, IdentityConstraint>> constraintStore;

		public IdentityConstraintMap()
		{
			constraintStore = new Hashtable<String, Map<String, IdentityConstraint>>();
		}
		
		public void add (IdentityConstraint constraint)
		{
			String[] fields = getUriLocalName(constraint.getName());
			if (fields[0] == null || fields[1] == null)
			{
				return;
			}
			
			Map<String, IdentityConstraint> identityStore = constraintStore.get(fields[0]);
			if (identityStore == null)
			{
				identityStore = new Hashtable<String, IdentityConstraint>();
				constraintStore.put(fields[0], identityStore);
			}
			
			identityStore.put(fields[1], constraint);
		}
		
		public IdentityConstraint get (String constraintName)
		{
			String[] fields = getUriLocalName(constraintName);		
			if (fields[0] == null || fields[1] == null)
			{
				return null;
			}
			
			Map<String, IdentityConstraint> identityStore = constraintStore.get(fields[0]);
			return identityStore == null ? null : identityStore.get(fields[1]);
		}		
	}
}
