/**********************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.databuilders;

import java.util.Hashtable;
import java.util.Map;

import org.eclipse.cosmos.rm.internal.validation.SMLActivator;
import org.eclipse.cosmos.rm.internal.validation.artifacts.DocumentNode;
import org.eclipse.cosmos.rm.internal.validation.artifacts.ElementEdge;
import org.eclipse.cosmos.rm.internal.validation.artifacts.ElementNode;
import org.eclipse.cosmos.rm.internal.validation.artifacts.ElementTypeMap;
import org.eclipse.cosmos.rm.internal.validation.common.ISMLConstants;
import org.eclipse.cosmos.rm.internal.validation.core.IFoundationBuilder;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;

public class ReferenceGraphDataBuilder extends AbstractDataBuilder<Map<String, DocumentNode>>
{
	/**
	 * The ID of this data builder
	 */
	public static final String ID = SMLActivator.PLUGIN_ID + ".ReferenceGraphDataBuilder";

	/**
	 * Represents the reference graph
	 * 
	 * KEY = The alias representing a document node
	 * VALUE = An object of type {@link DocumentNode}
	 */
	private Map<String, DocumentNode> referenceGraph;
	
	/**
	 * The current node which represents a document 
	 */
	private DocumentNode currentNode;
	
	/**
	 * Set to true when notified of the start of an alias element
	 */
	private boolean aliasElementHit;
	
	/**
	 * Set to true when notified of the start of an uri element
	 */
	private boolean uriElementHit;
	
	/**
	 * Stores the source node of a reference
	 */
	private ElementNode srcReferenceNode;
	
	/**
	 * Stores the last element that has been hit
	 */
	private ElementNode lastElement;
			
	/**
	 * Current edge of the reference graph node
	 */
	private ElementEdge elementEdge;
	
	/**
	 * The current alias
	 */
	private String currentAlias;
	
	/**
	 * The data builder for the element type map
	 */
	private IDataBuilder<ElementTypeMap> elementTypeMapBuilder; 
	
	
	/**
	 * Constructor
	 */
	public ReferenceGraphDataBuilder()
	{
		referenceGraph = new Hashtable<String, DocumentNode>();
		currentAlias = "";
		lastElement = new ElementNode();
		srcReferenceNode = new ElementNode();
		super.addEvent(IFoundationBuilder.EVENT_CHARACTER);
	}
	
	/**
	 * @see org.xml.sax.helpers.DefaultHandler#startElement(java.lang.String, java.lang.String, java.lang.String, org.xml.sax.Attributes)
	 */
	public void startElement(String uri, String localName, String qName, Attributes attributes) throws SAXException
	{
		resetElementStartStatus();
		if (ISMLConstants.SMLIF_URI.equals(uri))
		{
			if (ISMLConstants.DOCUMENT_ELEMENT.equals(localName))
			{							
				currentNode = new DocumentNode();					
			}
			else if (ISMLConstants.ALIAS_ELEMENT.equals(localName))
			{
				aliasElementHit = true;
			}
		}
		else if (ISMLConstants.SML_URI.equals(uri))
		{
			if (ISMLConstants.URI_ELEMENT.equals(localName))
			{
				uriElementHit = true;
				srcReferenceNode.setName(lastElement.getName());
				srcReferenceNode.setUri(lastElement.getUri());
			}
		}
		
		lastElement.setName(localName);
		lastElement.setUri(uri);
	}

	/**
	 * Resets flags that are set when the handler is notified of the start 
	 * of a specific element. 
	 */
	private void resetElementStartStatus()
	{
		aliasElementHit = false;
		elementEdge = null;
	}

	/**
	 * @see org.xml.sax.helpers.DefaultHandler#characters(char[], int, int)
	 */
	@SuppressWarnings("unchecked")
	public void characters(char[] ch, int start, int length) throws SAXException
	{
		String textNode = new String(ch, start, length).trim();
		if (currentNode == null || textNode.length() <= 0)
			return;
		
		if (aliasElementHit)
		{
			currentAlias += textNode;				
		}
		else if (uriElementHit)
		{
			if (elementEdge == null)
			{
				if (elementTypeMapBuilder == null)
					elementTypeMapBuilder = (IDataBuilder<ElementTypeMap>)DataBuilderRegistry.instance().getDataStructureBuilder(ElementTypeMapDataBuilder.ID);
				
				ElementTypeMap elementTypeMap = null;
				String referenceType = null;
				if (elementTypeMapBuilder == null || 
					(elementTypeMap = (ElementTypeMap)elementTypeMapBuilder.getDataStructure()) == null ||
					(referenceType = elementTypeMap.getType(srcReferenceNode)) == null)
					return;
				
				elementEdge = new ElementEdge(srcReferenceNode.getUri(), referenceType, locator.getLineNumber());
			}
											
			elementEdge.appendToReference(textNode);
		}
	}
	
	/**
	 * @see org.xml.sax.helpers.DefaultHandler#endElement(java.lang.String, java.lang.String, java.lang.String)
	 */
	public void endElement(String uri, String localName, String qName) throws SAXException
	{
		if (currentNode == null)
			return;
		
		
		if (ISMLConstants.SMLIF_URI.equals(uri))
		{
			if (ISMLConstants.DOCUMENT_ELEMENT.equals(localName))
			{
				String[] aliases = currentNode.getAliases();
				if (aliases == null)
					return;
				
				for (int i = 0; i < aliases.length; i++)
				{
					referenceGraph.put(aliases[i], currentNode);
				}
				
				currentNode = null;
			}
			else if (ISMLConstants.ALIAS_ELEMENT.equals(localName))
			{
				currentNode.addAlias(currentAlias);
				currentAlias = "";
			}
		}
		else if (ISMLConstants.SML_URI.equals(uri))
		{
			if (ISMLConstants.URI_ELEMENT.equals(localName) && elementEdge != null)
				currentNode.addEdge(elementEdge);
		}
	}

	
	/**
	 * The data structure returned is a Map with the following details:
	 * 
	 * KEY = ALIAS of a document
	 * VALUE = An element of type {@link DocumentNode}  
	 */
	public Map<String, DocumentNode> getDataStructure()
	{
		return referenceGraph;
	}
	
	public byte getPhase() {
		return ISMLConstants.DEFINITIONS_INSTANCES_PHASE;
	}

}
