/**********************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.reference;

import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;

import javax.xml.xpath.XPathFunction;
import javax.xml.xpath.XPathFunctionException;

import org.eclipse.cosmos.rm.internal.validation.artifacts.TypeNode;
import org.eclipse.cosmos.rm.internal.validation.common.ISMLConstants;
import org.eclipse.cosmos.rm.internal.validation.common.IValidationConstants;
import org.eclipse.cosmos.rm.internal.validation.common.SMLIFIdentity;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidationMessages;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidatorUtil;
import org.eclipse.cosmos.rm.internal.validation.databuilders.DocumentDOMBuilder;
import org.eclipse.cosmos.rm.internal.validation.databuilders.IdentityDataBuilder;
import org.eclipse.cosmos.rm.internal.validation.databuilders.TypeInheritanceDataBuilderImpl;
import org.eclipse.osgi.util.NLS;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * This is an implementaion of the dref function specified in the SML
 * specification.  This implementation makes the following assumptions:
 * <ul>
 * 	<li> The function is invoked after parsing the documents to be validated (i.e.
 * 	the data builders have constructed their associated data structure) 
 * 	</li>
 * 	<li> The data structure of the data builder with the ID 
 * 	{@link DocumentDOMBuilder#ID} exists.
 * 	</li>
 * </ul>
 * This is a singleton class that should be accessed via {@link DerefXPathFunction#instance()} 
 *  
 * @author Ali Mehregani
 */
public class DerefXPathFunction implements XPathFunction
{
	/**
	 * Arity - The number of arguments for the dref function
	 */
	public static final int ARITY = 1;
	
	/**
	 * The instance of this singleton class
	 */
	private static DerefXPathFunction instance;
	
	
	/**
	 * @see javax.xml.xpath.XPathFunction#evaluate(java.util.List)
	 */
	@SuppressWarnings("unchecked")
	public Object evaluate(List arguments) throws XPathFunctionException
	{
		if (arguments == null || arguments.size() <= 0)
			throw new XPathFunctionException(SMLValidationMessages.errorInvalidArgument);
		
		ArrayNodeList result = new ArrayNodeList();
		
		// Test the first argument and if it happens to be a string, then evaluate it differently
		if (arguments.get(0) instanceof String)
		{
			for (int i = 0, argCount = arguments.size(); i < argCount; i++)
			{
				evaluateExpression(result, SMLValidatorUtil.removeLineBreaks((String)arguments.get(i)));
			}
			
			return result;
		}
		
		
		// Otherwise assume that the content of the arguments list is a node list 
		// Retrieve the data structures required
		SMLIFIdentity identity = (SMLIFIdentity) SMLValidatorUtil.retrieveDataStructure(IdentityDataBuilder.ID);
		Map<String, Map<String, TypeNode>> inheritanceMap = (Map<String, Map<String, TypeNode>>) SMLValidatorUtil.retrieveDataStructure(TypeInheritanceDataBuilderImpl.ID);
		inheritanceMap = inheritanceMap == null ? new Hashtable<String, Map<String, TypeNode>>() : inheritanceMap;
		
		if (identity == null)
			throw new XPathFunctionException(SMLValidationMessages.errorMissingStructure);
		
		NodeList nodeList = null;
		if ((nodeList = (NodeList)arguments.get(0)).getLength() <= 0)
			throw new XPathFunctionException(SMLValidationMessages.errorInvalidArgument);		
		
		// For every node in the node list
		for (int i = 0, nodeCount = nodeList.getLength(); i < nodeCount; i++)
		{
			Node currentNode = nodeList.item(i);
			
			// Determine the uri and name of the node
			String localName = currentNode.getLocalName();
			String uri = currentNode.getNamespaceURI();
			uri = uri == null ? IValidationConstants.EMPTY_STRING : uri;						
			if (localName == null)
				throw new XPathFunctionException(SMLValidationMessages.errorDetermineName);
			
			// Throw an exception if the node is not a reference
			if (!SMLValidatorUtil.isReference(currentNode))
				throw new XPathFunctionException(NLS.bind(SMLValidationMessages.errorWrongType, localName));
			
			
			// Find the reference URI
			NodeList currentChildNodes = currentNode.getChildNodes();
			if (nodeList == null)
			{
				continue;
			}
			
			StringBuffer referenceExpression = new StringBuffer();
			for (int j = 0, childNodeCount = currentChildNodes.getLength(); j < childNodeCount; j++)
			{
				Node currentChildNode = currentChildNodes.item(j);
				if (ISMLConstants.SML_URI.equals(currentChildNode.getNamespaceURI()) && ISMLConstants.URI_ELEMENT.equals(currentChildNode.getLocalName()))
				{
					NodeList uriNodes = currentChildNode.getChildNodes();
					for (int k = 0, uriNodeCount = uriNodes.getLength(); k < uriNodeCount; k++)
					{
						Node uriNode = uriNodes.item(k);
						if (Node.TEXT_NODE == uriNode.getNodeType() && uriNode.getNodeValue() != null)
							referenceExpression.append(SMLValidatorUtil.removeLineBreaks(uriNode.getNodeValue(), false));
					}
				}
			}
			
			String referenceExp = referenceExpression.toString();
			if (referenceExp == null)
			{
				continue;
			}
			
			// Evaluate the reference
			evaluateExpression(result, referenceExp);
		}
		
		return result;
	}

	public void evaluateExpression(ArrayNodeList result, String referenceExp) throws XPathFunctionException
	{
		Exception exception = null;		
		try
		{
			URIReference uriReference = new URIReference(referenceExp);
			Node context = uriReference.retrieveDocumentDOM();
			
			if (context == null)
			{
				return;
			}
			
			String fragmentExpression = uriReference.getFragment();
			if (fragmentExpression == null)
			{
				result.add(context);
				return;
			}
							
			XPointerExpression xpointerExpression = XPointer.compile(fragmentExpression);
			result.add((Node)xpointerExpression.evaluate(context));				

		} catch (XPointerSyntaxException e)
		{
			exception = e;
		} catch (URISyntaxException e)
		{
			exception = e;
		} catch (BadContextException e)
		{
			exception = e;
		} catch (BadExpressionException e)
		{
			exception = e;
		}
		
		if (exception != null)
			throw new XPathFunctionException(exception); 
	}

	/**
	 * Evaluates expression to a node object
	 * @param referenceExp uri expression
	 * @return return a node object
	 * @throws XPathFunctionException thrown if an error occurs during evaluation
	 */
	public Object evaluate(String referenceExp) throws XPathFunctionException
	{
		SMLIFIdentity identity = (SMLIFIdentity) SMLValidatorUtil.retrieveDataStructure(IdentityDataBuilder.ID);
		if (identity == null)
			throw new XPathFunctionException(SMLValidationMessages.errorMissingStructure);
		
		Exception exception = null;
		if (referenceExp == null)
		{
			return null;
		}
		
		// Evaluate the reference
		try
		{
			URIReference uriReference = new URIReference(referenceExp);
			Node context = uriReference.retrieveDocumentDOM();			
			if (context == null)
			{
				return null;
			}
			
			String fragmentExpression = uriReference.getFragment();
			if (fragmentExpression == null)
			{
				return context;
			}
							
			XPointerExpression xpointerExpression = XPointer.compile(fragmentExpression);
			return xpointerExpression.evaluate(context);				

		} catch (XPointerSyntaxException e)
		{
			exception = e;
		} catch (URISyntaxException e)
		{
			exception = e;
		} catch (BadContextException e)
		{
			exception = e;
		} catch (BadExpressionException e)
		{
			exception = e;
		}
		
		if (exception != null)
			throw new XPathFunctionException(exception); 		
		return null;
	}
	/**
	 * Returns an instance of this class
	 * 
	 * @return The instance of this class
	 */
	public static XPathFunction instance()
	{
		if (instance == null)
			instance = new DerefXPathFunction();
		return instance;
	}
	
	
	private static class ArrayNodeList extends ArrayList<Node> implements NodeList
	{
		/**
		 * Serial version UID
		 */
		private static final long serialVersionUID = 7992097730072474662L;

		public int getLength()
		{
			return super.size();
		}

		public Node item(int index)
		{
			return (Node)super.get(index);
		}		
	}
}
