/**********************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.reference;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.Hashtable;
import java.util.Map;

import org.eclipse.cosmos.rm.internal.validation.common.ISMLConstants;
import org.eclipse.cosmos.rm.internal.validation.common.IValidationConstants;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidationMessages;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidatorUtil;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidatorUtil.RemoteRetrievalException;
import org.eclipse.cosmos.rm.internal.validation.databuilders.DocumentDOMBuilder;
import org.eclipse.cosmos.rm.internal.validation.databuilders.IdentityDataBuilder;
import org.w3c.dom.Node;


/**
 * Represents an URI as specified by the grammar: 
 * {@link http://www.ietf.org/rfc/rfc3986.txt}
 * 
 * @author Ali Mehregani
 */
public class URIReference implements IReferenceExpression
{	
	/**
	 * Reserved URI characters
	 */
	private static final char[] RESERVED_CHARACTERS = new char[] {
		'$', '&', '+', ',', '/',
		':', ';', '=', '?', '@',
		'-', '_', '.', '!', '*',
		'(', ')', '#', '[', ']', 
		'%', '\'', '\\'};
	
	/**
	 * The base URI
	 */
	private URI base;
	
	/**
	 * The reference URI.
	 */
	private URI reference;
	
	/**
	 * The document reference
	 */
	private String documentReference;

	/**
	 * Indicates whether this reference has already been
	 * transformed.
	 */
	private boolean transformed;
	
	
	/**
	 * Contains the aliases of the documents
	 * 
	 * KEY = alias names
	 * VALUE = root element name
	 */
	private Map<String, Node> aliases;
	
	
	/**
	 * Constructor.  The reference must be a valid XPointer expression.
	 * 
	 * @param reference The string representing the XPointer expression 
	 * @throws URISyntaxException If the URI has an incorrect syntax
	 */
	public URIReference(String reference) throws URISyntaxException
	{		
		if (reference == null)
			throw new URISyntaxException(null, SMLValidationMessages.errorReferenceNullURI);		
		
		// Trim the white spaces and take out line separators
		reference = trim(reference);
		
		// Encode the unsupported US-ASCII characters
		reference = encodeCharacters (reference);
		
		this.reference = new URI(reference);
	}


	private String encodeCharacters(String uri)
	{
		if (uri == null)
		{
			return null;
		}
		char[] uriChars = uri.toCharArray();
		StringBuffer buffer = new StringBuffer();
		for (int i = 0; i < uriChars.length; i++)
		{
			
			if (Character.isLetterOrDigit(uriChars[i])		||		// [0-9][A-Z][a-z]
				isCharacterReserved(uriChars[i]))					// [$&+,/:;=?@-_.!*'()#[]]
			{
				buffer.append(uriChars[i]);
				continue;
			}
			
			// Otherwise convert to hex and add to the buffer
			// The hex is expected to be 2 digit max
			buffer.append("%");
			buffer.append(convertToHex(((int)uriChars[i])/16));			
			buffer.append(convertToHex(((int)uriChars[i])%16));
		}
		
		return buffer.toString();
	}

	
	private char convertToHex(int dec)
	{
		return 	dec >= 0 && dec <= 9 ? String.valueOf(dec).toCharArray()[0] :		// [0-9]
				dec <= 15 ? (char)(65 + (dec - 10)) :								// [A-F]
				(char)dec; 															// Should never happen 	
	}

	private int convertToDecimal(char ch)
	{
		return 	Character.isDigit(ch) ? Integer.parseInt(String.valueOf(ch)) :		// [0-9]
				ch >= 65 && ch <= 70 ? 10 + (ch - 65) : 							// [A-F]
				-1;																	// Not a hex digit
	}
	
	private String decodeCharacters(String uri)
	{
		if (uri == null)
		{
			return null;
		}
		
		StringBuffer buffer = new StringBuffer();
		int cursorInx = 0;
		int precentInx;
		int uriPartLength = uri.length();
		while (uriPartLength > cursorInx && (precentInx = uri.indexOf('%', cursorInx)) >= 0)
		{
			buffer.append(uri.substring(cursorInx, precentInx));
			if (uriPartLength > precentInx + 2)
			{				
				char firstDigit = uri.charAt(precentInx + 1);
				char secondDigit = uri.charAt(precentInx + 2);
				
				int firstDecimalDigit = convertToDecimal(firstDigit);
				int secondDecimalDigit = convertToDecimal(secondDigit);
				
				// Convert if this is a number
				if (firstDecimalDigit >= 0 && secondDecimalDigit >= 0)
				{
					buffer.append((char)(firstDecimalDigit*16 + secondDecimalDigit));
				}
				// Otherwise just append blindly 
				else
				{
					buffer.append('%' + firstDigit + secondDigit);
				}
			}
			cursorInx = precentInx + 3;
		}
		
		if (uriPartLength > cursorInx)
		{
			buffer.append(uri.substring(cursorInx));
		}
		return buffer.toString();
	}


	private boolean isCharacterReserved(char c)
	{
		for (int i = 0; i < RESERVED_CHARACTERS.length; i++)
		{
			if (c == RESERVED_CHARACTERS[i])
				return true;
		}
		
		return false;
	}


	private String trim(String str)
	{
		str = str.trim();
		str = str.replaceAll("\\n", "");
		str = str.replaceAll("\\r", "");
		return str;
	}
	
	
	/**
	 * The document reference is basically the <scheme>+<authority>+<path> 
	 * of the URI.
	 * 
	 * @return the documentAlias
	 * @throws URISyntaxException 
	 */
	public String getDocumentReference() throws URISyntaxException
	{		
		if (documentReference == null)
		{
			String fragment = reference.getFragment();
			String referenceStr = reference.toString();
			int fragmentInx = -1;
			if (fragment != null && fragment.length() > 0 && (fragmentInx = referenceStr.indexOf(fragment)) > 0)
			{				
				referenceStr = referenceStr.substring(0, fragmentInx - 1);				
			}
			documentReference = referenceStr;
			
			/* Transform the URI if the aliases cannot be found */
			if (!isDocumentEmbedded(referenceStr) && !isTransformed())
			{
				documentReference = null;
				transform();
				return getDocumentReference();
			}
		}
		
		return decodeCharacters(documentReference);
	}


	/**
	 * Returns the query string of the URI
	 * 
	 * @return The query of the reference
	 * @throws URISyntaxException 
	 */
	public String getFragment() throws URISyntaxException
	{
		/* This has the side effect of transforming the URI if needed */
		getDocumentReference();		
		return reference == null ? null : decodeCharacters(reference.getFragment());
	}
	
	@SuppressWarnings("unchecked")
	private boolean isDocumentEmbedded(String document)
	{
		if (aliases == null)
		{
			aliases = (Map<String, Node>)SMLValidatorUtil.retrieveDataStructure(DocumentDOMBuilder.ID);
			aliases = aliases == null ? new Hashtable<String, Node>() : aliases;
		}
		
		return aliases.get(document) != null;
	}


	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.reference.IReferenceExpression#isTransformed()
	 */
	public boolean isTransformed()
	{
		return transformed;
	}


	/**
	 * The transformation will be based on [RFC 3986]: {@link http://www.ietf.org/rfc/rfc3986.txt}.
	 * See section 5 for more details.
	 * 
	 * @throws URISyntaxException If the base URI is invalid 
	 * @see org.eclipse.cosmos.rm.internal.validation.reference.IReferenceExpression#transform()
	 */
	public void transform() throws URISyntaxException
	{		
		transformed = true;
		
		// A transformation is not required if the URI is either absolute or only
		// contains a fragment
		if (reference.isAbsolute() || isFragmentOnly())
		{
			return;
		}
		
		if (base == null)
		{
			String baseURI = IdentityDataBuilder.retrieveBaseURI();
			if (baseURI == null)
				throw new URISyntaxException(IValidationConstants.EMPTY_STRING, SMLValidationMessages.errorBaseNullURI);
			base = new URI(baseURI);
		}
		
		String scheme, authority, path, query, fragment;
		
		// Determine the base URI
		if (defined(reference.getScheme())) 
		{
			scheme = reference.getScheme();
			authority = reference.getAuthority();
			path = reference.getPath();
			query = reference.getQuery();			
		}
		else
		{
			if (defined(reference.getAuthority()))
			{
				authority = reference.getAuthority();
				path = reference.getPath();
				query = reference.getQuery();
			}
			else
			{
				if (defined(reference.getPath())) {
					path = reference.getPath().startsWith(ISMLConstants.FORWARD_SLASH) ? 
							removeDotSegments(reference.getPath()) : removeDotSegments(mergePath(base, reference));
					query = reference.getQuery();
				} else {
					path = base.getPath();
					query = defined(reference.getQuery()) ? reference.getQuery() : base.getQuery();					
				}
				authority = base.getAuthority();				
			}
			scheme = base.getScheme();
		}
		fragment = reference.getFragment();
		
		try
		{
			reference = new URI(decodeCharacters(scheme), 
								decodeCharacters(authority), 
								decodeCharacters(path), 
								decodeCharacters(query), 
								decodeCharacters(fragment));
		} 
		catch (URISyntaxException e)
		{
			// This should never happen
		}
	}


	private boolean isFragmentOnly()
	{		
		return 	isNull(reference.getAuthority()) && isNull(reference.getHost()) && isNull(reference.getPath()) &&
				isNull(reference.getQuery()) && !isNull(reference.getFragment());
	}

	
	private boolean isNull(String str)
	{
		return str == null || str.length() <= 0;
	}

	/**
	 * See page 32 of {@link http://www.ietf.org/rfc/rfc3986.txt}
	 * 
	 * @param base The base URI
	 * @param relative The relative uri
	 * @return Merged path of the base and the relative URI
	 */
	private String mergePath(URI base, URI relative)
	{
		if (defined(base.getAuthority()) && !defined(base.getPath()))
		{
			return ISMLConstants.FORWARD_SLASH + relative.getPath();
		}
		else
		{
			String basePath = base.getPath();
			int rightMostSlash = basePath.lastIndexOf(ISMLConstants.FORWARD_SLASH);
			basePath = rightMostSlash > 0 ? basePath.substring(0, rightMostSlash + 1) : IValidationConstants.EMPTY_STRING;
			return basePath + relative.getPath();
		}		
	}


	/**
	 * See page 33 of {@link http://www.ietf.org/rfc/rfc3986.txt} 
	 * 
	 * @param input The input path
	 * @return removes the dot segments based on the RFC above
	 */
	private String removeDotSegments(String input)
	{
		String output = "";
		String matchedPrefix;
		
		while (input.length() > 0)
		{
			/* If the input buffer begins with a prefix of "../" or "./",
	           then remove that prefix from the input buffer; otherwise*/
			if ((matchedPrefix = findMatch(input, new String[]{"../", "./"}, new String[0])) != null) 
			{
				int length = matchedPrefix.length();
				input = input.length() > length ? input.substring(length) : IValidationConstants.EMPTY_STRING;
			}
			
			/* if the input buffer begins with a prefix of "/./" or "/.",
	           where "." is a complete path segment, then replace that
	           prefix with "/" in the input buffer; otherwise, */
			else if ((matchedPrefix = findMatch(input, new String[]{"/./"}, new String[]{"/."})) != null) 	
			{
				int length = matchedPrefix.length();
				input = input.length() > length ? input.substring(length) : IValidationConstants.EMPTY_STRING;
				input = ISMLConstants.FORWARD_SLASH + input;
			}

			
			/* if the input buffer begins with a prefix of "/../" or "/..",
			   where ".." is a complete path segment, then replace that
           	   prefix with "/" in the input buffer and remove the last
           	   segment and its preceding "/" (if any) from the output
           	   buffer; otherwise, */
			else if ((matchedPrefix = findMatch(input, new String[]{"/../"}, new String[]{"/.."})) != null) 	
			{
				int length = matchedPrefix.length();
				input = input.length() > length ? input.substring(length) : IValidationConstants.EMPTY_STRING;
				input = ISMLConstants.FORWARD_SLASH + input;
				
				int inx = output.lastIndexOf(ISMLConstants.FORWARD_SLASH);
				output = inx > 0 ? output.substring(0, inx) : IValidationConstants.EMPTY_STRING;  				
			}
			
			/* if the input buffer consists only of "." or "..", then remove
           	   that from the input buffer; otherwise, */
			else if ((matchedPrefix = findMatch(input, new String[0], new String[]{".", ".."})) != null)
			{
				input = IValidationConstants.EMPTY_STRING;				
			}
			
			/* move the first path segment in the input buffer to the end of
           	   the output buffer, including the initial "/" character (if
           	   any) and any subsequent characters up to, but not including,
           	   the next "/" character or the end of the input buffer. */
			else
			{
				int inx = input.indexOf(ISMLConstants.FORWARD_SLASH);
				
				if (inx == 0)
				{
					output += ISMLConstants.FORWARD_SLASH;
					input = input.substring(1);
					inx = input.indexOf(ISMLConstants.FORWARD_SLASH);
				}
					
				if (inx > 0)
				{
					output += input.substring(0, inx);
					input = input.substring(inx);
				}
				else
				{
					output += input;
					input = IValidationConstants.EMPTY_STRING;
				}				 
			}
			
		}
		
		return output;
	}


	private String findMatch(String input, String[] prefix, String[] equalityCheck)
	{
		for (int i = 0; i < prefix.length; i++)
		{
			if (input.startsWith(prefix[i]))
				return prefix[i];
		}
		
		for (int i = 0; i < equalityCheck.length; i++)
		{
			if (input.equals(equalityCheck[i]))
				return equalityCheck[i];
		}
		
		return null;
	}


	private boolean defined(String field)
	{
		return field != null && field.length() > 0;
	}


	/**
	 * Retrieves the document node and returns the result
	 * 
	 * @return The document node
	 * @throws URISyntaxException 
	 * @throws RemoteRetrievalException 
	 */
	public Node retrieveDocumentDOM() throws URISyntaxException
	{
		String document = getDocumentReference();
		
		/* Return what's stored under aliases */
		String scheme = reference.getScheme();
		if (scheme == null)
		{
			return (Node)aliases.get(document);
		}
		
		/* Otherwise we'll need to download the reference document */ 
		try
		{
			return SMLValidatorUtil.retrieveRemoteDocument(reference.toString());
		} 
		catch (RemoteRetrievalException e)
		{
			return null;
		}
	}


	/**
	 * @return the reference
	 */
	protected URI getReference()
	{
		return reference;
	}


	/**
	 * @param reference the reference to set
	 */
	protected void setReference(URI reference)
	{
		this.reference = reference;
	}


	/**
	 * @return the base
	 */
	protected URI getBase()
	{
		return base;
	}


	/**
	 * @param base the base to set
	 */
	protected void setBase(URI base)
	{
		this.base = base;
	}	
	
}
