/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.smlvalidators;

import java.util.Iterator;
import java.util.Map;

import javax.xml.namespace.QName;
import javax.xml.xpath.XPathFunctionException;

import org.eclipse.cosmos.rm.internal.validation.artifacts.ConstraintNode;
import org.eclipse.cosmos.rm.internal.validation.artifacts.ElementLocation;
import org.eclipse.cosmos.rm.internal.validation.artifacts.ElementTypeMap;
import org.eclipse.cosmos.rm.internal.validation.artifacts.TargetElementNode;
import org.eclipse.cosmos.rm.internal.validation.artifacts.TypeDeclarationCollection;
import org.eclipse.cosmos.rm.internal.validation.artifacts.TypeNode;
import org.eclipse.cosmos.rm.internal.validation.common.IValidationOutput;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidationMessages;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidatorUtil;
import org.eclipse.cosmos.rm.internal.validation.common.AbstractValidationOutput.ValidationMessageFactory;
import org.eclipse.cosmos.rm.internal.validation.core.AbstractSMLValidator;
import org.eclipse.cosmos.rm.internal.validation.databuilders.DataBuilderRegistry;
import org.eclipse.cosmos.rm.internal.validation.databuilders.DocumentDOMBuilder;
import org.eclipse.cosmos.rm.internal.validation.databuilders.ElementDeclarationBuilder;
import org.eclipse.cosmos.rm.internal.validation.databuilders.ElementTypeMapDataBuilder;
import org.eclipse.cosmos.rm.internal.validation.databuilders.GroupDeclarationBuilder;
import org.eclipse.cosmos.rm.internal.validation.databuilders.SubstitutionBuilder;
import org.eclipse.cosmos.rm.internal.validation.databuilders.TargetInstanceBuilder;
import org.eclipse.cosmos.rm.internal.validation.databuilders.TargetSchemaBuilder;
import org.eclipse.cosmos.rm.internal.validation.databuilders.TypeDeclarationBuilder;
import org.eclipse.cosmos.rm.internal.validation.databuilders.TypeInheritanceDataBuilderImpl;
import org.eclipse.cosmos.rm.internal.validation.reference.DerefXPathFunction;
import org.eclipse.cosmos.rm.internal.validation.reference.URIReference;
import org.eclipse.cosmos.rm.internal.validation.util.ParserHelper;
import org.eclipse.osgi.util.NLS;
import org.w3c.dom.Element;

/**
 * This class is used to validate the sml:targetElement, sml:targetRequired and 
 * sml:targetType constraints. 
 * 
 * @author sleeloy
 * @author Ali Mehregani
 */
public class TargetValidator extends AbstractSMLValidator
{
	private Map<String, ConstraintNode> targetSchemaBuilderStructure;
	private Map<TargetElementNode, Object> targetInstanceBuilderStructure;
	private Map<String, Object> documentDOMBuilderElems;
	private Map<String, Map<String, String>> substitutionElementElems;
	private ElementTypeMap elementTypeMapBuilderStructure;
	private Map<String, Map<Object, TypeNode>> inheritance;	
	private TypeDeclarationBuilder typeDeclarationBuilder;
	private IValidationOutput<String, Object> validationLogger;
	
	
	public void initialize(Map<String, Object> validationAttribute)
	{
		super.initialize(validationAttribute);
		DataBuilderRegistry builderRegistry = DataBuilderRegistry.instance();
		
		builderRegistry.registerDataStructureBuilder(TypeInheritanceDataBuilderImpl.ID, new TypeInheritanceDataBuilderImpl());		
		builderRegistry.registerDataStructureBuilder(ElementTypeMapDataBuilder.ID, new ElementTypeMapDataBuilder());
		builderRegistry.registerDataStructureBuilder(TargetSchemaBuilder.ID, new TargetSchemaBuilder());
		builderRegistry.registerDataStructureBuilder(TargetInstanceBuilder.ID, new TargetInstanceBuilder());		
		builderRegistry.registerDataStructureBuilder(SubstitutionBuilder.ID, new SubstitutionBuilder());
		builderRegistry.registerDataStructureBuilder(DocumentDOMBuilder.ID, new DocumentDOMBuilder());	
		builderRegistry.registerDataStructureBuilder(TypeDeclarationBuilder.ID, new TypeDeclarationBuilder());
		builderRegistry.registerDataStructureBuilder(ElementDeclarationBuilder.ID, new ElementDeclarationBuilder());
		builderRegistry.registerDataStructureBuilder(GroupDeclarationBuilder.ID, new GroupDeclarationBuilder());
	}

	@SuppressWarnings("unchecked")
	public boolean validate()
	{
		setTaskName(SMLValidationMessages.validationTarget);
		validationLogger = getValidationOutput();
		documentDOMBuilderElems = (Map<String, Object>)SMLValidatorUtil.retrieveDataStructure(DocumentDOMBuilder.ID);
		substitutionElementElems = (Map<String, Map<String, String>>)SMLValidatorUtil.retrieveDataStructure(SubstitutionBuilder.ID);
		targetSchemaBuilderStructure = (Map<String, ConstraintNode>)SMLValidatorUtil.retrieveDataStructure(TargetSchemaBuilder.ID);
		targetInstanceBuilderStructure = (Map<TargetElementNode, Object>)SMLValidatorUtil.retrieveDataStructure(TargetInstanceBuilder.ID);
		elementTypeMapBuilderStructure = (ElementTypeMap)SMLValidatorUtil.retrieveDataStructure(ElementTypeMapDataBuilder.ID);
		inheritance = (Map<String, Map<Object, TypeNode>>)SMLValidatorUtil.retrieveDataStructure(TypeInheritanceDataBuilderImpl.ID);
		typeDeclarationBuilder = (TypeDeclarationBuilder)DataBuilderRegistry.instance().getDataStructureBuilder(TypeDeclarationBuilder.ID);
		
		
		// Validate the target constraints
		return validateValues() && validateTargetInstance();
	}
	
	
	/**
	 * Verifies that the values specified for sml:targetElement and sml:targetType
	 * are valid (i.e. The values exist in the scope of the document)
	 * 
	 * @return true if validation passes; false otherwise
	 */
	private boolean validateValues()
	{
		for (Iterator<String> elementNames = targetSchemaBuilderStructure.keySet().iterator(); elementNames.hasNext();)
		{
			String elementName = elementNames.next();
			ConstraintNode schemaNode = targetSchemaBuilderStructure.get(elementName);
			
			QName targetElement = schemaNode.getTargetElement();
			if (targetElement != null && elementTypeMapBuilderStructure.getType(targetElement.getNamespaceURI(), targetElement.getLocalPart()) == null)
			{
				ElementLocation location = schemaNode.getLocation();
				validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(
						location.getFilePath(), location.getLineNumber(), NLS.bind(SMLValidationMessages.targetElementInvalidValue, targetElement.getPrefix() + ":" + targetElement.getLocalPart())));
				return false;
			}		
			
			QName targetType = schemaNode.getTargetType();
			TypeDeclarationCollection typeCollection = typeDeclarationBuilder.getDataStructure();
			if (targetType != null && typeCollection.get(targetType) == null)
			{
				ElementLocation location = schemaNode.getLocation();
				validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(
						location.getFilePath(), location.getLineNumber(), NLS.bind(SMLValidationMessages.targetTypeInvalidValue, targetType.getPrefix() + ":" + targetType.getLocalPart())));
				return false;
			}
		}
		return true;
	}

	
	protected boolean validateTargetInstance()
	{
		Iterator<TargetElementNode> iter = targetInstanceBuilderStructure.keySet().iterator();
		
		// Check to make sure the type declarations are consistent
		if (!typeDeclarationBuilder.isStructureValid())
		{
			validationLogger.reportMessage(typeDeclarationBuilder.getErrorMessage());
			return false;
		}
		
		
		// For each element that has an associated target* constraint
		while (iter.hasNext())
		{
			TargetElementNode targetElementNode = (TargetElementNode)iter.next();
			ConstraintNode targetSchemaNode = (ConstraintNode)targetSchemaBuilderStructure.get(targetElementNode.getName());

			// Check substitution group
			if (targetSchemaNode == null){
				Map<String, String> subGroupMap = substitutionElementElems.get(targetElementNode.getUri());
				if (subGroupMap != null){
					Object subGroup = subGroupMap.get(targetElementNode.getName());
					if (subGroup != null)
						targetSchemaNode = (ConstraintNode)targetSchemaBuilderStructure.get(ParserHelper.removeNameSpace((String)subGroup));
				}
				
			}
			
			if (targetSchemaNode.getTargetRequired())
			{ 
				//check if target is required
				if (!isURIValid(targetElementNode.getTargetURI())){
					Object[] errorParams = new String[3];
					errorParams[0] = targetElementNode.getName();
					errorParams[1] = targetElementNode.getLocation().getLineNumber()+""; //$NON-NLS-1$
					errorParams[2] = targetElementNode.getLocation().getColumnNumber()+""; //$NON-NLS-1$
					validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(targetElementNode.getLocation().getFilePath(), targetElementNode.getLocation().getLineNumber(), NLS.bind(SMLValidationMessages.unresolveTargetURI,errorParams)));
					return false;									
				}				
			}
			
			//check if we need to check the target element
			if (targetSchemaNode.getTargetElement() != null){
				if (!validateTargetElement(targetElementNode, targetSchemaNode))
					return false;
			}
			if (targetSchemaNode.getTargetType() != null){
				if (!validateTargetType(targetElementNode, targetSchemaNode))
					return false;
			}
			
		}
		return true;		
	}
	protected boolean validateTargetType(TargetElementNode targetElementNode,ConstraintNode targetSchemaNode ){

		String targetElementUri = targetElementNode.getTargetURI();
		String targetElementNamespace = targetElementNode.getUri();
		
		try {
			URIReference targetURI = new URIReference(targetElementUri);
			targetElementUri = targetURI.getDocumentReference();
		}
		catch(Exception exc)
		{
			exc.printStackTrace();
		}
		
		//check if schema node matches instance node
		Object getTargetType = elementTypeMapBuilderStructure.getType(targetElementNode.getUri(), targetElementUri);			
		if ((getTargetType != null)&&(compareElementInstance(qNameToString(targetSchemaNode.getTargetType()), getTargetType))){
			return true;
		}
		else{
			//check inheritance tree
			if (getTargetType != null){
				Map<Object, TypeNode> inheritanceMap = inheritance.get(targetElementNamespace);//getTargetType);
				boolean valid = false;
				if (inheritanceMap != null){
					TypeNode parent = (TypeNode)inheritanceMap.get(getTargetType);
					while (parent != null){
						if (compareElementInstance(parent.getType(), qNameToString(targetSchemaNode.getTargetType()))){
							valid = true;
							break;
						}
						parent = (TypeNode)inheritanceMap.get(parent.getType());
						
					}
				}
				if (valid) 
					return true;
			}						
			
			String targetTypeUri = null;
			//the ref could be an alias
			//Search through alias datastructure for actual element
			Object alias = documentDOMBuilderElems.get(targetElementUri);
			if ((alias != null) && (alias instanceof Element)){
				targetTypeUri = ((Element)alias).getNodeName();
				targetElementNamespace  = ((Element)alias).getNamespaceURI();
				
			}
			
			if (targetTypeUri != null)
				getTargetType = elementTypeMapBuilderStructure.getType(targetElementNamespace, targetTypeUri);			
			if ((getTargetType != null) && compareElementInstance(getTargetType, qNameToString(targetSchemaNode.getTargetType()))){
				return true;
			}		
			else{
				//check inheritance tree
				if (getTargetType != null){

					Map<Object, TypeNode> inheritanceMap = inheritance.get(targetElementNamespace);
					boolean valid = false;
					if (inheritanceMap != null){
						TypeNode parent = (TypeNode)inheritanceMap.get(getTargetType);
						while (parent != null){
							if (compareElementInstance(parent.getType(), qNameToString(targetSchemaNode.getTargetType())))
							{
								valid = true;
								break;
							}
							parent = (TypeNode)inheritanceMap.get(parent.getType());
							
						}
					}
					if (valid) 
						return true;
				}						
				
			}
			//there could be a substition group
			Object substitionGroup = substitutionElementElems.get(targetElementUri);				
			boolean valid = false;
			while (substitionGroup != null){
				getTargetType = elementTypeMapBuilderStructure.getType(targetElementUri,(String)substitionGroup);
				if (compareElementInstance(substitionGroup, qNameToString(targetSchemaNode.getTargetType())))
				{
					valid = true;
					break;
				}			
				substitionGroup = substitutionElementElems.get(substitionGroup);
			}
			//if dangling then it's okay
			if (valid || alias == null) return true;					
		}
		Object[] errorParams = new String[5];
		errorParams[0] = targetElementNode.getName();
		errorParams[1] = targetElementNode.getLocation().getLineNumber() < 0 ? SMLValidationMessages.commonUnknown : targetElementNode.getLocation().getLineNumber()+""; //$NON-NLS-1$
		errorParams[2] = targetElementNode.getLocation().getColumnNumber() < 0 ? SMLValidationMessages.commonUnknown : targetElementNode.getLocation().getColumnNumber()+""; //$NON-NLS-1$
		errorParams[3] = targetElementUri;
		errorParams[4] = qNameToString(targetSchemaNode.getTargetType());
		validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(targetElementNode.getLocation().getFilePath(), targetElementNode.getLocation().getLineNumber(), NLS.bind(SMLValidationMessages.targetTypeInvalid,errorParams)));
		
		return false;			
	}
	
	protected boolean validateTargetElement(TargetElementNode targetElementNode,ConstraintNode targetSchemaNode ){

		String targetElementUri = targetElementNode.getTargetURI();
		String targetElementNamespace = targetElementNode.getUri();
		
		try {
			URIReference targetURI = new URIReference(targetElementUri);
			targetElementUri = targetURI.getDocumentReference();
		}
		catch(Exception exc)
		{
			exc.printStackTrace();
		}
		
		//check if schema node matches instance node
		if (compareElementInstance(qNameToString(targetSchemaNode.getTargetElement()), targetElementUri))
		{
			return true;
		}
		else{
			//the ref could be an alias
			//Search through alias datastructure for actual element
			Object alias = documentDOMBuilderElems.get(targetElementUri);
			if ((alias != null) && (alias instanceof Element)){
				targetElementUri = ((Element)alias).getNodeName();
				targetElementNamespace  = ((Element)alias).getNamespaceURI();
			}
			if (compareElementInstance(targetElementUri, qNameToString(targetSchemaNode.getTargetElement()))){
				return true;
			}				
			//there could be a substition group
			Map<String, String> substitionGroupMap = substitutionElementElems.get(targetElementNamespace);
			boolean valid = false;
			if (substitionGroupMap != null){
				Object substitionGroup = substitionGroupMap.get(targetElementUri); 
				while (substitionGroup != null){
					if (compareElementInstance(substitionGroup, qNameToString(targetSchemaNode.getTargetElement()))){
						valid = true;
						break;
					}			
					substitionGroup = ParserHelper.removeNameSpace((String)substitionGroup);
					substitionGroup = substitionGroupMap.get(substitionGroup);				
				}
			}
			//if dangling then it's okay
			if (valid || alias == null) return true;					
		}
		Object[] errorParams = new String[5];
		errorParams[0] = targetElementNode.getName();
		errorParams[1] = targetElementNode.getLocation().getLineNumber() < 0 ? SMLValidationMessages.commonUnknown : targetElementNode.getLocation().getLineNumber()+""; //$NON-NLS-1$
		errorParams[2] = targetElementNode.getLocation().getColumnNumber() < 0 ? SMLValidationMessages.commonUnknown : targetElementNode.getLocation().getColumnNumber()+""; //$NON-NLS-1$
		errorParams[3] = targetElementUri;
		
		QName targetElement = targetSchemaNode.getTargetElement();
		String prefix = targetElement.getPrefix();
		errorParams[4] = prefix != null && prefix.length() > 0 ? prefix + ":" + targetElement.getLocalPart() : targetElement.getLocalPart();
		
		validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(targetElementNode.getLocation().getFilePath(), targetElementNode.getLocation().getLineNumber(), NLS.bind(SMLValidationMessages.invalidTargetInstance,errorParams)));		
		return false;			
	}
	
	private String qNameToString(QName qName)
	{		
		String prefix = qName.getPrefix();
		return prefix == null || prefix.length() <= 0 ? qName.getLocalPart() : prefix + ":" + qName.getLocalPart();
	}

	protected boolean compareElementInstance(Object elem1, Object elem2)
	{
		return (ParserHelper.removeNameSpace((String)elem1).equals(ParserHelper.removeNameSpace((String)elem2)));
	}
			
	/**
	 * Make sure that the uri can be resolved.
	 * @param uri the uri string to resolve
	 * @return true if the uri can be resolved otherwise false.
	 */
	protected static boolean isURIValid(String uri){
		try {
			return (((DerefXPathFunction)DerefXPathFunction.instance()).evaluate(SMLValidatorUtil.removeLineBreaks(uri)) != null);
		} catch (XPathFunctionException e) {
			return false;
		}
	}
}
