/*******************************************************************************
 * Copyright (c) 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.dc.broker.provisional.client;

import java.util.ArrayList;
import java.util.List;

import org.apache.axiom.om.OMAbstractFactory;
import org.apache.axiom.om.OMElement;
import org.apache.axiom.om.OMFactory;
import org.apache.axiom.om.OMNamespace;
import org.apache.axis2.AxisFault;
import org.apache.axis2.Constants;
import org.apache.axis2.addressing.EndpointReference;
import org.apache.axis2.client.Options;
import org.apache.axis2.client.ServiceClient;
import org.eclipse.cosmos.dc.broker.provisional.datamodel.BrokerConstants;
import org.eclipse.cosmos.dc.broker.provisional.datamodel.Group;
import org.eclipse.cosmos.dc.broker.provisional.datamodel.Groups;

public class BrokerClient {
	private EndpointReference epr;
	private ServiceClient sender = null;

	/**
	 * Constructor:  Creates a remote client of a COSMOS broker. 
	 * @param eprstr EPR of the broker
	 */
	public BrokerClient(String eprstr) {
		this.epr = new EndpointReference(eprstr);
	}
	
	private void initSender() throws BrokerException {
		if (sender != null) return;
		
        Options options = new Options();
        options.setTo(epr);
        options.setTransportInProtocol(Constants.TRANSPORT_HTTP);
        try {
        	sender = new ServiceClient();
        } catch (AxisFault fault) {
        	throw new BrokerException(fault.getMessage(), fault);
        }
        sender.setOptions(options);
	}
	
	/**
	 * Register a group of services with the broker.
	 * @param group
	 * @return true if the group is registered, 
	 * false if a group with the same identity has already been registered.
	 * @throws BrokerDatabaseException unexpected errors occurred
	 */
	public boolean registerGroup(Group group) throws BrokerException {
		initSender();
		
		OMFactory fac = OMAbstractFactory.getOMFactory();
        OMNamespace omNs = fac.createOMNamespace(BrokerConstants.BROKER_NAMESPACE, "tns");
        OMElement registerGroupRequest = fac.createOMElement("registerGroup", omNs);
        registerGroupRequest.addChild(group.toOM());

		OMElement result = null;
		try {
			result = sender.sendReceive(registerGroupRequest);
			if (result.getText().equals(BrokerConstants.REGISTRATION_SUCCESSFUL)) {
				return true;
			} else if (result.getText().equals(BrokerConstants.REGISTRATION_DUPLICATED_SERVICE)) {
				return false;
			}
		} catch (AxisFault fault) {
			throw new BrokerException(fault.getMessage(), fault);
		}

		return false;
	}
	
	/**
	 * Deregister a service group at the broker. 
	 * @param hostname  Hostname of the server that hosts the service group
	 * @param groupName The name of the service group.
	 * @return true if deregistration is successful, false if the group identified does not exist on the broker. 
	 * @throws BrokerDatabaseException unexpected error occurred
	 */
	public boolean deregisterGroup(String hostname, String groupName) throws BrokerException {
		initSender();
		
		OMFactory fac = OMAbstractFactory.getOMFactory();
        OMNamespace omNs = fac.createOMNamespace(BrokerConstants.BROKER_NAMESPACE, "tns");
        OMElement deregisterGroupRequest = fac.createOMElement("deregisterGroup", omNs);
        OMElement hostxml = fac.createOMElement("host", omNs, deregisterGroupRequest);
        OMElement groupNamexml = fac.createOMElement("name", omNs, deregisterGroupRequest);
        fac.createOMText(hostxml, hostname);
        fac.createOMText(groupNamexml, groupName);
        
        OMElement result = null;
        try {
        	result = sender.sendReceive(deregisterGroupRequest);
			if (result.getText().equals(BrokerConstants.DEREGISTRATION_SUCCESSFUL)) {
				return true;
			} else if (result.getText().equals(BrokerConstants.DEREGISTRATION_FAILED)) {
				return false;
			}
        } catch (AxisFault fault) {
        	throw new BrokerException(fault.getMessage(), fault);
        }
		
        return false;
	}
	
	/**
	 * Gets all service groups (data managers) registered at the broker
	 * @return
	 * @throws BrokerException
	 */
	public List<DataManager> getDataManagers() throws BrokerException {
		initSender();
		
        // TODO: NOTE:  This method is now return all service groups at the broker.
		// It is including groups that don't include the COSMOS web service. 
		OMFactory fac = OMAbstractFactory.getOMFactory();
        OMNamespace omNs = fac.createOMNamespace(BrokerConstants.BROKER_NAMESPACE, "tns");
        OMElement getAllGroupsRequest = fac.createOMElement("getAllGroups", omNs);

        OMElement result = null;
        try {
        	result = sender.sendReceive(getAllGroupsRequest);
        } catch (AxisFault fault) {
        	throw new BrokerException(fault.getMessage(), fault);
        }
        
        Groups groups = Groups.getGroups(result);
        
        List<DataManager> dataManagers = new ArrayList<DataManager>();
        for (Group group : groups.getGroups()) {
        	DataManager dm = new DataManager(group);
       		dataManagers.add(dm);
        }
        return dataManagers;
	}
	
	/**
	 * Query for a service group (data manager) from a broker. 
	 * @param hostname  Hostname of the server that hosts the service group
	 * @param groupName name of the service group
	 * @return data manager object, null if the specified data manager is not registered at this broker.
	 * @throws BrokerException unexpected errors occurred
	 */
	public DataManager getDataManager(String hostname, String groupName) throws BrokerException {
		initSender();
		
		OMFactory fac = OMAbstractFactory.getOMFactory();
        OMNamespace omNs = fac.createOMNamespace(BrokerConstants.BROKER_NAMESPACE, "tns");
        OMElement deregisterGroupRequest = fac.createOMElement("getGroup", omNs);
        OMElement hostxml = fac.createOMElement("host", omNs, deregisterGroupRequest);
        OMElement groupNamexml = fac.createOMElement("name", omNs, deregisterGroupRequest);
        fac.createOMText(hostxml, hostname);
        fac.createOMText(groupNamexml, groupName);
        
        OMElement result = null; 
        try {
        	result = sender.sendReceive(deregisterGroupRequest);
        } catch (AxisFault fault) {
        	throw new BrokerException(fault.getMessage(), fault);
        }
		
        if (result != null) {
        	return new DataManager(Group.getGroup(result));
        }
        
        return null;
	}
	
	/**
	 * Query for data managers registered at the broker that contains 
	 * a web service with the specified namespace. 
	 * @param ns  namespace of web service
	 * @return  all service groups (data managers) that contain a web service with the namespace defined in the parameter.
	 * @throws BrokerException  unexpected error occurred
	 */
	public List<DataManager> getDataManagerByNamespace(String ns) throws BrokerException {
		initSender();
		
		OMFactory fac = OMAbstractFactory.getOMFactory();
        OMNamespace omNs = fac.createOMNamespace(BrokerConstants.BROKER_NAMESPACE, "tns");
        OMElement getDataManagerByNamespaceRequest = fac.createOMElement("getGroupsByNamespace", omNs);
        OMElement hostxml = fac.createOMElement("namespace", omNs, getDataManagerByNamespaceRequest);
        fac.createOMText(hostxml, ns);
        
        OMElement result = null; 
        try {
        	result = sender.sendReceive(getDataManagerByNamespaceRequest);
        } catch (AxisFault fault) {
        	throw new BrokerException(fault.getMessage(), fault);
        }
		
        Groups groups = Groups.getGroups(result);
        
        List<DataManager> dataManagers = new ArrayList<DataManager>();
        for (Group group : groups.getGroups()) {
        	DataManager dm = new DataManager(group);
       		dataManagers.add(dm);
        }
        return dataManagers;
	}
}
