/**********************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.artifacts;

import java.net.URISyntaxException;

import org.eclipse.cosmos.rm.internal.validation.reference.URIReference;

/**
 * An element edge represents an edge from a target to a source
 * element from/in the contained document
 * 
 * @author Ali Mehregani
 */
public class ElementEdge
{
	/**
	 * The reference string that will be used to resolve the target 
	 * document and element.
	 */
	private String reference;

	/**
	 * The reference namespace URI
	 */
	private String referenceNameSpace;
	
	/**
	 * The source element
	 */
	private String referenceType;	
	
	/**
	 * The reference expression
	 */
	private URIReference referenceURI;
	
	/**
	 * The line number of the reference in the original file
	 */
	private int lineNumber;
	
	
	/**
	 * The constructor
	 * 
	 * @param referenceNameSpace The namespace URI for the reference
	 * @param referenceType The reference type
	 * @param lineNumber The line number of the reference in the original file 
	 */
	public ElementEdge(String referenceNameSpace, String referenceType, int lineNumber)
	{
		reference = "";
		this.referenceNameSpace = referenceNameSpace;
		this.referenceType = referenceType;
		this.lineNumber = lineNumber;
	}
	
	
	/**
	 * @return the referenceNameSpace
	 */
	public String getReferenceNameSpace()
	{
		return referenceNameSpace;
	}


	/**
	 * @param referenceNameSpace the referenceNameSpace to set
	 */
	public void setReferenceNameSpace(String referenceNameSpace)
	{
		this.referenceNameSpace = referenceNameSpace;
	}


	/**
	 * Set the source element
	 * 
	 * @param sourceElement The source element
	 */
	public void setReferenceType(String sourceElement)
	{
		this.referenceType = sourceElement;
	}

	
	/**
	 * Returns the source element
	 * 
	 * @return source element
	 */
	public String getReferenceType()
	{
		return referenceType;
	}
	
	
	/**
	 * Appends to the reference path
	 * 
	 * @param referece the string to be appended to the reference path
	 */
	public void appendToReference(String referece)
	{
		this.reference += referece;		
	}
	
	
	/**
	 * Returns the reference string
	 * 
	 * @return The reference string
	 */
	public String getReference()
	{
		return reference;
	}

	
	/**
	 * Sets the reference string
	 * 
	 * @param reference The reference string
	 */
	public void setReference(String reference)
	{
		this.reference = reference;
	}


	/**
	 * Returns the document that this edge orginates from
	 * 
	 * @return The alias of the document that this edge orginates
	 * from.
	 * @throws URISyntaxException 
	 */
	public String getDocument() throws URISyntaxException
	{		
		if (referenceURI == null)
			referenceURI = new URIReference(reference);
		return referenceURI.getDocumentReference();
	}


	/**
	 * @return the lineNumber
	 */
	public int getLineNumber()
	{
		return lineNumber;
	}


	/**
	 * @param lineNumber the lineNumber to set
	 */
	public void setLineNumber(int lineNumber)
	{
		this.lineNumber = lineNumber;
	}	
}
