/**********************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.artifacts;

import java.util.ArrayList;
import java.util.List;

/**
 * Model object to track schemas found in validating SML resources.
 * It serves the same purpose as described in the ElementModel class
 * comment.  In addition keeps track of the targetNamespace of the
 * schema, and any prerequisite schemas, for the purpose of validating
 * the schema.
 * 
 * @author David Whiteman
 */
public class ElementSchemaModel extends ElementModel {

	/**
	 * The namespace indicated by the targetNamespace attribute
	 * of the <schema> element.
	 */
	private String targetNamespace;
	
	/**
	 * The namespace context associated with this schema
	 * element
	 */
	private MappedNamespaceContext namespaceContext;
	
	/**
	 * Indicates whether the schema referenced by this model has already
	 * been validated by the validator.  Used to ensure that schemas aren't validated
	 * multiple times by the recursive process.
	 */
	private boolean validated;
	
	/**
	 * The namespace of schemas that have been imported
	 */
	private List<String> imported;
	
	
	public ElementSchemaModel(String name, String filePath, int lineNumber,
			int columnNumber) {
		super(name, filePath, lineNumber, columnNumber);
		namespaceContext = new MappedNamespaceContext();
		validated = false;
		imported = new ArrayList<String>();
	}
	
	public String toString() {
		return super.toString() + ", targetNamespace="+getTargetNamespace();
	}

	public String getTargetNamespace() {
		return targetNamespace;
	}

	public void setTargetNamespace(String targetNamespace) {
		this.targetNamespace = targetNamespace;
	}


	public boolean hasBeenValidated() {
		return validated;
	}

	public void markValidated() {
		this.validated = true;
	}

	public void addImportedSchema(String namespace) 
	{
		imported.add(namespace);
	}
	
	public List<String> getImportedSchemas()
	{
		return imported;
	}

	public boolean equals(Object o)
	{
		if (!(o instanceof ElementSchemaModel))
		{
			return false;
		}
		
		ElementSchemaModel elementSchemaModel = (ElementSchemaModel) o;
		return elementSchemaModel.getTargetNamespace().equals(this.getTargetNamespace()) && super.equals(o);
	}

	/**
	 * @return the namespaceContext
	 */
	public MappedNamespaceContext getNamespaceContext()
	{
		return namespaceContext;
	}

	/**
	 * @param namespaceContext the namespaceContext to set
	 */
	public void setNamespaceContext(MappedNamespaceContext namespaceContext)
	{
		this.namespaceContext = namespaceContext;
	}
}
