/**********************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.artifacts;

import java.util.Hashtable;
import java.util.Map;


/**
 * This structure is used to find the mapping between an element and its
 * associated type.
 * 
 * @author Ali Mehregani
 */
public class ElementTypeMap
{
	/**
	 * Stores the element types
	 * KEY = A URI indicating the namespace of the element
	 * VALUE = An object of type {@link Map} that stores the following:
	 * 		KEY = An object of type {@link String} indicating an element name
	 * 		VALUE = An object of type {@link String} indicating the type of the element
	 */
	private Map<String, Map<String, String>> types; 
	
	
	/**
	 * Constructor
	 */
	public ElementTypeMap()
	{
		types= new Hashtable<String, Map<String, String>>();
	}
	
	
	/**
	 * Adds an element declaration
	 * 
	 * @param uri The associated URI of the element 
	 * @param name The name of the element
	 * @param type The type of the element
	 */
	public void addElementDeclaration(String uri, String name, String type)
	{
		ElementNode node = new ElementNode();
		node.setUri(uri);
		node.setName(name);
		node.setType(type);
		addElementDeclaration(node);
	}
	
	/**
	 * Adds an element declaration
	 *  
	 * @param elementNode The element node, which is expected to have a valid name, type
	 * and an URI indicating its namespace
	 */
	public void addElementDeclaration(ElementNode elementNode)
	{
		String uri, name, type;
		
		/* It's expected for the element node to have a name and an URI */
		if ((uri = elementNode.getUri()) == null || (name = elementNode.getName()) == null || (type = elementNode.getType()) == null) {
			return;
		}
		
		Map<String, String> elementTypes = types.get(uri);
		if (elementTypes == null)
		{
			elementTypes = new Hashtable<String, String>();
			types.put(uri, elementTypes);
		}
		
		elementTypes.put(name, type);
	}
	
	/**
	 * Retrieves and returns the type of the element passed in
	 * 
	 * @param elementNode The element node
	 * @return The type of elementNode or null if none can be found
	 */
	public String getType(ElementNode elementNode)
	{
		return getType(elementNode.getUri(), elementNode.getName());
	}

	
	/**
	 * Retrieves and returns the type of an element
	 * 
	 * @param uri The uri of the element
	 * @param name The name of the element
	 * @return The type of the element or null if none can be found
	 */
	public String getType(String uri, String name)
	{
		if (uri == null) {
			return null;
		}
		Map<String, String> elementTypes = types.get(uri);
		return elementTypes == null ? null : elementTypes.get(name);
	}
}
