/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.artifacts;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.xml.namespace.NamespaceContext;

/**
 * Stores information about the identity constraints.  Using the alias
 * of a document, a client can retrieve all instances that have an
 * associated identity constraint.  In case a document is missing aliases,
 * its index is used to retrieve the same information.
 * 
 * <p/>
 * The format of the data stored: <br/>
 * [alias, data about the instance and its associated constraint] OR <br/>
 * [index of orphaned document, data about the instance and its constraint]
 * 
 * @author Ali Mehregani
 */
public class IdentityConstraintStructure
{					
	/**
	 * Identity constraint instance contained in documents with aliases 
	 * 
	 * KEY = A {@link String} indicating the document alias
	 * VALUE = A {@link List} containing elements of type IdentityConstraintInstance
	 */
	private Map<String, List<IdentityInstance>> constrainedInstances;
	
	/**
	 * Identity constraint instance contained in documents without an alias
	 *  
	 * KEY = A {@link String} indicating the index of the orphaned document
	 * VALUE = A {@link List} of elements containing objects of type IdentityConstraintInstance
	 */
	private Map<String, List<IdentityInstance>> orphanedConstrainedInstances;
	
	
	/**
	 * Constructor
	 */
	public IdentityConstraintStructure()
	{
		constrainedInstances = new Hashtable<String, List<IdentityInstance>>();
		orphanedConstrainedInstances = new Hashtable<String, List<IdentityInstance>>();
	}

	
	/**
	 * Adds a constrained instance to this structure
	 * 
	 * @param alias The alias of the document
	 * @param identityInstance The instance to be added
	 */
	public void addConstrainedInstance(String alias, IdentityInstance identityInstance)
	{			
		addToNestedList (constrainedInstances, alias, identityInstance);
	}
	
	
	/**
	 * Add the constraint instance passed in to the i-th orphaned
	 * document list 
	 * 
	 * @param orphanInx The index of the orphaned document list
	 * @param identityInstance The identity instance to be added 
	 */
	public void addOrphanedConstrainedInstance(int orphanInx, IdentityInstance identityInstance)
	{
		addToNestedList (orphanedConstrainedInstances, String.valueOf(orphanInx), identityInstance);
	}
	
	
	/**
	 * Given map M, key K, and value V, the following method will
	 * append V to the list contained in M(K).
	 * 
	 * @param map Map M
	 * @param key Key K
	 * @param value Value V
	 */
	private void addToNestedList(Map<String, List<IdentityInstance>> map, String key, IdentityInstance value)
	{
		List<IdentityInstance> nestedList = map.get(key);
		if (nestedList == null)
		{
			nestedList = new ArrayList<IdentityInstance>();
			map.put(key, nestedList);
		}
		nestedList.add(value);
	}
	
	
	/**
	 * Returns the identity constraint instances associated with the
	 * alias passed in
	 * 
	 * @param alias The alias of a document
	 * @return A list of associated identity constraints
	 */
	public IdentityInstance[] getConstrainedInstances(String alias)
	{
		List<IdentityInstance> list = constrainedInstances.get(alias);
		return list == null ? new IdentityInstance[0] : list.toArray(new IdentityInstance[list.size()]);
	}
	

	/**
	 * Return the aliases of the documents containing identity
	 * constraint instances
	 * 
	 * @return Aliases of documents containing identity constraint
	 * instances
	 */
	public String[] getConstrainedAliases()
	{
		String[] constrainedAliases = new String[constrainedInstances.size()];
		int counter = 0;
		for (Iterator<String> aliases = constrainedInstances.keySet().iterator(); aliases.hasNext();)
		{
			constrainedAliases[counter++] = aliases.next();			
		}
		
		return constrainedAliases;
	}
	
	
	/**
	 * Returns the identity constraint instances associated with the
	 * orphaned document with the index passed in
	 * 
	 * @param index The index of the orphaned document
	 * @return A list of associated identity constraints
	 */
	public IdentityInstance[] getOrphanedConstrainedInstances(int index)
	{
		List<IdentityInstance> list = orphanedConstrainedInstances.get(String.valueOf(index));
		return list == null ? new IdentityInstance[0] : list.toArray(new IdentityInstance[list.size()]);
	}

	
	/**
	 * Returns the indices of orphaned documents containing identity
	 * constraint instances.
	 * 
	 * @return The indices of documents containing identity constraint
	 * instances
	 */
	public int[] getOrphanedIndices()
	{
		int[] keys = new int[orphanedConstrainedInstances.size()];
		int counter = 0;
		for (Iterator<String> indices = orphanedConstrainedInstances.keySet().iterator(); indices.hasNext();)
		{
			String index = indices.next();
			keys[counter++] = Integer.parseInt(index);			
		}
		
		return keys;
	}
	
	
	/**
	 * Retrieves the identity constraints of this structure
	 * 
	 * @return The identity instances of this structure
	 */
	public IdentityInstance[] getIdentityConstraints()
	{
		List<IdentityInstance> instances = new ArrayList<IdentityInstance>();
		addInstances(instances, constrainedInstances);
		addInstances(instances, orphanedConstrainedInstances);
		return instances.toArray(new IdentityInstance[instances.size()]);
	}
	
	
	private void addInstances (List<IdentityInstance> instances, Map<String, List<IdentityInstance>> map)
	{
		for (Iterator<List<IdentityInstance>> identityInstances = map.values().iterator(); 
		 	 identityInstances.hasNext();)
		{
			instances.addAll(identityInstances.next());		
		}
	}
	
			
	/**
	 * Represents an instance with an associated identity constraint
	 * 
	 * @author Ali Mehregani
	 */
	public static class IdentityInstance
	{
		/**
		 * The indices leading to the context node
		 */
		private int[] nodePath;
		
		/**
		 * The constraint
		 */
		private IdentityConstraint constraint;

		/**
		 * The line number where this identity constraint
		 * appears
		 */
		private int lineNumber;
		
		
		/**
		 * @return the nodePath
		 */
		public int[] getNodePath()
		{
			return nodePath;
		}

		/**
		 * @param nodePath the nodePath to set
		 */
		public void setNodePath(int[] nodePath)
		{
			this.nodePath = nodePath;
		}

		/**
		 * @return the constraint
		 */
		public IdentityConstraint getConstraint()
		{
			return constraint;
		}

		/**
		 * @param constraint the constraint to set
		 */
		public void setConstraint(IdentityConstraint constraint)
		{
			this.constraint = constraint;
		}

		/**
		 * @return the lineNumber
		 */
		public int getLineNumber()
		{
			return lineNumber;
		}

		/**
		 * @param lineNumber the lineNumber to set
		 */
		public void setLineNumber(int lineNumber)
		{
			this.lineNumber = lineNumber;
		}		
	}
	
	
	
	/**
	 * Represents an identity constraint declaration
	 * 
	 * @author Ali Mehregani
	 */
	public static class IdentityConstraint
	{
		/**
		 * Indicates the key identity constraint
		 */
		public static final byte KEY_TYPE = 0x00;
		
		/**
		 * Indicates the key ref identity constraint
		 */
		public static final byte KEY_REF_TYPE = 0x01;
		
		/**
		 * Indicates the unique identity constraint
		 */
		public static final byte UNIQUE_TYPE = 0x02;
		
	
		/**
		 * The type of this declaration
		 */
		private byte type;
		
		/**
		 * The name of this declaration
		 */
		private String name;
		
		/**
		 * The fields associated with this constraint
		 */
		private List<String> fields;
		
		/**
		 * The xpath expression indicating the selector of this constraint
		 */
		private String selector;
		
		/**
		 * The refer attribute
		 */
		private String refer;
		
		/**
		 * The namespace associated with this
		 * constraint
		 */
		private NamespaceContext namespaceContext;
		
		/**
		 * Store the 'ref' attribute.  This attribute is different
		 * from the 'refer' attribute belonging to keyref constraints
		 */
		private String ref;
				
		/**
		 * The namespace of the element this constraint 
		 * belongs to
		 */
		private String namespace;
		
		
		/**
		 * Default constructor
		 */
		public IdentityConstraint()
		{
			this.fields = new ArrayList<String>();
		}
		
		
		/**
		 * Constructor
		 * 
		 * @param name The name of the declaration
		 */
		public IdentityConstraint(String name)
		{
			this();
			this.name = name;
		}

		
		/**
		 * @return the type
		 */
		public byte getType()
		{
			return type;
		}


		/**
		 * @param type the type to set
		 */
		public void setType(byte type)
		{
			this.type = type;
		}


		/**
		 * @return the name
		 */
		public String getName()
		{
			return name;
		}


		/**
		 * @param name the name to set
		 */
		public void setName(String name)
		{
			this.name = name;
		}


		/**
		 * @return the fields
		 */
		public String[] getFields()
		{
			return fields.toArray(new String[fields.size()]);
		}


		/**
		 * @param fields the fields to set
		 */
		public void addField(String fields)
		{
			if (fields == null)
			{
				return;
			}
			
			this.fields.add(fields);
		}


		/**
		 * @return the selector
		 */
		public String getSelector()
		{
			return selector;
		}


		/**
		 * @param selector the selector to set
		 */
		public void setSelector(String selector)
		{
			this.selector = selector;
		}


		/**
		 * @return the refer
		 */
		public String getRefer()
		{
			return refer;
		}


		/**
		 * @param refer the refer to set
		 */
		public void setRefer(String refer)
		{
			this.refer = refer;
		}


		/**
		 * @return the namespaceContext
		 */
		public NamespaceContext getNamespaceContext()
		{
			return namespaceContext;
		}


		/**
		 * @param namespaceContext the namespaceContext to set
		 */
		public void setNamespaceContext(NamespaceContext namespaceContext)
		{
			this.namespaceContext = namespaceContext;
		}


		/**
		 * @return the ref
		 */
		public String getRef()
		{
			return ref;
		}


		/**
		 * @param ref the ref to set
		 */
		public void setRef(String ref)
		{
			this.ref = ref;
		}


		/**
		 * @return the namespace
		 */
		public String getNamespace()
		{
			return namespace;
		}


		/**
		 * @param namespace the namespace to set
		 */
		public void setNamespace(String namespace)
		{
			this.namespace = namespace;
		}	
	}
}