/**********************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.artifacts;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.xml.namespace.NamespaceContext;

import org.eclipse.cosmos.rm.internal.validation.common.IValidationConstants;

public class MappedNamespaceContext implements NamespaceContext
{
	/**
	 * The URI, prefix relation is stored in a map
	 * KEY = URI
	 * VALUE = A list of prefix
	 */
	private Map<String, List<String>> contextWithURIKey;
	
	/**
	 * The URI, prefix relation is stored in a map
	 * KEY = Prefix
	 * VALUE = The URI
	 */
	private Map<String, String> contextWithPrefixKey;

	public String getNamespaceURI(String prefix)
	{
		if (contextWithPrefixKey == null)
			return null;
		String uri = (String)contextWithPrefixKey.get(prefix);
		return uri == null ? IValidationConstants.EMPTY_STRING : uri;
	}

	public String getPrefix(String uri)
	{
		return (String)getPrefixGeneric(uri, true);
	}

	@SuppressWarnings("unchecked")
	public Iterator<String> getPrefixes(String uri)
	{
		return (Iterator<String>)getPrefixGeneric(uri, false);
	}
	
	private Object getPrefixGeneric (String uri, boolean first)
	{
		if (contextWithURIKey == null)
			return null;
		List<String> prefixes = contextWithURIKey.get(uri);
		if (prefixes != null && prefixes.size() > 0)
			return first ? prefixes.get(0) : prefixes.iterator();
		return null;
	}
	
	public void addEntry (String prefix, String uri)
	{
		if (contextWithURIKey == null)
			contextWithURIKey = new Hashtable<String, List<String>>();
		if (contextWithPrefixKey == null)
			contextWithPrefixKey = new Hashtable<String, String>();
		
		List<String> listOfPrefix = contextWithURIKey.get(uri);
		if (listOfPrefix == null)
		{
			listOfPrefix = new ArrayList<String>();
			contextWithURIKey.put(uri, listOfPrefix);
		}
		listOfPrefix.add(prefix);
		contextWithPrefixKey.put(prefix, uri);
	}
	
	
	/**
	 * Add all the entries of the mapped namespace context to this
	 * mapped namespace context.
	 * 
	 * @param mappedNamespaceContext The entries of the mapped namespace
	 * context to be added
	 */
	public void addEntries (MappedNamespaceContext mappedNamespaceContext)
	{
		String[] prefixes = mappedNamespaceContext.getPrefixes();
		for (int i = 0; i < prefixes.length; i++)
		{
			addEntry (prefixes[i], mappedNamespaceContext.getNamespaceURI(prefixes[i]));
		}
	}
	
	
	/**
	 * Retrieves and returns all the namespaces
	 * 
	 * @return All namespaces
	 */
	public String[] getNamespaces()
	{
		return contextWithURIKey.keySet().toArray(new String[contextWithURIKey.size()]);
	}
	
	
	/**
	 * Retrieves and returns all prefixes
	 * 
	 * @return All prefixes
	 */
	public String[] getPrefixes()
	{
		return contextWithPrefixKey.keySet().toArray(new String[contextWithPrefixKey.size()]);
	}

	
	/**
	 * Removes the entries of thies namespacecontext
	 */
	public void clear()
	{
		contextWithURIKey.clear();
		contextWithPrefixKey.clear();
	}
	
}