/**********************************************************************
 * Copyright (c) 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.artifacts;

import java.util.ArrayList;
import java.util.List;

/**
 * Stores schema binding information
 * 
 * @author Ali Mehregani
 */
public class SchemaBindings
{
	/**
	 * The default schema that is to be applied to all
	 * instance documents
	 */
	private SchemaBinding defaultBinding;
	
	/**
	 * The bindings 
	 */
	private List<SchemaBinding> bindings;
	
	/**
	 * Indicates if schemaBinding is present in the current
	 * document being validated
	 */
	private boolean present;
	
	
	/**
	 * Constructor
	 */
	public SchemaBindings()
	{
		bindings = new ArrayList<SchemaBinding>();
	}
	
	
	/**
	 * Set the default binding that should be applied
	 * to all instance documents
	 * 
	 * @param binding the default binding
	 */
	public void setDefaultBinding(SchemaBinding schemaBinding)
	{
		this.defaultBinding = schemaBinding;
	}

	
	/**
	 * Returns the default schema binding
	 * 
	 * @return The default namespace
	 */
	public SchemaBinding getDefaultBinding()
	{
		return this.defaultBinding;
	}
	
	
	/**
	 * Add a new schema binding to this structure
	 * 
	 * @param schemaBinding The schema binding to be added
	 */
	public void addBinding(SchemaBinding schemaBinding)
	{		
		bindings.add(schemaBinding);	
	}
	
	
	/**
	 * Returns the schema bindings contained by this
	 * structure.
	 * 
	 * @return All schema bindings
	 */
	public SchemaBinding[] getBindings ()
	{
		return bindings.toArray(new SchemaBinding[bindings.size()]);
	}
	
	
	/**
	 * @return the present
	 */
	public boolean isPresent()
	{
		return present;
	}

	
	/**
	 * @param present the present to set
	 */
	public void setPresent(boolean present)
	{
		this.present = present;
	}



	/**
	 * Represents each individual schema binding
	 * 
	 * @author Ali Mehregani
	 */
	public static class SchemaBinding
	{
		/**
		 * The schema aliases
		 */
		private List<String> schemaAliases;
		
		/**
		 * The document aliases
		 */
		private List<String> documentAliases;

		
		/**
		 * Constructor
		 */
		public SchemaBinding()
		{
			schemaAliases = new ArrayList<String>();
			documentAliases = new ArrayList<String>();
		}
		
		
		/**
		 * Add a schema alias to this schema binding
		 * 
		 * @param alias The alias to be added
		 */
		public void addSchemaAlias(String alias)
		{
			schemaAliases.add(alias);
		}


		/**
		 * Retrieves and returns the schema aliases
		 * associated with this binding.
		 * 
		 * @return The schema aliases
		 */
		public String[] getSchemaAliases()
		{
			return schemaAliases.toArray(new String[schemaAliases.size()]);
		}

		
		/**
		 * Add a document alias to this schema binding
		 * 
		 * @param alias The alias to be added
		 */
		public void addDocumentAlias(String alias)
		{
			documentAliases.add(alias);
		}

		
		/**
		 * Retrieves and returns the schema aliases
		 * associated with this binding.
		 * 
		 * @return The schema aliases
		 */
		public String[] getDocumentAliases()
		{
			return documentAliases.toArray(new String[documentAliases.size()]);
		}
	}
}
