/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.common;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.lang.reflect.InvocationTargetException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;

import javax.xml.namespace.QName;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathFactory;
import javax.xml.xpath.XPathFunction;
import javax.xml.xpath.XPathFunctionResolver;

import org.apache.xerces.xs.ElementPSVI;
import org.apache.xerces.xs.PSVIProvider;
import org.apache.xerces.xs.XSAnnotation;
import org.apache.xerces.xs.XSComplexTypeDefinition;
import org.apache.xerces.xs.XSElementDeclaration;
import org.apache.xerces.xs.XSObjectList;
import org.apache.xerces.xs.XSTypeDefinition;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.content.IContentDescriber;
import org.eclipse.core.runtime.content.IContentType;
import org.eclipse.cosmos.rm.internal.validation.artifacts.ElementSchemaModel;
import org.eclipse.cosmos.rm.internal.validation.artifacts.ResourceWrapper;
import org.eclipse.cosmos.rm.internal.validation.common.AbstractValidationOutput.ValidationMessage;
import org.eclipse.cosmos.rm.internal.validation.common.AbstractValidationOutput.ValidationMessageFactory;
import org.eclipse.cosmos.rm.internal.validation.content.DefinitionContentDescriber;
import org.eclipse.cosmos.rm.internal.validation.content.InstanceContentDescriber;
import org.eclipse.cosmos.rm.internal.validation.content.SMLIFContentDescriber;
import org.eclipse.cosmos.rm.internal.validation.databuilders.DataBuilderRegistry;
import org.eclipse.cosmos.rm.internal.validation.databuilders.IDataBuilder;
import org.eclipse.cosmos.rm.internal.validation.databuilders.AbstractDataBuilder.AnnotationResult;
import org.eclipse.cosmos.rm.internal.validation.reference.DerefXPathFunction;
import org.w3c.dom.Document;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.SAXNotRecognizedException;
import org.xml.sax.SAXNotSupportedException;
import org.xml.sax.ext.LexicalHandler;
import org.xml.sax.helpers.DefaultHandler;

/**
 * Provides common utility method used by multiple classes under
 * different packages 
 * 
 * @author Ali Mehregani
 * @author David Whiteman
 */
public class SMLValidatorUtil
{
	/**
	 * A static XPath instance used to compile XPath expressions
	 */
	public static XPath xpath;
	
	/**
	 * A SAX parser factory
	 */
	private static SAXParserFactory saxParserFactory;
	
	static
	{
		XPathFactory factory = XPathFactory.newInstance();
	    xpath = factory == null ? null : factory.newXPath();
	    if (xpath != null)
	    {
	    	/* Register the dref XPath function extension */
	    	xpath.setXPathFunctionResolver(new XPathFunctionResolver()
	    	{
	    		/**
	    		 * @see XPathFunctionResolver#resolveFunction(QName, int)
	    		 */
				public XPathFunction resolveFunction(QName functionName, int arity)
				{
					if (ISMLConstants.SML_FN_URI.equals(functionName.getNamespaceURI()) && 
						ISMLConstants.DEREF_FN.equals(functionName.getLocalPart()) &&
						arity == DerefXPathFunction.ARITY)
					{
						return DerefXPathFunction.instance();
					}
					return null;
				}
			});
	    }
	    
	    
		SAXParserFactory saxParserFactory = SAXParserFactory.newInstance();
		saxParserFactory.setNamespaceAware(true);
		
		try
		{
			saxParserFactory.setFeature(ISMLConstants.SAX_PARSER_FEATURE_NAMESPACES, true);
			saxParserFactory.setFeature(ISMLConstants.SAX_PARSER_FEATURE_NAMESPACE_PREFIXES, true);
		} 
		catch (Exception e)
		{
			// Set the static instance varialbe to null to avoid re-use
			saxParserFactory = null;
		}
	}
	
	
	/**
	 * Equivalent to {@link #retrieveDataStructure(DataBuilderRegistry.INSTANCE_LEVEL_MODE, String)}
	 */
	public static Object retrieveDataStructure(String id)
	{		
		return retrieveDataStructure(DataBuilderRegistry.INSTANCE_LEVEL_MODE, id);
	}

	
	/**
	 * A helper method provided to retrieve the data structure for a data builder
	 * with the given mode and id.
	 * 
	 * @param mode The type of data builder to be retrieved
	 * @param id The data builder id
	 */	
	public static Object retrieveDataStructure(int mode, String id)
	{		
		DataBuilderRegistry registry = mode == DataBuilderRegistry.TOP_LEVEL_MODE ? DataBuilderRegistry.getTopLevelRegistry() : DataBuilderRegistry.getInstanceLevelRegistry();
		IDataBuilder<?> databuilder = registry.getDataStructureBuilder(id);
		if (databuilder == null)
			return null;
		
		return databuilder.getDataStructure();
	}
	
	
	/**
	 * Removes line breaks and any spaces that follow a line break
	 * 
	 * @param str The string
	 * @param replaceWithSpace If set, replaces line breaks with a space
	 * @return str with line break and white spaces that follow line breaks
	 * removed. 
	 */
	public static String removeLineBreaks(String str, boolean replaceWithSpace)
	{
		if (str == null)
			return null;
		
		StringBuffer sb = new StringBuffer();
		char currentChar, lastChar = 0;
		for (int i = 0, strLength = str.length(); i < strLength; i++)
		{
			currentChar = str.charAt(i);
				
			if (currentChar == '\r' || (lastChar == '\n' && (currentChar == '\t' || currentChar == ' ')))
			{
				continue;
			}
						
			sb.append(currentChar == '\n' ? (replaceWithSpace ? " " : "") : String.valueOf(currentChar));			
			lastChar = currentChar;
		}
		
		return sb.toString();
	}
	
	/**
	 * Equivalent SMLValidatorUtil#removeLineBreaks(String, true) 
	 */
	public static String removeLineBreaks(String str)
	{
		return removeLineBreaks(str, true);
	}
	
	@SuppressWarnings("unchecked")
	public static <T> Map<?,?> retrieveNestedMap(Map<String, Map<String, T>> map, String key, boolean create) 
	{
		if (key == null)
		{
			return null;
		}
		Map nestedMap = map.get(key);
		if (create && nestedMap == null)
		{
			nestedMap = new Hashtable<Object, Object>();
			map.put(key, nestedMap);
		}
		return nestedMap;
	}

	/**
	 * This method is used to retrieve an XML document referenced
	 * by a URI.
	 * 
	 * @param uri The uri where the document resides
	 * @return The root node of the XML document
	 */
	public static Node retrieveRemoteDocument(String uri) throws RemoteRetrievalException
	{
		Exception exception = null;
		InputStream stream = null;
		try
		{
			URL locator = new URL(uri);
			stream = locator.openStream();			
			DocumentBuilderFactory docBuilderFactory = DocumentBuilderFactory.newInstance();
	        docBuilderFactory.setNamespaceAware(true);
	        docBuilderFactory.setIgnoringElementContentWhitespace(true);
	        Document document = docBuilderFactory.newDocumentBuilder().parse(stream);
	        return document.getChildNodes() == null || document.getChildNodes().getLength() != 1 ? null : document.getChildNodes().item(0);
		} 
		catch (IOException e)
		{
			exception = e;
		} 
		catch (SAXException e)
		{
			exception = e;
		} 
		catch (ParserConfigurationException e)
		{
			exception = e;
		} 
		finally {
			if (stream != null)
				try {
					stream.close();
				} catch (IOException e) {
					// ignore
				}
		}	
		
		throw new RemoteRetrievalException(exception);
	}
	
	
	/**
	 * Thrown when there is an error while retrieving a remote 
	 * document
	 * 
	 * @author Ali Mehregani
	 */
	public static class RemoteRetrievalException extends Exception
	{
		/**
		 * The serial version UID 
		 */
		private static final long serialVersionUID = 7633895444717500045L;
	
		public RemoteRetrievalException(String message)
		{
			super(message);
		}
		
		public RemoteRetrievalException(Exception cause)
		{
			super(cause);
		}
	}
	
	
	/**
	 * Parses the document passed in 
	 * 
	 * @param resource The resource to read from
	 * @param contentHandler The content handler
	 * 
	 * @throws ParserConfigurationException
	 * @throws SAXException
	 * @throws IOException
	 */
	public static void saxParseDocument(InputStream resource, DefaultHandler contentHandler) throws ParserConfigurationException, SAXException, IOException
	{
		SAXParserFactory saxParserFactory = SAXParserFactory.newInstance();		
		saxParserFactory.setFeature(ISMLConstants.SAX_PARSER_FEATURE_NAMESPACES, true);				
		saxParserFactory.setFeature(ISMLConstants.SAX_PARSER_FEATURE_NAMESPACE_PREFIXES, true);
		saxParserFactory.newSAXParser().parse(resource, contentHandler);			
	}
	
	/**
	 * Returns {@link ISMLConstants#TYPE_DEFINITION} if the document type is determined to be definition,
	 * {@link ISMLConstants#TYPE_INSTANCE} if instance, {@link ISMLConstants#TYPE_SMLIF}
	 * if SML-IF, and {@link ISMLConstants#TYPE_UNKNOWN} if content type is unknown
	 * 
	 * @param resource The file to be checked 
	 * @return The content type of the file passed in
	 * @throws CoreException 
	 * @throws IOException 
	 */
	public static int identifyDocumentType (File resource) throws IOException, CoreException
	{
		InputStream fileInputStream = new FileInputStream(resource);		
		try
		{
			int type = ISMLConstants.TYPE_UNKNOWN;
			Object[][] describer = 
			{
					new Object[]{new SMLIFContentDescriber(), new Integer(ISMLConstants.TYPE_SMLIF)},
					new Object[]{new DefinitionContentDescriber(), new Integer(ISMLConstants.TYPE_DEFINITION)},
					new Object[]{new InstanceContentDescriber(), new Integer(ISMLConstants.TYPE_INSTANCE)},
			};
			
			BufferedReader br = new BufferedReader(new InputStreamReader(fileInputStream));
			StringBuffer sb = new StringBuffer();
			String input = null;
			while ((input = br.readLine()) != null)
			{
				sb.append(input + IValidationConstants.LINE_SEPARATOR);
			}
							
			for (int i = 0; i < describer.length; i++)
			{
				ByteArrayInputStream byteArrayInputStream = new ByteArrayInputStream(sb.toString().getBytes());
				if (((IContentDescriber)describer[i][0]).describe(byteArrayInputStream, null) == IContentDescriber.VALID)
				{
					return ((Integer)describer[i][1]).intValue();
				}
			}					
			
			return type;
		}
		finally
		{
			if (fileInputStream != null)
				fileInputStream.close();
		}
	}
	
	public static int identifyDocumentType (IFile file) throws IOException, CoreException
	{
		int type = ISMLConstants.TYPE_UNKNOWN;
		InputStream fileInputStream = null;
		fileInputStream = file.exists() ? file.getContents() : new FileInputStream(file.getFullPath().toOSString());	
		IContentType[] types = Platform.getContentTypeManager().findContentTypesFor(fileInputStream, file.getName());
		
		for (int i = 0; i < types.length; i++)
		{
			if (SMLIFContentDescriber.ID.equals(types[i].getId()))
			{
				return ISMLConstants.TYPE_SMLIF;
			}
			else if (DefinitionContentDescriber.ID.equals(types[i].getId()))
			{
				return ISMLConstants.TYPE_DEFINITION;
			}
			else if (InstanceContentDescriber.ID.equals(types[i].getId()))
			{
				type = ISMLConstants.TYPE_INSTANCE;
			}
		}			
		return type;
	}
	
	public static int identifyDocumentType (ResourceWrapper resourceWrapper) throws IOException, CoreException
	{
		return resourceWrapper.getResource() instanceof File ?
				identifyDocumentType((File)resourceWrapper.getResource()) :
				identifyDocumentType((IFile)resourceWrapper.getResource());
	}
	
	/**
     * This method attempts to delete the specified root file.  If root is a directory
     * with children, this method attempts to first recursively delete all children 
     * of root.  If this method returns true, then root has been deleted.  If this
     * method returns false, root has not been deleted; however, if root is a 
     * directory, it is possible that some of its children were successfully 
     * deleted.
     *  
     * @param root
     * @return true if root has been deleted, false otherwise.
     */
    public static boolean deleteFileTree(File root)
    {
    	if ( root.exists() )
    	{
    		if ( root.isFile() )
    		{
    			return root.delete();
    		}
    		else
    		{
    			File[] children = root.listFiles();
    			if ( children != null )
    			{
	    			for (int i=0; i<children.length; i++)
	    			{
	    				deleteFileTree(children[i]);
	    			}
    			}
    			return root.delete();
    		}
    	}
    	return true;
    }
    
    
	/**
	 * Retrieves and returns the child elements of 
	 * node (i.e. nodes of type Node.ELEMENT_NODE)
	 * 
	 * @param node The node
	 * @return Returns the child elements of node
	 */
	public static Node[] retrieveChildElements(Node node)
	{
		if (node == null)
			return new Node[0];
		
		List<Node> childElements = new ArrayList<Node>();
		NodeList children = node.getChildNodes();
		for (int i = 0, childCount = children.getLength(); i < childCount; i++)
		{
			Node child = children.item(i);
			if (child.getNodeType() == Node.ELEMENT_NODE)
				childElements.add(child);			
		}
		return (Node[])childElements.toArray(new Node[childElements.size()]);
	}
	


	/**
	 * Return a string that forms a tag name for the namespace
	 * and elementName in namespace:elementName format.
	 * 
	 * @param namespace
	 * @param elementName
	 * @return
	 */
	public static String tagNameFor(String namespace, String elementName) {
		StringBuffer buffer = new StringBuffer("");
		if (namespace != null) {
			buffer.append(namespace).append(ISMLConstants.COLON);
		}
		buffer.append(elementName);
		return buffer.toString();
	}


	/**
	 * Return an XML begin tag for the elementName.  For example,
	 * with an argument of "foo", "<foo>" will be returned.
	 * 
	 * @param elementName
	 * @return
	 */
	public static String beginTagFor(String elementName) {
		return tagFor(false, null, elementName);
	}


	/**
	 * Return an XML begin tag for the namespace and element name.
	 * For example, with a namespace of "foo" and an element name
	 * of "bar", "<foo:bar>" will be returned.
	 * 
	 * @param namespace
	 * @param elementName
	 * @return
	 */
	public static String beginTagFor(String namespace, String elementName) {
		return tagFor(false, namespace, elementName);
	}


	/**
	 * Return an XML end tag for the elementName.  For example,
	 * with an argument of "foo", "</foo>" will be returned.
	 * 
	 * @param elementName
	 * @return
	 */
	public static String endTagFor(String elementName) {
		return tagFor(true, null, elementName);
	}


	/**
	 * Return an XML end tag for the namespace and element name.
	 * For example, with a namespace of "foo" and an element name
	 * of "bar", "</foo:bar>" will be returned.
	 * 
	 * @param namespace
	 * @param elementName
	 * @return
	 */
	public static String endTagFor(String namespace, String elementName) {
		return tagFor(true, namespace, elementName);
	}


	/**
	 * Utility method to return a begin or end tag for the namespace
	 * and elementName input.
	 * 
	 * @param end
	 * @param namespace
	 * @param elementName
	 * @return
	 */
	public static String tagFor(boolean end, String namespace, String elementName) {
		StringBuffer tagBuffer = new StringBuffer();
		tagBuffer.append(ISMLConstants.OPEN_ANGLE_BRACKET);
		if (end) {
			tagBuffer.append(ISMLConstants.FORWARD_SLASH);
		}
		tagBuffer.append(tagNameFor(namespace, elementName));
		tagBuffer.append(ISMLConstants.CLOSE_ANGLE_BRACKET);
		return tagBuffer.toString();
	}


	/**
	 * Return a full XML element with begin and end tags, and data in
	 * between.  With an elementName of "foo" and elementData of "bar",
	 * the String "<foo>bar</foo>" will be returned.
	 * 
	 * @param elementName
	 * @param elementData
	 * @return
	 */
	public static String createElementTag(String elementName, String elementData) {
		return beginTagFor(elementName) + elementData + endTagFor(elementName);
	}


	/**
	 * Return a full XML element with begin and end tags, and data in
	 * between.  With a namespace of "sml", an elementName of "foo", and elementData of "bar",
	 * the String "<sml:foo>bar</sml:foo>" will be returned.
	 * 
	 * @param namespace
	 * @param elementName
	 * @param elementData
	 * @return
	 */
	public static String createElementTag(String namespace, String elementName, String elementData) {
		return beginTagFor(namespace, elementName) + elementData + endTagFor(namespace, elementName);
	}


	public static String getExceptionStackTrace(Throwable t)
	{
		if (t == null) return "";
		
		Throwable throwable = findCause(t);
				
		StringWriter stackTrace = new StringWriter(); 
		throwable.printStackTrace(new PrintWriter(stackTrace));	
		
		return stackTrace.toString();
	}
	
	protected static Throwable findCause (Throwable t)
	{
		Throwable cause = null;
		if (t instanceof InvocationTargetException)
			cause = ((InvocationTargetException)t).getTargetException();			
		else if (t instanceof CoreException)
			cause = ((CoreException)t).getStatus().getException();				
		
		if (cause == null)
			return t;
		return findCause(cause);
	}


	/**
	 * Replace all occurrences of oldString in originalString with newString.
	 * 
	 * Example:	<xmp>stringReplace("sofa", "of", "ant")</xmp> evaluates to: <xmp>"santa"</xmp>
	 * 
	 * This offers comparable functionality to String.replace(String, String) found
	 * in Java 1.5.
	 *
	 * @param originalString original source string
	 * @param oldSubstring substring to be replaced
	 * @param newSubstring new text to be substituted for oldSubstring
	 * @throws NullPointerException if any argument is null
	 * @return new string with substitutions made
	 */
	public static String stringReplace(String originalString, String oldSubstring, String newSubstring) {
		StringBuffer result = new StringBuffer();
		int index = 0;
		if (newSubstring == null) {
			throw new NullPointerException();
		}
		while (index < originalString.length()) {
			int prevIndex = index;
			if ((index + oldSubstring.length() <= (originalString.length())) && ((oldSubstring.length() == 0) || (originalString.substring(index, index + oldSubstring.length()).equals(oldSubstring)))) {
				result.append(newSubstring);
				index = index + oldSubstring.length();
			}
			if ((prevIndex == index) || "".equals(oldSubstring)) {
				result.append(originalString.charAt(index));
				index++;
			}
		}
		return result.toString();
	}
	
	
	/**
	 * Answer a new SAXParser instance with the specified attributes.
	 */
	public static SAXParser createSAXParser(Object schemaSource, boolean validating, LexicalHandler lexicalHandler, IValidationOutput<?, ?> validationLogger) throws ParserConfigurationException, SAXException, SAXNotRecognizedException, SAXNotSupportedException 
	{
	    SAXParserFactory saxParserFactory = SMLValidatorUtil.saxParserFactory;
	    
	    if (saxParserFactory == null)
	    {
	    	saxParserFactory = SAXParserFactory.newInstance();
	    	saxParserFactory.setNamespaceAware(true);
	    	saxParserFactory.setValidating(validating);
	    	saxParserFactory.setFeature(ISMLConstants.SAX_PARSER_FEATURE_NAMESPACES, true);
			saxParserFactory.setFeature(ISMLConstants.SAX_PARSER_FEATURE_NAMESPACE_PREFIXES, true);	
			saxParserFactory.setFeature(ISMLConstants.SAX_PARSER_FEATURE_ANNOTATIONS, true);
			SMLValidatorUtil.saxParserFactory = saxParserFactory;
	    }
		    
		saxParserFactory.setFeature(ISMLConstants.SAX_PARSER_FEATURE_VALIDATION, validating);
		SAXParser newSaxParser = saxParserFactory.newSAXParser();
		
		if (schemaSource == null) {
			return newSaxParser;
		}
	
		try
		{					
			newSaxParser.setProperty(ISMLConstants.SAX_PARSER_PROPERTY_SCHEMA_LANGUAGE, ISMLConstants.SCHEMA_URI); 
			newSaxParser.setProperty(ISMLConstants.SAX_PARSER_PROPERTY_SCHEMA_SOURCE, schemaSource);					
			if (lexicalHandler != null) {
				newSaxParser.setProperty(ISMLConstants.SAX_PARSER_PROPERTY_LEXICAL_HANDLER, lexicalHandler);
			}
		}
		catch (Exception e)
		{
			// Some parsers cannot validate the xsd, such as the standard one with JDK 1.4. Therefore turn
			// off validation.
			validationLogger.reportMessage(ValidationMessageFactory.createWarningMessage(ValidationMessage.NO_LINE_NUMBER, SMLValidationMessages.errorUnsupportedSAXProperty0));						
			saxParserFactory.setFeature(ISMLConstants.SAX_PARSER_FEATURE_VALIDATION, false);
			newSaxParser = saxParserFactory.newSAXParser();
		}
		return newSaxParser;
	}


	public static String retrieveAbsolutePath(String path)
	{
		IPath resourcePath = new Path(path);
		if (resourcePath.segmentCount() == 1)
			return ResourcesPlugin.getWorkspace().getRoot().getProject(resourcePath.segment(0)).getLocation().toOSString();
		
		IFile file = ResourcesPlugin.getWorkspace().getRoot().getFile(new Path(path));
		if ((file != null) && (checkWorkspaceResourceName(path))) 
		{
			return file.getLocation().toOSString();
		}
		return path;
	}


	/**
	 * Answer whether the destination directory entered is a valid
	 * project or folder name.  The directory does not need to
	 * exist.
	 * @param dirName  
	 * @return
	 */
	public static boolean checkWorkspaceDirName(String dirName) {
		boolean okForFolder = ResourcesPlugin.getWorkspace().validatePath(
				dirName, IResource.FOLDER).isOK();
		if (!okForFolder) {
			okForFolder = ResourcesPlugin.getWorkspace().validatePath(
					dirName, IResource.PROJECT).isOK();
		}
		return okForFolder;
	}


	public static boolean checkWorkspaceResourceName(String resourceName) {
		if (!resourceName.startsWith("/")) {
			// handle full workspace paths not preceded by slash
			resourceName = '/' + resourceName;
		}
		return ResourcesPlugin.getWorkspace().validatePath(
				resourceName, IResource.FILE).isOK();
	}


	/**
	 * Returns true if the node passed in has the attribute "sml:ref" with
	 * the value "true" or "1", where "sml" is bound to the SML namespace.
	 * 
	 * @param node The node to be checked
	 * @return true if node is a reference; false otherwise
	 */
	public static boolean isReference(Node node)
	{
		NamedNodeMap attributes = node.getAttributes();
		Node refAttribute = attributes.getNamedItemNS(ISMLConstants.SML_URI, ISMLConstants.REF_ATTRIBUTE);
    	return isTrue(refAttribute == null ? null : refAttribute.getNodeValue());    	
	}
	
	
	/**
	 * Returns true if the attributes contains the attribute: "sml:acyclic" = "true" or "1", 
	 * where "sml" is bound to the SML namespace.
	 * 
	 * @param attributes The attributes to be checked
	 * @return true if attributes contains sml:ref="true"/"1"
	 */

	public static boolean isAcyclic(Attributes attributes)
	{
		return isTrue(attributes.getValue(ISMLConstants.SML_URI, ISMLConstants.ACYCLIC_ATTRIBUTE));
	}


	/**
	 * Returns true if the trimmed version of the value passed in is 
	 * either "true" or "1".
	 * 
	 * @param value The value to be analyzed
	 * @return true if value.trim() is "true" or "1"; false otherwise
	 */
	public static boolean isTrue(String value)
	{
		if (value == null)
    	{
    		return false;
    	}
    	
    	value = value.trim();
    	return IValidationConstants.TRUE_VALUE.equals(value) || IValidationConstants.ONE_VALUE.equals(value);
	}
	
	
	/**
	 * Returns true if the trimmed version of the value passed in is 
	 * either "false" or "0".
	 * 
	 * @param value The value to be analyzed
	 * @return true if value.trim() is "false" or "0"; false otherwise
	 */
	public static boolean isFalse(String value)
	{
		if (value == null)
    	{
    		return false;
    	}
    	
    	value = value.trim();
    	return IValidationConstants.FALSE_VALUE.equals(value) || IValidationConstants.ZERO_VALUE.equals(value);
	}
	
	
	/**
	 * Extracts the text node representing the SML document reference
	 * from the root node passed in.
	 *  
	 * @param root The root node
	 * @return A string representing the SML document reference
	 */
	public static String extractReference(Node root)
	{
		NodeIterator nodeIterator = new NodeIterator(root, ISMLConstants.SML_URI, ISMLConstants.URI_ELEMENT);
		if (!nodeIterator.hasNext())
			return "";
		
		Node uriElement = (Node)nodeIterator.next();
		NodeList children = uriElement.getChildNodes();
		StringBuffer reference = new StringBuffer();
		for (int i = 0, childCount = children.getLength(); i < childCount; i++)
		{
			if (children.item(i).getNodeType() == Node.TEXT_NODE)
			{
				String value = children.item(i).getNodeValue();
				value = value.trim();
				value = value.replaceAll("\n", "");
				reference.append(value);
			}
		}
		
		return reference.toString();
	}

	/**
	 * Given a target namespace and a qualified value, the following method
	 * returns a QName with all appropriate fields set
	 * 
	 * @param targetNamespace namespace URI
	 * @param value A value 
	 * @return A qualified name object of value
	 */
	public static QName toQName(String targetNamespace, String value)
	{
		if (value == null)
		{
			return null;
		}
		
		String[] tokenizedValue = value.split(":");		
		if (tokenizedValue.length >= 2)
		{
			return new QName(targetNamespace, tokenizedValue[1], tokenizedValue[0]);
		}
		return new QName(targetNamespace, tokenizedValue[0]);
	}
	
	/**
	 * Given a (prefix, uri) map and a qualified value, the following method
	 * returns a QName with all appropriate fields set
	 * 
	 * @param prefixUriMap A (prefix, uri) map
	 * @param value A value 
	 * @return A qualified name object of value
	 */
	public static QName toQName(Map<String,String> prefixUriMap, String targetNamespace, String value)
	{
		if (value == null)
		{
			return null;
		}
		
		String[] tokenizedValue = value.split(":");		
		if (tokenizedValue.length >= 2)
		{
			return new QName(prefixUriMap.get(tokenizedValue[0]), tokenizedValue[1], tokenizedValue[0]);
		}
		return new QName(targetNamespace, tokenizedValue[0]);
	}
	
	/**
	 * 
	 * @param namespace
	 * @return
	 */
	public static boolean isBuiltInNamespace(String namespace) {
		
		if(namespace != null && namespace.equals(ISMLConstants.SML_URI)
				|| namespace.equals(ISMLConstants.SCHEMATRON_URI) || namespace.equals(ISMLConstants.SMLIF_URI)
				|| namespace.equals(ISMLConstants.SCHEMA_URI) || namespace.equals(ISMLConstants.SML_FN_URI) 
				|| namespace.equals(ISMLConstants.XLINK_URI) || namespace.equals(ISMLConstants.SCHEMA_INSTANCE_URI) 
				|| namespace.equals(ISMLConstants.XSL_URI))
		return true;
		
		return false;
	}


	/**
	 * Returns true if type is 'anyType'
	 * 
	 * @param type The type to be checked
	 * @return true if type is 'anyType'; false otherwise
	 */
	public static boolean isAnyType(XSTypeDefinition type)
	{
		return type == null ? false : ISMLConstants.SCHEMA_URI.equals(type.getNamespace()) && ISMLConstants.ANY_TYPE_ELEMENT.equals(type.getName()); 		
	}
	
	
	/**
	 * Converts a set of definition schema model nodes
	 * into input streams
	 * 
	 * @param definitions The definitions to convert to inputstream
	 * @return Inputstream representation of the definition documents
	 */
	public static InputStream[] toInputStream(ElementSchemaModel[] definitions)
	{
		List<InputStream> inputStreamList = new ArrayList<InputStream>();
		for (int i = 0; i < definitions.length; i++)
		{
			inputStreamList.add(new ByteArrayInputStream(definitions[i].getSource().getBytes()));
		}
		
		return inputStreamList.toArray(new InputStream[inputStreamList.size()]);
	}
	
	
	public static AnnotationResult retrieveAnnotation (PSVIProvider psvi, String uri, String localName, boolean checkHierarchy)
	{
		ElementPSVI elementPSVI = psvi.getElementPSVI();
		XSTypeDefinition typeDefinition = elementPSVI == null ? null : elementPSVI.getTypeDefinition();
		return retrieveAnnotation(typeDefinition, uri, localName, checkHierarchy);
	}
	
	
	/**
	 * Convenience method - Equivalent to retrieveAnnotation(typeDefinition, uri, String[]{localName}, checkHierarchy)
	 * 
	 * Note that localNames represents child *elements* unlike the similarly named
	 * {@link #retrieveAnnotation(XSTypeDefinition, String, String, boolean)} which
	 * expects the localName to represent a child attribute.
	 */
	public static AnnotationResult retrieveAnnotation (XSTypeDefinition typeDefinition, String uri, String[] localNames, boolean checkHierarchy)
	{
		XSObjectList xsObjectList = retrieveAnnotations(typeDefinition);
		if (xsObjectList == null)
		{
			return null;
		}
		
		AnnotationResult annotationResult = null;
		Document document = createDocument();
		for (int i = 0, objectListCount = xsObjectList.getLength(); i < objectListCount; i++)
		{	
			XSAnnotation annotation = (XSAnnotation)xsObjectList.item(i);
			annotation.writeAnnotation(document, XSAnnotation.W3C_DOM_DOCUMENT);
			Node[] nodes = findNodes(document.getFirstChild(), uri, localNames);
			if (nodes != null && nodes.length > 0)
			{
				annotationResult = new AnnotationResult();
				annotationResult.setType(typeDefinition);
				for (int j = 0; j < nodes.length; j++)
				{
					annotationResult.addNode(nodes[j]);
				}
				break;
			}
		}
		
		return annotationResult;
	}
	
	
	/**
	 * Given a type CT, determine if CT [or its super types] contains
	 * an attribute with the namespace N and local name L.
	 * 
	 * @param typeDefinition The type CT
	 * @param uri The namespace N
	 * @param localNames The local name L
	 * @param checkHierarchy If true, the super types of CT are checked;
	 * otherwise the hierarchy of CT is not checked.
	 * 
	 * @return The annotation result or null if none is found
	 */
	public static AnnotationResult retrieveAnnotation (XSTypeDefinition typeDefinition, String uri, String localName, boolean checkHierarchy)
	{				
		XSObjectList xsObjectList = retrieveAnnotations(typeDefinition);
		int objectListCount;
		if (xsObjectList == null)
		{
			return null;
		}
		
		if ((objectListCount = xsObjectList.getLength()) <= 0)
		{
			return checkHierarchy ? retrieveAnnotation(typeDefinition.getBaseType(), uri, localName, checkHierarchy) : null;
		}
	
		Document document = createDocument();
		AnnotationResult annotationResult = null;
		
		for (int i = 0; i < objectListCount; i++)
		{	
			annotationResult = handleAnnotation(document, typeDefinition, (XSAnnotation)xsObjectList.item(i), uri, localName);		
			annotationResult = annotationResult == null && checkHierarchy ? retrieveAnnotation(typeDefinition.getBaseType(), uri, localName, checkHierarchy) : annotationResult;
			if (annotationResult != null)
			{
				return annotationResult;
			}
		}
		
		return null;
	}
	
	
	private static XSObjectList retrieveAnnotations(XSTypeDefinition typeDefinition)
	{
		if (typeDefinition == null || 
				SMLValidatorUtil.isAnyType(typeDefinition) || 
				typeDefinition.getTypeCategory() != XSTypeDefinition.COMPLEX_TYPE)
		{
			return null;
		}
			
		return ((XSComplexTypeDefinition)typeDefinition).getAnnotations();
	}
	
	
	/**
	 * Given element declaration E, determine if E contains
	 * an attribute with the namespace N and local name L.
	 * 
	 * @param elementDeclaration The element declaration E
	 * @param uri The namespace N
	 * @param localName The local name L
	 * 
	 * @return The annotation result or null if none is found
	 */
	public static AnnotationResult retrieveAnnotation(XSElementDeclaration elementDeclaration, String uri, String localName) 
	{
		if (elementDeclaration == null) 
		{
			return null;
		}
		
		XSAnnotation annotation = elementDeclaration.getAnnotation();
		Document domDocument = createDocument();		
		return handleAnnotation (domDocument, elementDeclaration.getTypeDefinition(), annotation, uri, localName);
	}
	
	
	/**
	 * Given element declaration E, namespace N, and a set of local names L, determine
	 * if the annotation of E contains any nodes matching the namespace N and any
	 * of the local names L
	 * 
	 * @param elementDeclaration The element declaration E
	 * @param uri The namespace N
	 * @param localNames The local names L
	 * @return A set of nodes included in the annotation of E, matching
	 * the namespace N and any of the local names L.
	 */
	public static Node[] retrieveAnnotation(XSElementDeclaration elementDeclaration, String uri, String[] localNames)
	{
		XSAnnotation annotation = null;
		if (elementDeclaration == null || (annotation = elementDeclaration.getAnnotation()) == null)
		{
			return new Node[0];
		}
				
		Document document = createDocument();		
		annotation.writeAnnotation(document, XSAnnotation.W3C_DOM_DOCUMENT);
		return findNodes(document.getFirstChild(), uri, localNames);
	}
	
	
	public static Document createDocument()
	{
		try
		{
			DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();       
			return factory.newDocumentBuilder().newDocument();
		} 
		catch (ParserConfigurationException e)
		{				
			e.printStackTrace();
		}
		
		return null;
	}
	
	/**
	 * Given node N, namespace NS, and local names L, find all nodes
	 * starting with N that match NS and L
	 *  
	 * @param node The starting node N
	 * @param uri The namespace NS
	 * @param localNames The local names L
	 * @return All nodes starting from N that match NS and L
	 */
	public static Node[] findNodes(Node node, String uri, String[] localNames)
	{
		List<Node> nodes = new ArrayList<Node>();
		findNodes (nodes, node, uri, localNames);
		return nodes.toArray(new Node[nodes.size()]);		
	}
	
	
	/**
	 * Convenience method - equivalent to findNodes(node, uri, new String[]{localNames)
	 */
	public static Node[] findNodes(Node node, String uri, String localNames)
	{
		return findNodes(node, uri, new String[]{localNames});
	}
	
	
	private static void findNodes(List<Node> nodes, Node node, String uri, String[] localIds)
	{
		if (node == null || node.getNodeType() != Node.ELEMENT_NODE)
		{
			return;
		}
		
		if (uri.equals(node.getNamespaceURI()))
		{
			for (int i = 0; i < localIds.length; i++)
			{
				if (localIds[i].equals(node.getLocalName()))
				{
					nodes.add(node);
					return;
				}
			}
		}
		
		NodeList children = node.getChildNodes();
		for (int i = 0, childCount = children.getLength(); i < childCount; i++)
		{
			findNodes(nodes, children.item(i), uri, localIds);
		}
	}
	
	
	private static AnnotationResult handleAnnotation (Document document, XSTypeDefinition type, XSAnnotation annotation, String uri, String localName)
	{
		return handleAnnotation(document, type, annotation, uri, new String[] {localName});
	}
	
	
	private static AnnotationResult handleAnnotation (Document document, XSTypeDefinition type, XSAnnotation annotation, String uri, String[] localNames)
	{
		if (annotation == null) 
		{
			return null;
		}
		annotation.writeAnnotation(document, XSAnnotation.W3C_DOM_DOCUMENT);		
		Node firstChild = document.getFirstChild();
		NamedNodeMap attributes = firstChild == null ? null : firstChild.getAttributes();		
		if (attributes == null)
		{
			return null;
		}
		
		
		for (int i = 0; i < localNames.length; i++)
		{
			Node attribute = attributes.getNamedItemNS(uri, localNames[i]);			
			if (attribute != null)
			{
				AnnotationResult annotationResult = new AnnotationResult();
				annotationResult.addNode(attribute);
				annotationResult.setType(type);
				return annotationResult;
			}
		}
		
		
		return null;
	}
	
	
	/**
	 * A helper method used to tokenize a qualified name into its
	 * prefix and local name.  A string array of size 2 is returned, where
	 * String[0] = prefix or empty string if no prefix is found and 
	 * String[1] = local name
	 * 
	 * @param qName The qualified name of an element/attribute
	 * @return The prefix and local name tokens of the qualified name
	 */
	public static String[] tokenizeName(String qName)
	{
		String[] tokenizedQName = qName.split(":");				
		if (tokenizedQName.length <= 1)
		{
			return new String[]{null, tokenizedQName[0]};
		}
		return new String[]{tokenizedQName[0], tokenizedQName[1]};
	}
}
