/**********************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.common;

import org.eclipse.core.resources.IResource;
import org.eclipse.osgi.util.NLS;

/**
 * An abstract class used to format and output a verification message 
 * to an output stream defined by a subclass
 * 
 * @author Ali Mehregani
 */
public abstract class TextOutput extends AbstractValidationOutput
{
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.common.AbstractValidationOutput#reportMessage(org.eclipse.cosmos.rm.internal.validation.common.AbstractValidationOutput.ValidationMessage)
	 */
	public void writeMessageToSource(IValidationMessage validationMessage)
	{
		Object resource = validationMessage.getAttribute(ValidationMessage.ATTRIBUTE_RESOURCE);
		int lineNumber = validationMessage.getAttribute(ValidationMessage.ATTRIBUTE_LINE_NUMBER, ValidationMessage.NO_LINE_NUMBER);
		int severity = validationMessage.getAttribute(ValidationMessage.ATTRIBUTE_SEVERITY, ValidationMessage.SEVERITY_INFO);
		String validationMsg = validationMessage.getAttribute(ValidationMessage.ATTRIBUTE_MESSAGE, IValidationConstants.EMPTY_STRING);
		
		String message = formatMessage(resource instanceof IResource ? (IResource)resource : null, lineNumber, 
				severity == ValidationMessage.SEVERITY_ERROR ? SMLValidationMessages.outputSeverityError : 
				severity == ValidationMessage.SEVERITY_WARNING ? SMLValidationMessages.outputSeverityWarning : SMLValidationMessages.outputSeverityInfo, validationMsg);
		
		writeMessage(message);	
	}
	
	
	/**
	 * Formats the attributes of a verification message into a message that can
	 * be logged to an output stream
	 * 
	 * @param resource The resource that is associated with the verification message; null if none is available
	 * @param lineNumber The line number associated with the message; -1 if none is available
	 * @param severity The string representation of the message's severity
	 * @param validationMsg The message
	 * @return A formatted message that can be logged to an output stream
	 */
	public String formatMessage(IResource resource, int lineNumber, String severity, String validationMsg)
	{
		StringBuffer formattedMessage = new StringBuffer("");
		if (resource != null)
		{
			formattedMessage.append(resource.getFullPath().toString()).append(" ");			
		}
		if (lineNumber > 0)
		{
			formattedMessage.append(NLS.bind(SMLValidationMessages.outputLineNumber, String.valueOf(lineNumber))).append(" ");
		}
		formattedMessage.append(NLS.bind(SMLValidationMessages.outputSeverity, severity)).append(" ").append(validationMsg);
		
		return formattedMessage.toString();
	}
	
	
	/**
	 * Concrete classes are expected to write the passed in message to 
	 * an output stream.
	 * 
	 * @param message The message to be written
	 */
	protected abstract void writeMessage(String message);
}
