/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.databuilders;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.xml.namespace.NamespaceContext;

import org.apache.xerces.xs.ElementPSVI;
import org.apache.xerces.xs.PSVIProvider;
import org.apache.xerces.xs.XSElementDeclaration;
import org.eclipse.cosmos.rm.internal.validation.SMLActivator;
import org.eclipse.cosmos.rm.internal.validation.artifacts.IdentityConstraintStructure;
import org.eclipse.cosmos.rm.internal.validation.artifacts.IdentityConstraintStructure.IdentityConstraint;
import org.eclipse.cosmos.rm.internal.validation.artifacts.IdentityConstraintStructure.IdentityInstance;
import org.eclipse.cosmos.rm.internal.validation.common.ISMLConstants;
import org.eclipse.cosmos.rm.internal.validation.core.IFoundationBuilder;
import org.eclipse.cosmos.rm.internal.validation.core.IValidator;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;

/**
 * This builder will construct a data structure that will be used
 * as part of the identity constraint (key, keyref, and unique) 
 * validation.  The data structure returned by this builder is of type
 * {@link IdentityConstraintStructure}
 * 
 * @author Ali Mehregani
 */
public class IdentityConstraintDataBuilder extends AbstractDataBuilder<IdentityConstraintStructure>
{
	/**
	 * The ID of this builder
	 */
	public static final String ID = SMLActivator.PLUGIN_ID + ".IdentityConstraintDataBuilder";
	
	/**
	 * The identity constraints
	 */
	public static final String[] IDENTITY_CONSTRAINTS =
		new String[]
		{
			ISMLConstants.KEY_ELEMENT,
			ISMLConstants.KEY_REF_ELEMENT,
			ISMLConstants.UNIQUE_ELEMENT
		};
	
	/**
	 * Constants representing the begin,
	 * startElement, and closeElement event
	 */
	private static int START = 0x00;
	private static int OPEN = 0x01;
	private static int CLOSE = 0x02;
	
	/**
	 * The identity constraint structure
	 */
	private IdentityConstraintStructure identityStruct;
	
	/**
	 * The current index of the document node
	 */
	private List<Integer> documentNodeInx;
		
	/**
	 * The current orphan document index
	 */
	private int orphanInx;

	/**
	 * The namespace context associated with this builder.
	 * This is retrieved from the schemas bound to the instance
	 * document this builder will run against.
	 */
	private NamespaceContext namespaceContext;
	
	/**
	 * Stores the last event detected
	 */
	private int lastEvent;
	
	
	/**
	 * Constructor
	 */
	public IdentityConstraintDataBuilder()
	{
		identityStruct = new IdentityConstraintStructure();
		documentNodeInx = new ArrayList<Integer>();
		orphanInx = -1;	
		addEvent(IFoundationBuilder.EVENT_CHARACTER);
	}
	
	
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.AbstractDataBuilder#initialize(java.util.Map)
	 */
	public void initialize(Map<String, Object> init)
	{
		super.initialize(init);
		String[] aliases = getAliases();
		documentNodeInx.clear();
		orphanInx += (aliases == null || aliases.length <= 0) ? 1 : 0;
		namespaceContext = (NamespaceContext)init.get(IValidator.ATTRIBUTE_NAMESPACE_CONTEXT);
		lastEvent = START;
	}
	
	
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.AbstractDataBuilder#startElement(java.lang.String, java.lang.String, java.lang.String, org.xml.sax.Attributes)
	 */
	public void startElement(String uri, String localName, String qName, Attributes attributes) throws SAXException
	{		
		super.startElement(uri, localName, qName, attributes);

		if (lastEvent == OPEN || lastEvent == START)
		{
			documentNodeInx.add(0);
		}
		else if (lastEvent == CLOSE)
		{
			int lastInx = documentNodeInx.size() - 1;
			documentNodeInx.set(lastInx, documentNodeInx.get(lastInx) + 1);
		}
		
		
		IdentityConstraint[] constraints = retrieveConstraints();
		
		// The indices leading to the context node
		int[] nodePath = new int[documentNodeInx.size()]; 
		for (int j = 0; j < nodePath.length; j++)
		{
			nodePath[j] = ((Integer)documentNodeInx.get(j)).intValue();
		}
		
		// The aliases of the current document
		String[] aliases = getAliases();
		
		// For each constraint, add an entry to the structure
		for (int i = 0; i < constraints.length; i++)
		{			
			IdentityInstance identityInstance = new IdentityInstance();
			identityInstance.setConstraint(constraints[i]);
			identityInstance.setNodePath(nodePath);
			identityInstance.setLineNumber(getBaseLineNumber() + getLocator().getLineNumber() - 1);
			
			if (aliases == null || aliases.length <= 0)
			{			
				identityStruct.addOrphanedConstrainedInstance(orphanInx, identityInstance);
			}
			else
			{
				identityStruct.addConstrainedInstance(aliases[0], identityInstance);				
			}
		}
		
		lastEvent = OPEN;
	}
	
	
	/**
	 * Returns a list of identity constraints associated with the
	 * current element.  Returns null if none is found
	 * 
	 * @return Identity constraints of the current element
	 */
	private IdentityConstraint[] retrieveConstraints()
	{		
		PSVIProvider psviProvider = getPsvi();
		ElementPSVI elementPSVI = psviProvider.getElementPSVI();
		XSElementDeclaration elementDeclaration = elementPSVI == null ? null : elementPSVI.getElementDeclaration();
		
		List<IdentityConstraint> identityConstraints = IdentityConstraintUtil.retrieveConstraints (elementDeclaration, namespaceContext);		
		return identityConstraints.toArray(new IdentityConstraint[identityConstraints.size()]);
	}

	
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.AbstractDataBuilder#endElement(java.lang.String, java.lang.String, java.lang.String)
	 */
	public void endElement(String uri, String localName, String qName) throws SAXException
	{
		super.endElement(uri, localName, qName);

		if (lastEvent == CLOSE)
		{
			documentNodeInx.remove(documentNodeInx.size() - 1);
		}	
		
		lastEvent = CLOSE;
	}
	
	
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.IDataBuilder#getDataStructure()
	 */
	public IdentityConstraintStructure getDataStructure()
	{
		return identityStruct;
	}
}
