/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.rm.internal.validation.databuilders;

import java.util.HashMap;
import java.util.Map;


import org.apache.xerces.xs.ElementPSVI;
import org.apache.xerces.xs.XSComplexTypeDefinition;
import org.apache.xerces.xs.XSElementDeclaration;
import org.apache.xerces.xs.XSTypeDefinition;
import org.eclipse.cosmos.rm.internal.validation.SMLActivator;
import org.eclipse.cosmos.rm.internal.validation.artifacts.ConstraintNode;
import org.eclipse.cosmos.rm.internal.validation.artifacts.ElementTypeMap;
import org.eclipse.cosmos.rm.internal.validation.artifacts.TargetElementNode;
import org.eclipse.cosmos.rm.internal.validation.common.ISMLConstants;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidatorUtil;
import org.eclipse.cosmos.rm.internal.validation.common.AbstractValidationOutput.ValidationMessage;
import org.eclipse.cosmos.rm.internal.validation.core.IFoundationBuilder;
import org.eclipse.cosmos.rm.internal.validation.util.ParserHelper;
import org.w3c.dom.DOMException;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;

/**
 * Builds data structure representing target element nodes.
 * 
 * @author Ali Mehregani
 * @author David Whiteman
 */
public class TargetInstanceBuilder extends AbstractDataBuilder<Map<TargetElementNode, String>>
{

	public static final String ID = SMLActivator.PLUGIN_ID + ".TargetInstanceBuilder";
    protected TargetElementNode currentTargetElement; 
    protected ElementTypeMap elementTypeMap;
    protected boolean uriElementFound = false; 	
    protected Map<TargetElementNode, String> dataStructure;
    protected Map<String, ConstraintNode> targetSchemaBuilderStructure;
    protected Map<String, Map<String, String>> substitutionElementStructure;
	
	public TargetInstanceBuilder()
	{		
		dataStructure = new HashMap<TargetElementNode, String>();
		targetSchemaBuilderStructure = new HashMap<String, ConstraintNode>();
		substitutionElementStructure = new HashMap<String, Map<String, String>> ();
		elementTypeMap = new ElementTypeMap();
		
		super.addEvent(IFoundationBuilder.EVENT_CHARACTER);
	}

	public Map<TargetElementNode, String> getDataStructure() {
		return dataStructure;
	}
	
	public void startInstances() {
	}
	

	public void startElement(String uri, String localName, String qName, Attributes attributes) throws SAXException 
	{
		updateElementTypeMap();

		if (uri == null) {
			return;
		}
		
		ConstraintNode constraintNode = getConstraintNode(localName);

		if (constraintNode == null) {
			constraintNode = getSubstitutionGroupNode(uri, localName);
		}
		//check if the element is defined as having a target element attribute and the ref:targetRequired is true and 
		if (constraintNode != null){
			currentTargetElement = new TargetElementNode(localName, getFilePath(), getAdjustedLocatorLine(), getLocator() == null ? ValidationMessage.NO_LINE_NUMBER : getLocator().getColumnNumber());
			currentTargetElement.setUri(uri);
		} else if ((currentTargetElement != null) && (ISMLConstants.URI_ELEMENT.equals(localName))){
			uriElementFound = true;
		} else {
			uriElementFound = false;
		}
	}

	protected void updateElementTypeMap() {
		XSElementDeclaration elementDeclaration = getPsvi().getElementPSVI().getElementDeclaration();
		if (elementDeclaration != null) {
			String name = elementDeclaration.getName();
			XSTypeDefinition type = getPsvi().getElementPSVI().getTypeDefinition();
			if (type.getTypeCategory() == XSTypeDefinition.SIMPLE_TYPE) {
				return;
			}
			if (((XSComplexTypeDefinition) type).getContentType() != XSComplexTypeDefinition.CONTENTTYPE_ELEMENT) {
				return;
			}
			String namespace = type.getNamespace() == null ? getDefaultNamespace() : type.getNamespace();
			String typeName = type.getName();
			if (namespace != null && name != null && typeName != null) {
				elementTypeMap.addElementDeclaration(namespace, name, typeName);
			}
		}
	}

	/**
	 * Answer the constraint node after looking for a substitution element
	 * 
	 * @param uri
	 * @param localName
	 * @return
	 */
	protected ConstraintNode getSubstitutionGroupNode(String uri, String localName) {
		ConstraintNode constraintNode = null;
		String substitutionGroup = null;
		Map<String, String> subGroupMap = substitutionElementStructure.get(uri);
		if (subGroupMap != null) {
			// If an entry already exists for this URI, go get the sub group for the current element
			substitutionGroup = subGroupMap.get(localName);
		}
		if (substitutionGroup == null) {
			// If we don't yet have a cached sub group, go see if we can get one from the PSVI
			substitutionGroup = getSubstitutionGroup();
		}
		if (substitutionGroup != null) {
			// If we found a sub group in either of the previous steps, make sure we cache it
			if (subGroupMap == null) {
				// If there wasn't a map yet for the URI, create a new empty one
				subGroupMap = new HashMap<String, String> ();
				substitutionElementStructure.put(uri, subGroupMap);
			}
			// store the mapping of element to sub group
			subGroupMap.put(localName, substitutionGroup);
		}
		if (substitutionGroup != null) {
			constraintNode = getConstraintNode(ParserHelper.removeNameSpace(substitutionGroup));
		}
		return constraintNode;
	}

	/**
	 * Retrieve the substitutionGroup information for the current element from
	 * the PSVI.
	 * 
	 * @return
	 */
	protected String getSubstitutionGroup() {
		ElementPSVI elementPSVI = getPsvi().getElementPSVI();
		XSElementDeclaration elementDeclaration = elementPSVI == null ? null : elementPSVI.getElementDeclaration();
		if ((elementDeclaration == null) || (elementDeclaration.getSubstitutionGroupAffiliation() == null)) {
			return null;
		}
		return elementDeclaration.getSubstitutionGroupAffiliation().getName();
	}

	// Look up cached constraint node.  Construct one if necessary.
	private ConstraintNode getConstraintNode(String localName) {
		ConstraintNode constraintNode = (ConstraintNode)targetSchemaBuilderStructure.get(localName);
		if (constraintNode == null) {
			String elementName = localName;
			String targetElement = getAttributeValue(ISMLConstants.SML_URI,
					ISMLConstants.TARGETELEMENT_ATTRIBUTE);
			String targetType = getAttributeValue(ISMLConstants.SML_URI,
					ISMLConstants.TARGETTYPE_ATTRIBUTE);
			String targetRequired = getAttributeValue(ISMLConstants.SML_URI,
					ISMLConstants.TARGETREQUIRED_ATTRIBUTE);
			if (targetElement != null || targetType != null
					|| targetRequired != null) {

				int adjustedLine = getAdjustedLocatorLine();
				constraintNode = new ConstraintNode(elementName, getFilePath(),
						adjustedLine,
						getLocator() == null ? ValidationMessage.NO_LINE_NUMBER
								: getLocator().getColumnNumber());

				String namespace = getNamespace();
				constraintNode.setTargetElement(SMLValidatorUtil.toQName(namespace,
						targetElement));
				constraintNode.setTargetType(SMLValidatorUtil
						.toQName(namespace, targetType));
				constraintNode.setTargetRequired(targetRequired);
				targetSchemaBuilderStructure.put(elementName, constraintNode);
			}
		}
		return constraintNode;
	}

	private String getNamespace() {
		ElementPSVI elementPsvi = getPsvi().getElementPSVI();
		if (elementPsvi != null) {
			XSElementDeclaration elementDecl = elementPsvi.getElementDeclaration();
			if (elementDecl != null) {
				return elementDecl.getNamespace();
			}
		}
		return super.getTargetNamespace();
	}

	protected int getAdjustedLocatorLine() {
		if (getLocator() == null) {
			return ValidationMessage.NO_LINE_NUMBER;
		}
		return getBaseLineNumber() + getLocator().getLineNumber() - 1;
	}
	
	public ElementTypeMap getElementTypeMap() {
		return elementTypeMap;
	}

	protected String getAttributeValue(String uri, String attribute)
			throws DOMException {
		AnnotationResult annotationResult = retrieveElementAnnotation (uri, attribute);
		if (annotationResult != null) {
			return annotationResult.getNodes()[0].getNodeValue();
		}
		return null;
	}
	
	public void characters(char[] ch, int start, int len) throws SAXException {
		// extract the URI value
		if (uriElementFound){
			String target = new String(ch, start, len);
			dataStructure.put(currentTargetElement, target);
			String uri = currentTargetElement.getTargetURI();
			currentTargetElement.setTargetURI(uri == null ? target : uri+target);			
		}
	}	

	public void endElement(String uri, String localName, String qName) throws SAXException {
		if ((uriElementFound) && (ISMLConstants.URI_ELEMENT.equals(localName))){
			uriElementFound = false;
			currentTargetElement = null;
		}
	}

	public Map<String, ConstraintNode> getTargetSchemaBuilderStructure() {
		return targetSchemaBuilderStructure;
	}

	public Map<String, Map<String, String>> getSubstitutionGroupStructure() {
		return substitutionElementStructure;
	}	

}
