/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.smlvalidators;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.xml.xpath.XPathFunctionException;

import org.apache.xerces.xs.ElementPSVI;
import org.apache.xerces.xs.XSComplexTypeDefinition;
import org.apache.xerces.xs.XSConstants;
import org.apache.xerces.xs.XSElementDeclaration;
import org.apache.xerces.xs.XSModel;
import org.apache.xerces.xs.XSNamedMap;
import org.apache.xerces.xs.XSTypeDefinition;
import org.eclipse.cosmos.rm.internal.validation.artifacts.AcyclicStructure;
import org.eclipse.cosmos.rm.internal.validation.artifacts.DOMStructure;
import org.eclipse.cosmos.rm.internal.validation.artifacts.AcyclicStructure.AcyclicEntry;
import org.eclipse.cosmos.rm.internal.validation.common.ISMLConstants;
import org.eclipse.cosmos.rm.internal.validation.common.IValidationConstants;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidationMessages;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidatorUtil;
import org.eclipse.cosmos.rm.internal.validation.common.AbstractValidationOutput.ValidationMessage;
import org.eclipse.cosmos.rm.internal.validation.common.AbstractValidationOutput.ValidationMessageFactory;
import org.eclipse.cosmos.rm.internal.validation.core.AbstractSMLValidator;
import org.eclipse.cosmos.rm.internal.validation.core.IValidator;
import org.eclipse.cosmos.rm.internal.validation.databuilders.AcyclicDataBuilder;
import org.eclipse.cosmos.rm.internal.validation.databuilders.DataBuilderRegistry;
import org.eclipse.cosmos.rm.internal.validation.databuilders.DocumentDOMBuilder;
import org.eclipse.cosmos.rm.internal.validation.databuilders.AbstractDataBuilder.AnnotationResult;
import org.eclipse.cosmos.rm.internal.validation.reference.DerefXPathFunction;
import org.eclipse.osgi.util.NLS;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;


/**
 * This class is used to validate the acyclic sml extension 
 * 
 * @author Ali Mehregani
 */
public class AcyclicValidator extends AbstractSMLValidator
{	

	/**
	 * A list of elements representing the graph nodes 
	 */
	private List<ResolvedReference> graphNodes;

	/**
	 * The acyclic structure
	 */
	private AcyclicStructure acyclicStructure;
	
	/**
	 * DOM representation of the instance documents
	 */
	private DOMStructure domDocuments;
	
	
	/**
	 * Constructor
	 */
	public AcyclicValidator()
	{
		graphNodes = new ArrayList<ResolvedReference>();
	}
	
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.launcher.IValidator#initialize(java.util.Map)
	 */
	public void initialize(Map<String, Object> validationAttribute)
	{
		super.initialize(validationAttribute);
		
		/* Register the right data builders with the registry */
		DataBuilderRegistry databuilderRegistry = DataBuilderRegistry.getInstanceLevelRegistry();
		databuilderRegistry.registerDataStructureBuilder(AcyclicDataBuilder.ID, new AcyclicDataBuilder());		
	}
	

	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.launcher.IValidator#validate()
	 */
	public boolean validate()
	{
		setTaskName(SMLValidationMessages.validationAcyclic);
		isSyntaxValid();
		
		// Retrieve the data structure required by this validator
		acyclicStructure = (AcyclicStructure)SMLValidatorUtil.retrieveDataStructure(AcyclicDataBuilder.ID);		
		domDocuments = (DOMStructure)SMLValidatorUtil.retrieveDataStructure(DocumentDOMBuilder.ID);
				
		String[] acyclicTypes = acyclicStructure.keys();
		
		// For each acyclic type
		for (int i = 0; i < acyclicTypes.length; i++)
		{
			AcyclicEntry[] entries = acyclicStructure.get(acyclicTypes[i]);
			
			// For each acyclic entry, follow the references to determine if
			// it forms a cycle.
			for (int j = 0; j < entries.length; j++)
			{
				graphNodes.clear();
				
				// Add the node of the alias containing the current entry
				String ownerAlias = entries[j].getOwnerAlias();
				Node documentNode = ownerAlias == null ? null : domDocuments.get(ownerAlias);
				if (documentNode != null)
				{
					ResolvedReference resolevedReference = new ResolvedReference(ownerAlias, documentNode);
					if (!graphNodes.contains(resolevedReference))
					{
						graphNodes.add(resolevedReference);
					}
				}
				
				
				if (isCyclePresent(entries[j]))
				{
					// Print an error message
					getValidationOutput().reportMessage(ValidationMessageFactory.createErrorMessage(
									entries[j].getLineNumber(), 
									NLS.bind(SMLValidationMessages.acyclicCycleFound, entries[j].getType().getName())));
					
					getValidationOutput().reportMessage(ValidationMessageFactory.createErrorMessage(
									entries[j].getLineNumber(), 
									NLS.bind(SMLValidationMessages.acyclicDetectedCycle, graphNodestoString())));
					return false;
				}
			}
		}
		
		return true;
	}

	
	/**
	 * Check to make sure the syntax of the acyclic constraint
	 * is correct.  We need to use the XSModel object to retrieve 
	 * all acyclic constraints. 
	 * 
	 * @return true if the identity constraints are syntactically
	 * correct; false otherwise
	 */
	private boolean isSyntaxValid()
	{
		XSModel xsModel = (XSModel)getAttributes().get(IValidator.ATTRIBUTE_XS_MODEL);
		boolean status = true;
		if (xsModel == null)
		{
			return status;
		}	
		
		boolean shouldAbortOnError = shouldAbortOnError();
		
		// Walk through the type definitions and display an error if
		// there is a definition with acyclic set to false and it has
		// an ancestor with acyclic set to true
		XSNamedMap typeDeclarations = xsModel.getComponents(XSTypeDefinition.COMPLEX_TYPE);
		for (int i = 0, declarationCount = typeDeclarations.getLength(); 
	 	 	 i < declarationCount && (!shouldAbortOnError || status); i++)
		{
			XSComplexTypeDefinition typeDefinition = (XSComplexTypeDefinition)typeDeclarations.item(i);			
			AnnotationResult annotationResult = SMLValidatorUtil.retrieveAnnotation(typeDefinition, ISMLConstants.SML_URI, ISMLConstants.ACYCLIC_ATTRIBUTE, true);
			if (annotationResult == null)
			{
				continue;
			}
			
			String acylicValue = annotationResult.getNodes()[0].getNodeValue();
			
			// Acyclic is set to false
			if (SMLValidatorUtil.isFalse(acylicValue))
			{
				// Walk up the hierarchy and ensure none of the types have acyclic = true
				XSTypeDefinition type = annotationResult.getType();
				XSTypeDefinition baseType = type.getBaseType();
				while (baseType != null && !SMLValidatorUtil.isAnyType(baseType))
				{
					AnnotationResult baseResult = SMLValidatorUtil.retrieveAnnotation(baseType, ISMLConstants.SML_URI, ISMLConstants.ACYCLIC_ATTRIBUTE, false);
					if (baseResult != null && SMLValidatorUtil.isTrue(baseResult.getNodes()[0].getNodeValue()))
					{					
						getValidationOutput().reportMessage(ValidationMessageFactory.createErrorMessage(
										ValidationMessage.NO_LINE_NUMBER, 
										NLS.bind(SMLValidationMessages.acyclicBadDerivation, new String[]{typeDefinition.getName(), baseType.getName()})
						));
						status = false;
						break;
					}
					baseType = type.getBaseType();
				}							
			}
		}
		
		
		// Display a warning if the acyclic attribute is defined on 
		// an element declaration
		XSNamedMap elementDeclarations = xsModel.getComponents(XSConstants.ELEMENT_DECLARATION);
		for (int i = 0, declarationCount = elementDeclarations.getLength(); 
		 	 i < declarationCount && (!shouldAbortOnError || status); i++)
		{
			XSElementDeclaration elementDeclaration = (XSElementDeclaration) elementDeclarations.item(i);
			AnnotationResult result = SMLValidatorUtil.retrieveAnnotation(elementDeclaration, ISMLConstants.SML_URI, ISMLConstants.ACYCLIC_ATTRIBUTE);
			if (result != null)
			{
				getValidationOutput().reportMessage(ValidationMessageFactory.createWarningMessage(
						ValidationMessage.NO_LINE_NUMBER, NLS.bind(SMLValidationMessages.acyclicBadDeclaration, elementDeclaration.getName())));
				continue;
			}
		}
		
		
		return status;
	}
	
	
	/**
	 * Converts the entries in {@link #graphNodes} to
	 * a string representation.
	 * 
	 * @return A string representation of {@link #graphNodes}
	 */
	private String graphNodestoString()
	{
		// Given that a cycle was detected, the last entry should always be 
		// the same as the first entry
		if (graphNodes.size() > 0)
		{
			graphNodes.add(graphNodes.get(0));
		}
		
		
		StringBuffer sb = new StringBuffer();
		for (int i = 0, gnCount = graphNodes.size(); i < gnCount; i++)
		{
			sb.append (i > 0 ? ", " : IValidationConstants.EMPTY_STRING);			
			sb.append (graphNodes.get(i).reference);
		}
		
		return sb.toString();
	}
	

	/**
	 * Given an acyclic entry, follow its references to determine if
	 * it forms a cycle.
	 * 
	 * @param acyclicEntry The acyclic entry to be followed
	 * @return true if a cycle is present; false otherwise
	 */
	private boolean isCyclePresent(AcyclicEntry acyclicEntry)
	{	
		// Resolve the reference of this acyclic entry
		ResolvedReference resolvedReference = resolveReference(acyclicEntry.getReference());				
		if (resolvedReference != null)
		{
			if (graphNodes.contains(resolvedReference))
			{
				return true;
			}
			
			graphNodes.add(resolvedReference);			
			AcyclicEntry[] entries = retrieveAcyclicEntries (resolvedReference.getAlias(), resolvedReference.node, acyclicEntry.getType());
			for (int i = 0; i < entries.length; i++)
			{
				boolean cycle = isCyclePresent(entries[i]);
				if (cycle)
				{
					return true;
				}				
			}
			graphNodes.remove(resolvedReference);
		}
		
		return false;
	}
	

	/**
	 * Given a node N and an acyclic complex type CT, retrieve all acyclic
	 * entries contained by N that are of type CT.
	 * 
	 * @param alias The alias of the document containing N
	 * @param node The node N
	 * @param type The acyclic complex type CT
	 * @return The acyclic entries of type CT contained by N
	 */
	private AcyclicEntry[] retrieveAcyclicEntries(String alias, Node node, XSComplexTypeDefinition type)
	{
		List<AcyclicEntry> acyclicEntries = new ArrayList<AcyclicEntry>();
		retrieveAcyclicEntries(acyclicEntries, alias, node, type);
		return acyclicEntries.toArray(new AcyclicEntry[acyclicEntries.size()]);
	}


	/**
	 * Convenience method used by {@link #retrieveAcyclicEntries(String, Node, XSComplexTypeDefinition)}
	 */
	private void retrieveAcyclicEntries(List<AcyclicEntry> entries, String alias, Node node, XSComplexTypeDefinition type)
	{
		if (checkNodeType(node, type))
		{
			addAcyclicEntry (entries, alias, node, type);
		}
		
		// Traverse through the children of N
		NodeList children = node.getChildNodes();
		for (int i = 0, childCount = children.getLength(); i < childCount; i++)
		{
			retrieveAcyclicEntries(entries, alias, children.item(i), type);
		}
	}

	
	/**
	 * Return true if node N is of type CT or is a derived type of
	 * CT.
	 * 
	 * @param node Node N
	 * @param type Type CT
	 * @return True if N is of type or derived from CT
	 */
	private boolean checkNodeType(Node node, XSComplexTypeDefinition type)
	{
		if (node.getNodeType() != Node.ELEMENT_NODE)
		{
			return false;
		}
		
		ElementPSVI elementPSVI = (ElementPSVI)node;
		XSElementDeclaration elementDeclaration = elementPSVI.getElementDeclaration();
		if (elementDeclaration == null)
		{
			return false;
		}
		XSTypeDefinition typeDefinition = elementPSVI.getElementDeclaration().getTypeDefinition();
		return isEqual(typeDefinition, type);
	}

	
	/**
	 * Returns true if T is of type or a derived type of CT
	 * 
	 * @param actual Type T
	 * @param expected Type CT
	 * @return true if T is or derived from CT 
	 */
	private boolean isEqual(XSTypeDefinition actual, XSComplexTypeDefinition expected)
	{
		if (SMLValidatorUtil.isAnyType(actual) || actual.getTypeCategory() != XSTypeDefinition.COMPLEX_TYPE)
		{
			return false;
		}
				
		return actual.getNamespace().equals(expected.getNamespace()) && actual.getName().equals(expected.getName()) ? 
				true : 
				isEqual(((XSComplexTypeDefinition)actual).getBaseType(), expected);
	}

	
	/**
	 * Determine if node N is an acyclic entry registered with the acyclic
	 * structure.  Add N to entries if an associated acyclic entry is found
	 * 
	 * @param entries The list that entries are added to
	 * @param alias The alias of the document containing N
	 * @param node The node N
	 * @param type The type of N
	 */
	private void addAcyclicEntry(List<AcyclicEntry> entries, String alias, Node node, XSComplexTypeDefinition type)
	{
		String reference = SMLValidatorUtil.extractReference(node);
		AcyclicEntry[] discoveredEntries = acyclicStructure.get(type, alias);
		for (int i = 0; i < discoveredEntries.length; i++)
		{
			if (discoveredEntries[i].getURI().equals(node.getNamespaceURI()) &&
				discoveredEntries[i].getLocalName().equals(node.getLocalName()) &&
				discoveredEntries[i].getReference().equals(reference))
			{
				entries.add(discoveredEntries[i]);
				return;
			}
		}
	}
	
	
	/**
	 * Resolve the SML reference R
	 * 
	 * @param reference Reference R
	 * @return Node T, where T is the resolved SML reference R
	 */
	private ResolvedReference resolveReference(String reference)
	{
		if (reference == null)
		{
			return null;
		}
		List<String> arguments = new ArrayList<String>();
		arguments.add(reference);
		
		
		try
		{
			// The output is expected to be one element
			Object output = DerefXPathFunction.instance().evaluate(arguments);
			NodeList result;
			if (output instanceof NodeList && (result = (NodeList)output).getLength() == 1)
			{
				return new ResolvedReference(reference, result.item(0));
			}
		} 
		catch (XPathFunctionException e)
		{
			getValidationOutput().reportMessage(ValidationMessageFactory.createErrorMessage(
					ValidationMessage.NO_LINE_NUMBER, 
					NLS.bind(SMLValidationMessages.errorReferenceResolving, reference)));
		}		
		
		return null;
	}

	
	private static class ResolvedReference
	{
		private String reference;
		private Node node;
		
		public ResolvedReference(String reference, Node resolvedReference)
		{
			this.reference = reference;
			this.node = resolvedReference;
		}		
		
		
		public String getAlias()
		{
			int hashInx = reference.indexOf('#');
			return hashInx >= 0 ? reference.substring(0, hashInx) : reference;			
		}


		@Override
		public boolean equals(Object o)
		{
			if (!(o instanceof ResolvedReference))
			{
				return false;
			}
			
			ResolvedReference other = (ResolvedReference)o;
			return node == null ? other.node == null : node.equals(other.node);
		}
	}
}
