/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.smlvalidators;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;

import javax.xml.namespace.NamespaceContext;
import javax.xml.namespace.QName;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpression;
import javax.xml.xpath.XPathExpressionException;

import org.apache.xerces.xs.XSAnnotation;
import org.apache.xerces.xs.XSComplexTypeDefinition;
import org.apache.xerces.xs.XSConstants;
import org.apache.xerces.xs.XSElementDeclaration;
import org.apache.xerces.xs.XSModel;
import org.apache.xerces.xs.XSNamedMap;
import org.apache.xerces.xs.XSTypeDefinition;
import org.eclipse.cosmos.rm.internal.validation.artifacts.DOMStructure;
import org.eclipse.cosmos.rm.internal.validation.artifacts.IdentityConstraintStructure;
import org.eclipse.cosmos.rm.internal.validation.artifacts.MappedNamespaceContext;
import org.eclipse.cosmos.rm.internal.validation.artifacts.IdentityConstraintStructure.IdentityConstraint;
import org.eclipse.cosmos.rm.internal.validation.artifacts.IdentityConstraintStructure.IdentityInstance;
import org.eclipse.cosmos.rm.internal.validation.common.ISMLConstants;
import org.eclipse.cosmos.rm.internal.validation.common.IValidationConstants;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidationMessages;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidatorUtil;
import org.eclipse.cosmos.rm.internal.validation.common.AbstractValidationOutput.ValidationMessage;
import org.eclipse.cosmos.rm.internal.validation.common.AbstractValidationOutput.ValidationMessageFactory;
import org.eclipse.cosmos.rm.internal.validation.core.AbstractSMLValidator;
import org.eclipse.cosmos.rm.internal.validation.core.IValidator;
import org.eclipse.cosmos.rm.internal.validation.databuilders.DataBuilderRegistry;
import org.eclipse.cosmos.rm.internal.validation.databuilders.DocumentDOMBuilder;
import org.eclipse.cosmos.rm.internal.validation.databuilders.IdentityConstraintDataBuilder;
import org.eclipse.cosmos.rm.internal.validation.databuilders.IdentityConstraintUtil;
import org.eclipse.osgi.util.NLS;
import org.w3c.dom.Document;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * This class is used to validate SML identity constraints.  i.e.:
 * <ul>
 * 	<li> key </li>
 * 	<li> keyref </li>
 * 	<li> unique </li>
 * </ul>
 * 
 * @author Ali Mehregani
 */
public class IdentityConstraintValidator extends AbstractSMLValidator
{
	/**
	 * The identity constraint data structure
	 */
	private IdentityConstraintStructure identityStruct;
	
	/**
	 * The referenced identity constraints
	 */
	private List<IdentityConstraint> referencedConstraints;
	
	/**
	 * The concreted identity constraints
	 */
	private Map<QName, IdentityConstraint> concreteConstraints;
	
	
	/**
	 * Constructor
	 */
	public IdentityConstraintValidator()
	{
		referencedConstraints = new ArrayList<IdentityConstraint>();
		concreteConstraints = new Hashtable<QName, IdentityConstraint>();
	}
	
	
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.launcher.IValidator#initialize(java.util.Map)
	 */
	public void initialize(Map<String, Object> validationAttribute)
	{
		super.initialize(validationAttribute);
		
		/* Register the right data builders with the registry */
		DataBuilderRegistry databuilderRegistry = DataBuilderRegistry.getInstanceLevelRegistry();		
		databuilderRegistry.registerDataStructureBuilder(IdentityConstraintDataBuilder.ID, new IdentityConstraintDataBuilder());		
	}
	

	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.launcher.IValidator#validate()
	 */
	public boolean validate()
	{		
		// Check the syntax first
		if (!isSyntaxValid())
		{
			if (shouldAbortOnError()) 
			{
				return false;
			}
		}
		
		boolean isValidationSuccess = true;
		setTaskName(SMLValidationMessages.validationIdentity);		
		
		// Retrieve the data structures required as part of the validation		
		identityStruct = (IdentityConstraintStructure)SMLValidatorUtil.retrieveDataStructure(IdentityConstraintDataBuilder.ID);	
		DOMStructure domStructure = (DOMStructure)SMLValidatorUtil.retrieveDataStructure(DocumentDOMBuilder.ID);
				
		// For each document containing an identity constraint instance 
		String[] constrainedInstances = identityStruct.getConstrainedAliases();		
		for (int i = 0; i < constrainedInstances.length; i++)
		{
			isValidationSuccess = handleIdentityInstances(
					identityStruct.getConstrainedInstances(constrainedInstances[i]), 
					domStructure.get(constrainedInstances[i]));
			
			if (shouldAbortOnError()) 
			{
				return false;
			}
		}
		
		// For each orphan document containing an identity constraint
		int[] orphanedIndices = identityStruct.getOrphanedIndices();
		for (int i = 0; i < orphanedIndices.length; i++)
		{
			isValidationSuccess = handleIdentityInstances(
					identityStruct.getOrphanedConstrainedInstances(orphanedIndices[i]), 
					domStructure.getOrphan(orphanedIndices[i]));
			if (shouldAbortOnError()) 
			{
				return false;
			}
		}
		
		return isValidationSuccess;
	}
	
	
	/**
	 * Check to make sure the syntax of the identity constraints
	 * is correct.  We need to use the XSModel object to retrieve 
	 * all identity constraints.  The reason why the identity constraint
	 * structure can't be used is because there is no guarantee 
	 * for all element declarations to have an instance (i.e. You
	 * can have an element declaration E with an invalid identity
	 * constraint that has no instance)
	 * 
	 * @return true if the identity constraints are syntactically
	 * correct; false otherwise
	 */
	private boolean isSyntaxValid()
	{
		XSModel xsModel = (XSModel)getAttributes().get(IValidator.ATTRIBUTE_XS_MODEL);
		if (xsModel == null)
		{
			return true;
		}
		
		XSNamedMap elementDeclarations = xsModel.getComponents(XSConstants.ELEMENT_DECLARATION);
		boolean shouldAbortOnError = shouldAbortOnError();
		boolean status = true;
		
		// For every element declaration
		for (int i = 0, declarationCount = elementDeclarations.getLength(); 
			 i < declarationCount && (!shouldAbortOnError || status); i++)
		{
			XSElementDeclaration elementDeclaration = (XSElementDeclaration)elementDeclarations.item(i);			
			List<IdentityConstraint> identityConstraints = IdentityConstraintUtil.retrieveConstraints (elementDeclaration, null);
			NamespaceContext namespaceContext = createNamespaceContext(elementDeclaration);
			List<QName> constraintNames = new ArrayList<QName>();
			
			for (int j = 0, constraintCount = identityConstraints.size(); j < constraintCount; j++)
			{
				IdentityConstraint identityConstraint = identityConstraints.get(j);				
				
				// If this happens to be a referenced identity constraint, try
				// to resolve the reference
				String ref = identityConstraint.getRef();
				if (ref != null)
				{
					// A name cannot be specified - show an error and skip the constraint
					if (identityConstraint.getName() != null)
					{
						getValidationOutput().reportMessage(ValidationMessageFactory.createErrorMessage(
								ValidationMessage.NO_LINE_NUMBER, NLS.bind(SMLValidationMessages.identityInvalidNameAttribute, identityConstraint.getRef())));		
						status = false;
						continue;
					}
					
					// Make sure the referenced constraint doesn't have any child elements
					String[] fields = identityConstraint.getFields();
					String selector = identityConstraint.getSelector();
					if ((fields != null && fields.length > 0) || (selector != null && selector.length() > 0))
					{
						getValidationOutput().reportMessage(ValidationMessageFactory.createErrorMessage(
								ValidationMessage.NO_LINE_NUMBER, NLS.bind(SMLValidationMessages.identityInvalidElements, identityConstraint.getRef())));					
						status = false;
						continue;
					}					
					
					identityConstraint.setNamespaceContext(namespaceContext);
					referencedConstraints.add(identityConstraint);
				}
				else
				{
					String constraintName = identityConstraint.getName();									
					
					String[] tokenized = SMLValidatorUtil.tokenizeName(constraintName);
					String namespace = tokenized[0] == null ? identityConstraint.getNamespace() : IdentityConstraintUtil.lookupNamespace(elementDeclaration, tokenized[0]);
					
					QName constraintQName = new QName(namespace, tokenized[1]);
					// An element declaration cannot have multiple identity constraints
					// with the same name
					if (constraintNames.contains(constraintQName))
					{
						getValidationOutput().reportMessage(ValidationMessageFactory.createErrorMessage(
								ValidationMessage.NO_LINE_NUMBER, NLS.bind(SMLValidationMessages.identityDuplicateName, constraintName)));
						status = false;
						continue;
					}
					
					constraintNames.add(constraintQName);	
					concreteConstraints.put(constraintQName, identityConstraint);
				}
			}		
		}
		
		
		// Make sure all referenced constraints can be resolved
		for (int i = 0, refCount = referencedConstraints.size(); i < refCount && (!shouldAbortOnError || status); i++)
		{
			IdentityConstraint referenceConstraint = referencedConstraints.get(i);
			
			// Lookup the reference 
			String[] qNameStr = SMLValidatorUtil.tokenizeName(referenceConstraint.getRef());					
			String namespace = qNameStr[0] == null ? IValidationConstants.EMPTY_STRING : referenceConstraint.getNamespaceContext().getNamespaceURI(qNameStr[0]);
			namespace = namespace == null ? IValidationConstants.EMPTY_STRING : namespace;				
					
			IdentityConstraint resolvedConstraint = concreteConstraints.get(new QName(namespace, qNameStr[1]));
			if (resolvedConstraint != null)
			{
				// Return an error if the type of the referenced
				// constraint is inconsistent
				if (resolvedConstraint.getType() != referenceConstraint.getType())
				{
					getValidationOutput().reportMessage(ValidationMessageFactory.createErrorMessage(
							ValidationMessage.NO_LINE_NUMBER, NLS.bind(SMLValidationMessages.identityBadReferenceType, referenceConstraint.getRef())));					
					status = false;
					continue;
				}
				
				changeReference(referenceConstraint, resolvedConstraint);
			}
			else
			{
				getValidationOutput().reportMessage(ValidationMessageFactory.createErrorMessage(
						ValidationMessage.NO_LINE_NUMBER, NLS.bind(SMLValidationMessages.identityMissingConstraint, referenceConstraint.getRef())));					
				status = false;
				continue;
			}
			
		}
		
		
		// Return a warning if an identity constraint is defined
		// on a type declaration		
		XSNamedMap typeDeclarations = xsModel.getComponents(XSTypeDefinition.COMPLEX_TYPE);
		for (int j = 0, count = typeDeclarations.getLength(); j < count; j++)
		{
			XSComplexTypeDefinition complexTypeDefinition = (XSComplexTypeDefinition) typeDeclarations.item(j);
			if (SMLValidatorUtil.retrieveAnnotation(complexTypeDefinition, ISMLConstants.SML_URI, IdentityConstraintDataBuilder.IDENTITY_CONSTRAINTS, false) != null)
			{
				getValidationOutput().reportMessage(ValidationMessageFactory.createWarningMessage(
						ValidationMessage.NO_LINE_NUMBER, NLS.bind(SMLValidationMessages.identityBadDeclaration, complexTypeDefinition.getName())));
			}
		}						
				
		return status;
	}

	
	private NamespaceContext createNamespaceContext(XSElementDeclaration elementDeclaration)
	{
		MappedNamespaceContext namespaceContext = new MappedNamespaceContext();		
		Document document = SMLValidatorUtil.createDocument();
		XSAnnotation annotation = elementDeclaration.getAnnotation();
		if (annotation == null)
		{
			return namespaceContext;
		}
		elementDeclaration.getAnnotation().writeAnnotation(document, XSAnnotation.W3C_DOM_DOCUMENT);
		NamedNodeMap attributes = document.getFirstChild().getAttributes();
		for (int i = 0, attCount = attributes.getLength(); i < attCount; i++)
		{
			Node attribute = attributes.item(i);			
			if (ISMLConstants.XML_NS_ATTRIBUTE.equals(attribute.getPrefix()))
			{								
				namespaceContext.addEntry(attribute.getLocalName(), attribute.getNodeValue());
			}
		}
		
		return namespaceContext;
	}


	private void changeReference(IdentityConstraint referenceConstraint, IdentityConstraint resolvedConstraint)
	{
		IdentityInstance[] identityConstraint = identityStruct.getIdentityConstraints();
		
		// For each identity constraint, change the referenced constraint to
		// its concrete one
		String referencedConstraint = referenceConstraint.getRef();		
		for (int i = 0; i < identityConstraint.length; i++)
		{
			IdentityConstraint constraint = identityConstraint[i].getConstraint();
			String ref = constraint.getRef();
			if (ref != null && ref.equals(referencedConstraint))
			{
				identityConstraint[i].setConstraint(resolvedConstraint);				
			}
		}
	}

	
	/**
	 * Given a set of identity constraint instances, along with the root 
	 * document node N, validate the identity constraint instance
	 * 
	 * @param identityInstances A set of identity constraint instances
	 * @param node The node N
	 * @return true if validation passes; false otherwise
	 */
	private boolean handleIdentityInstances(IdentityInstance[] identityInstances, Node node)
	{
		if (node == null)
		{
			return true;
		}
		
		boolean status = true;
		Map<String, List<String>> documentConstraints = new Hashtable<String, List<String>>();
		
		// For each identity constraint instance
		for (int j = 0; j < identityInstances.length; j++)
		{
			// Validate the constraint			
			String error = validateIdentityConstraints (node, identityInstances[j], documentConstraints);
			if (error != null)
			{															
				status = false;
				getValidationOutput().reportMessage(ValidationMessageFactory.createErrorMessage(identityInstances[j].getLineNumber(), error));
				if (shouldAbortOnError()) 
				{
					return false;
				}
			}
		}
		
		return status;
	}

	
	/**
	 * Given a context node N and an identity constraint instance I,
	 * validate I.
	 * 
	 * @param root The context node N
	 * @param identityInstance The identity instance I
	 * @param constraintFields A map where key = constraint name and the value
	 * is a list of fields the constraint resolves to.
	 * 
	 * @return true if validation passes; false otherwise
	 */
	private String validateIdentityConstraints (Node root, IdentityInstance identityInstance, Map<String, List<String>> constraintFields)
	{
		Node context = findNode(root, identityInstance.getNodePath());			
		IdentityConstraint constraint = identityInstance.getConstraint();
										
		String error = null;
		try
		{												
			XPathExpression selector = null;
			String[] fields = null;
			List<String> fieldValues = new ArrayList<String>();
			
			synchronized(SMLValidatorUtil.xpath)
			{
				if (constraint.getNamespaceContext() != null)
				{
					SMLValidatorUtil.xpath.setNamespaceContext(constraint.getNamespaceContext());
				}
				
				selector = SMLValidatorUtil.xpath.compile(constraint.getSelector());													
				fields = constraint.getFields();
				NodeList selectorList = (NodeList)selector.evaluate(context, XPathConstants.NODESET);
				if (selectorList == null || selectorList.getLength() <= 0)
				{
					return null;
				}
									
				fieldValues = new ArrayList<String>();
				
				// For every node in the scope of the selector xpath expression
				for (int j = 0, nodeCount = selectorList.getLength(); j < nodeCount; j++)
				{
					String fieldsPerNode = "";
					
					// For every field of the constraint
					for (int k = 0; k < fields.length; k++)
					{
						Node currentNode = selectorList.item(j);
						String currentField = extractField(currentNode, fields[k]);
						fieldsPerNode += currentField == null ? IValidationConstants.EMPTY_STRING : (fieldsPerNode.length() > 0 ? ", " + currentField : currentField);
						
						// The field must exist if the constraint is a key
						if (currentField == null && constraint.getType() == IdentityConstraint.KEY_TYPE)
						{
							return NLS.bind(SMLValidationMessages.identityMissingField, new String[]{fields[k], constraint.getName()});
						}
					}
											
					fieldValues.add(fieldsPerNode);
				}										
			}
			
			constraintFields.put(constraint.getName(), fieldValues);
			error = validateConstraint(constraintFields, constraint, fieldValues);
			if (error != null)
				return error;
			
		}
		catch (XPathExpressionException e)
		{		
			return NLS.bind(
							SMLValidationMessages.identityXPathError, 
							new String[]{constraint.getSelector(), constraint.getName()}) + 
							IValidationConstants.LINE_SEPARATOR + (e.getMessage() == null ? IValidationConstants.EMPTY_STRING : e.getMessage()
							); 
		}
		
		return null;
	}
	
	private String validateConstraint(Map<String, List<String>> constraintMap, IdentityConstraint constraint, List<String> fields)
	{ 
		switch (constraint.getType())
		{
			case IdentityConstraint.UNIQUE_TYPE:
			case IdentityConstraint.KEY_TYPE:				
				
				Hashtable<String, String> indexedHashtable = new Hashtable<String, String>();
				for (int i = 0, listSize = fields.size(); i < listSize; i++)
				{
					if (indexedHashtable.get(fields.get(i)) != null)
						return NLS.bind(SMLValidationMessages.identityDuplicateKey, new String[]{constraint.getName(), fields.get(i).toString()});
					indexedHashtable.put(fields.get(i), IValidationConstants.EMPTY_STRING);
				}			
				break;
				
			case IdentityConstraint.KEY_REF_TYPE:
				
				String reference = constraint.getRefer();
				List<String> referencedConstraint = null;
				if (reference == null)
				{
					return NLS.bind(SMLValidationMessages.identityMissingReference, constraint.getName());
				}
				else if ((referencedConstraint = constraintMap.get(reference)) == null)
				{
					return NLS.bind(SMLValidationMessages.identityMissingConstraint, constraint.getName());
				}
				
				/* For every field of the key ref constraint */
				for (int i = 0, fieldSize = fields.size(); i < fieldSize; i++)
				{
					String currentField = fields.get(i);
					boolean foundField = false;
					
					/* For every field of the referenced constraint */ 
					for (int j = 0, referenceList = referencedConstraint.size(); j < referenceList; j++)
					{
						if (currentField.equals(referencedConstraint.get(j)))
						{
							foundField = true;
							break;
						}
					}
					
					if (!foundField)
					{
						return NLS.bind(SMLValidationMessages.identityMissingKey, new String[]{currentField, constraint.getName()});
					}
				}
				break;
			default: 
				break;
		}

		return null;
	}

	private String extractField(Node context, String field)
	{
		try
		{
			XPathExpression fieldXPath = SMLValidatorUtil.xpath.compile(field);
			NodeList nodeList = (NodeList)fieldXPath.evaluate(context, XPathConstants.NODESET);
			Node fieldNode = nodeList.getLength() == 1 ? nodeList.item(0) : null;
			if (fieldNode == null)
				return null;
			
			if (field.charAt(0) == '@')
				return fieldNode.getNodeValue();
			NodeList childNodes = fieldNode.getChildNodes();
			StringBuffer fieldText = new StringBuffer();
			for (int i = 0, chidNodeCount = childNodes.getLength(); i < chidNodeCount; i++)
			{
				Node currentNode = childNodes.item(0);
				if (Node.TEXT_NODE != currentNode.getNodeType())
					return field.toString();
				fieldText.append(currentNode.getNodeValue() == null ? IValidationConstants.EMPTY_STRING : currentNode.getNodeValue());
			}
			return fieldText.toString();
		}
		catch (XPathExpressionException e)
		{
			return null;
		}
	}

	private Node findNode(Node node, int[] indices)
	{
		Node currentNode = node;
		NodeList nodeList = null;
		for (int i = 1; i < indices.length; i++)
		{
			nodeList = node.getChildNodes();
			if (indices[i] < 0 || indices[i] >= nodeList.getLength())
				return null;
			currentNode = nodeList.item(indices[i]);
		}
		
		return currentNode;
	}
	
}
