/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.repository;

import java.io.FileInputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;

import org.eclipse.cosmos.rm.internal.repository.application.ISMLResourceFacet;
import org.eclipse.cosmos.rm.internal.repository.application.ISMLResourceInstance;
import org.eclipse.cosmos.rm.internal.repository.application.ISMLResourceProperty;
import org.eclipse.cosmos.rm.internal.repository.application.impl.SMLFileResourceFacet;
import org.eclipse.cosmos.rm.internal.repository.application.impl.SMLFileResourceInstance;
import org.eclipse.cosmos.rm.internal.repository.application.impl.SMLResourceProperty;
import org.eclipse.cosmos.rm.internal.repository.application.impl.TypeInfoHandler;
import org.eclipse.cosmos.rm.internal.repository.core.IFileSystemSMLProperties;
import org.eclipse.cosmos.rm.internal.repository.resource.SMLFileDocument;
import org.eclipse.cosmos.rm.internal.repository.resource.SMLFileInstanceDocument;
import org.eclipse.cosmos.rm.internal.validation.common.ISMLConstants;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidatorUtil;
import org.eclipse.cosmos.rm.provisional.repository.core.ISMLRepository;
import org.eclipse.cosmos.rm.provisional.repository.exception.RepositoryOperationException;
import org.eclipse.cosmos.rm.provisional.repository.resource.ISMLDocument;
import org.eclipse.cosmos.rm.provisional.repository.resource.ISMLIFIdentity;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * Consists of common utility methods
 * 
 * @author Ali Mehregani
 */
public class SMLRepositoryUtil
{
	/**
	 * Retrieves and returns the resources of 'root'.  Only the resources that are direct
	 * children of root are returned.
	 * 
	 * @param repository The repository
	 * @param document The parent document
	 * @param root The root node that will be used as the context
	 * 
	 * @return The direct resources of 'root'
	 */
	public static ISMLResourceInstance[] retrieveResources(ISMLRepository repository, ISMLResourceInstance document, Node root)
	{
		if (document == null || root == null)
			return new ISMLResourceInstance[0];
				
		parseDocument ((SMLFileInstanceDocument)document, repository);
		List<ISMLResourceInstance> resourceList = new ArrayList<ISMLResourceInstance>();
		loadResources(repository, document, resourceList, root, 0, 1);
				
		return resourceList.toArray(new ISMLResourceInstance[resourceList.size()]);
	}
	
	
	/**
	 * Retrieves and returns the facets of 'root'.  Only the facets that are direct
	 * children of root are returned.
	 * 
	 * @param document The resource instance
	 * @param root The root node 
	 * @param repository The repository
	 * @return The direct facets of 'root'
	 */
	public static ISMLResourceFacet[] retrieveFacets(ISMLResourceInstance document, Node root, ISMLRepository repository)
	{
		if (document == null || root == null)
			return new ISMLResourceFacet[0];
				
		parseDocument ((SMLFileInstanceDocument)document, repository);
		List<ISMLResourceFacet> facetList = new ArrayList<ISMLResourceFacet>();
		loadFacets(repository, document, facetList, root, 0, 1);
				
		return facetList.toArray(new ISMLResourceFacet[facetList.size()]);
	}

	
	private static void loadResources(ISMLRepository repository, ISMLResourceInstance document, List<ISMLResourceInstance> resourceList, Node root, int currentDepth, int maximumDepth)
	{
		loadChildren(repository, resourceList, document,
				root, currentDepth, maximumDepth, 
				null,
				ISMLRepositoryConstants.ITEM_TYPE,
				ISMLRepositoryConstants.ITEM_REF_ELEMENT,
				SMLFileResourceInstance.class);
	}
	
	
	private static void loadFacets(ISMLRepository repository, ISMLResourceInstance document, List<ISMLResourceFacet> facetList, Node root, int currentDepth, int maximumDepth)
	{
		loadChildren(repository, facetList, document,
					root, currentDepth, maximumDepth, 
					ISMLRepositoryConstants.FACETS_TYPE,
					ISMLRepositoryConstants.FACET_TYPE,
					ISMLRepositoryConstants.FACET_REF_ELEMENT,
					SMLFileResourceFacet.class);
	}
	
	
	@SuppressWarnings("unchecked")
	public static boolean isDerivedType(ISMLRepository repository, Node child, String uri, String localTypeName)
	{		
		Map<String, Map<String, String>> typeInfo = (Map<String, Map<String, String>>)repository.getProperty(IFileSystemSMLProperties.TYPE_INFORMATION, (Object)null);
		Map<String, Map<String, String>> hierarchyInfo = (Map<String, Map<String, String>>)repository.getProperty(IFileSystemSMLProperties.HIERARCHY_INFORMATION, (Object)null);
		if (hierarchyInfo == null || typeInfo == null)
			return false;
				
		String expectedType = uri + ":" + localTypeName;
		String currentType = retrieveType(typeInfo, child.getNamespaceURI(), child.getLocalName()); 
				
		do
		{
			if (currentType != null && currentType.equals(expectedType))
				return true;
			
		}while ((currentType = getParent(hierarchyInfo, currentType)) != null);
				
		return false;
	}


	@SuppressWarnings("unchecked")
	public static String retrieveType(Map<String, Map<String, String>> typeInfo, String namespaceURI, String localTypeName)
	{
		if (typeInfo == null || namespaceURI == null || localTypeName == null)
			return null;
		
		Map<String, String> types = (Map<String, String>)SMLValidatorUtil.retrieveNestedMap(typeInfo, namespaceURI, false);	
		return types == null ? null : types.get(localTypeName);
	}


	@SuppressWarnings("unchecked")
	private static String getParent(Map<String, Map<String, String>> hierarchyInfo, String type)
	{
		String[] typeTokens = tokenizeQualifiedName(type);
		
		if (typeTokens[0] == null || typeTokens[1] == null)
			return null;
		
		Map<String, String> hierarchy = (Map<String, String>)SMLValidatorUtil.retrieveNestedMap(hierarchyInfo, typeTokens[0], false);
		if (hierarchy == null)
			return null;
		
		return hierarchy.get(typeTokens[1]);
	}


	/**
	 * Retrieves type information of each element that is included as part of
	 * this document.  The information is stored as a property of the repository
	 *  
	 * @param document The document
	 * @param repository The repository
	 */
	@SuppressWarnings("unchecked")
	public static void parseDocument(SMLFileInstanceDocument document, ISMLRepository repository)
	{
		Map<String, Map<String, String>> typeInfo = (Map)repository.getProperty(IFileSystemSMLProperties.TYPE_INFORMATION, (Object)null);
		Map<String, Map<String, String>> hierarchyInfo = (Map)repository.getProperty(IFileSystemSMLProperties.HIERARCHY_INFORMATION, (Object)null);
		
		
		typeInfo = typeInfo == null ? new Hashtable<String, Map<String, String>>() : typeInfo;
		hierarchyInfo = hierarchyInfo == null ? new Hashtable<String, Map<String, String>>() : hierarchyInfo;
		
		try
		{
			// Find the schema location attribute of the document
			Node root = document.getDOMDocument().getFirstChild();
			NamedNodeMap attributes = root.getAttributes();
			Node schemaLocation = attributes == null ? null : attributes.getNamedItemNS(ISMLConstants.SCHEMA_INSTANCE_URI, ISMLConstants.SCHEMA_LOCATION_ATTRIBUTE);
			if (schemaLocation == null)
				return;
			
			StringTokenizer schemaFiles = new StringTokenizer(schemaLocation.getNodeValue());
			
			// For every schema name space and location
			while (schemaFiles.hasMoreTokens())
			{
				String uri = schemaFiles.nextToken();
				String location = schemaFiles.nextToken();
							
				// Process the schema file
				processSchemaFile (getDocumentPath(document), typeInfo, hierarchyInfo, uri, location);
			}
			
		} 
		catch (RepositoryOperationException e)
		{			
			e.printStackTrace();
			return;
		}
		
		
		repository.setProperty(IFileSystemSMLProperties.TYPE_INFORMATION, typeInfo);
		repository.setProperty(IFileSystemSMLProperties.HIERARCHY_INFORMATION, hierarchyInfo);
	}


	/**
	 * Processes the schema file that is passed in 
	 * 
	 * @param parentLocation The parent location
	 * @param typeInfo Stores the type information
	 * @param hierarchyInfo 
	 * @param uri The URI corresponding to the schema file
	 * @param location The location of the schema file 
	 */
	private static void processSchemaFile(String parentLocation, Map<String, Map<String, String>> typeInfo, Map<String, Map<String, String>> hierarchyInfo, String uri, String location)
	{
		if (typeInfo.get(uri) != null)
			return;
		
		try
		{
			parentLocation = parentLocation.replace('\\', '/');
			if (!location.startsWith(parentLocation))
			{
				int lastSegment = parentLocation.lastIndexOf('/');
				String parentFolder = lastSegment < 0 ? parentLocation : parentLocation.substring(0, lastSegment);
				location = parentFolder + "/" + location;
			}
			
			FileInputStream fis = new FileInputStream(location);
			TypeInfoHandler typeInfoHandler = new TypeInfoHandler(typeInfo, hierarchyInfo);
			SMLValidatorUtil.saxParseDocument(fis, typeInfoHandler);
			
			Map<String, String> importedStatement = typeInfoHandler.getImportStatements();
			for (Iterator<String> namespaceURI = importedStatement.keySet().iterator(); namespaceURI.hasNext();)
			{
				String importedURI = namespaceURI.next();
				processSchemaFile (location, typeInfo, hierarchyInfo, importedURI, (String)importedStatement.get(importedURI));
			}			
		} 
		catch (Exception e)
		{
			// Ignore the exception
		} 		
	}


	/**
	 * Retrieves and returns the properties of 'root'.  Only the properties that are direct
	 * children of root are returned.
	 * 
	 * @param repository The repository
	 * @param document The parent document
	 * @param root The root node that will be used as the context
	 * 
	 * @return The direct properties of 'root'
	 */
	public static ISMLResourceProperty[] retrieveProperties (ISMLRepository repository, ISMLResourceInstance document, Node root)
	{
		if (document == null || root == null)
			return new ISMLResourceProperty[0];
				
		parseDocument ((SMLFileInstanceDocument)document, repository);
		List<ISMLResourceProperty> propertiesList = new ArrayList<ISMLResourceProperty>();
		loadProperties(repository, document, propertiesList, root, 0, 1);
				
		return propertiesList.toArray(new ISMLResourceProperty[propertiesList.size()]);		
	}


	private static void loadProperties(ISMLRepository repository, ISMLResourceInstance document, List<ISMLResourceProperty> propertiesList, Node root, int currentDepth, int maxDepth)
	{
		loadChildren(repository, propertiesList, document,
				root, currentDepth, maxDepth, 
				ISMLRepositoryConstants.PROPERTIES_TYPE,
				ISMLRepositoryConstants.PROPERTY_TYPE,
				ISMLRepositoryConstants.PROPERTY_REF_ELEMENT,
				SMLResourceProperty.class);		
	}


	@SuppressWarnings("unchecked")
	private static void loadChildren(ISMLRepository repository, List list, ISMLResourceInstance document, 
									 Node root, int currentDepth, int maxDepth, String groupElementType, 
									 String elementType, String referenceType, Class entity)
	{
		NodeList children = root.getChildNodes();
		for (int i = 0, childCount = children.getLength(); i < childCount; i++)
		{
			Node child = children.item(i);
			if (child.getNodeType() != Node.ELEMENT_NODE)
				continue;
			
			if (groupElementType != null && currentDepth < maxDepth && 
				isDerivedType(repository, child, ISMLRepositoryConstants.URI_ITEM, groupElementType))
			{
				loadChildren (	repository, list, document, 
								child, currentDepth+1, maxDepth, 
								groupElementType, elementType, referenceType, 
								entity);
			}
			else if (isDerivedType(repository, child, ISMLRepositoryConstants.URI_ITEM, elementType) ||
					 (ISMLRepositoryConstants.URI_ITEM.equals(child.getNamespaceURI()) && 
					  referenceType.equals(child.getLocalName())))
			{
				Object item;
				try
				{
					item = entity.getMethod("load", new Class[]{ISMLRepository.class, Node.class}).invoke(null, new Object[]{repository, (Node)children.item(i)});					
					if (item != null)
					{
						entity.getMethod("setParentDocument", new Class[]{ISMLDocument.class}).invoke(item, new Object[]{document});
						list.add(item);
					}
				} 
				catch (Exception e)
				{
					e.printStackTrace();
				}				
			}			
		}
	}	


	/**
	 * Retrieve the value of the attribute with the name space and local name 
	 * passed in 
	 * 
	 * @param node The context node
	 * @param uri The name space URI
	 * @param localName The local name
	 * @return The value of the attribute or null if it can't be found
	 */
	public static String retrieveRootAttribute(Node node, String uri, String localName)
	{
		NamedNodeMap attributes = node == null ? null : node.getAttributes();
		Node attribute = attributes == null ? null : attributes.getNamedItemNS(uri, localName);
		
		/* If the attribute is not found, then walk through all attributes and look for its local name */
		if (attribute == null && attributes != null)
		{
			for (int i = 0, attributeCount = attributes.getLength(); i < attributeCount; i++)
			{
				Node currentAttribute = attributes.item(i);
				if (localName.equals(currentAttribute.getLocalName()))
				{
					attribute = currentAttribute;
					break;
				}
			}
		}
		
		return attribute == null ? null : attribute.getNodeValue();	
	}


	/**
	 * Returns the index-th segment of path starting from the beginning.
	 * Null is returned if it is not found 
	 * 
	 * @param path The path
	 * @param index The index of the segment
	 * @return index-th segment from left; null if it can't be found
	 */
	public static String firstSegment(String path, int index)
	{
		StringTokenizer segments = new StringTokenizer (path, "\\/");
		if (index > segments.countTokens())
			return null;
		
		String desiredSegment = null;
		for (int i = 0; i < index; i++)
		{
			desiredSegment = segments.nextToken();
		}
		
		return desiredSegment;
	}
	
	
	/**
	 * Returns a flag indicating if the passed in string is null 
	 * or an empty string
	 * 
	 * @param string The string to be checked
	 * @return true if string is null or string.length <= 0
	 */
	public static boolean isNullOrEmpty(String string) 
	{
		return (string == null) || (string.length() <= 0);
	}
	
	
	/**
	 * Create the starting tags to a new SML-IF file, including the identity
	 * information, and append them to the string buffer.
	 * 
	 * @param buffer The string buffer to append the data to
	 * @param identity The SML-IF identity
	 */
	public static void appendSMLIFIdentityInformation(StringBuffer buffer, ISMLIFIdentity identity) 
	{
		buffer.append(ISMLRepositoryConstants.smlifStart)
		.append(ISMLConstants.tab).append(ISMLConstants.OPEN_ANGLE_BRACKET).append(SMLValidatorUtil.tagNameFor(ISMLConstants.SMLIF_NAMESPACE_PREFIX, ISMLConstants.IDENTITY_ELEMENT));
		
		buffer.append(ISMLConstants.CLOSE_ANGLE_BRACKET).append(ISMLConstants.nl).append(ISMLConstants.tab).append(ISMLConstants.tab)
		.append(SMLValidatorUtil.beginTagFor(ISMLConstants.SMLIF_NAMESPACE_PREFIX, ISMLConstants.NAME_ELEMENT)).append(identity.getName()).append(SMLValidatorUtil.endTagFor(ISMLConstants.SMLIF_NAMESPACE_PREFIX, ISMLConstants.NAME_ELEMENT));		

		if (!isNullOrEmpty(identity.getVersion())) 
		{
			buffer.append(ISMLConstants.nl).append(ISMLConstants.tab).append(ISMLConstants.tab)
			.append(SMLValidatorUtil.beginTagFor(ISMLConstants.SMLIF_NAMESPACE_PREFIX, ISMLConstants.VERSION_ELEMENT)).append(identity.getVersion()).append(SMLValidatorUtil.endTagFor(ISMLConstants.SMLIF_NAMESPACE_PREFIX, ISMLConstants.VERSION_ELEMENT));
		}
		
		if (!isNullOrEmpty(identity.getDisplayName())) 
		{
			buffer.append(ISMLConstants.nl).append(ISMLConstants.tab).append(ISMLConstants.tab)
			.append(SMLValidatorUtil.beginTagFor(ISMLConstants.SMLIF_NAMESPACE_PREFIX, ISMLConstants.DISPLAY_NAME_ELEMENT))
			//.append(baseURI).append(ISMLConstants.FORWARD_SLASH).append(LANG_NAME).append(ISMLConstants.DOUBLE_QUOTE).append(ISMLConstants.CLOSE_ANGLE_BRACKET)
			.append(identity.getDisplayName()).append(SMLValidatorUtil.endTagFor(ISMLConstants.SMLIF_NAMESPACE_PREFIX, ISMLConstants.DISPLAY_NAME_ELEMENT));
		}
		
		buffer.append(ISMLConstants.nl).append(ISMLConstants.tab).append(ISMLConstants.tab).append(SMLValidatorUtil.beginTagFor(ISMLConstants.SMLIF_NAMESPACE_PREFIX, ISMLConstants.DESCRIPTION_ELEMENT))
		//.append(baseURI).append(ISMLConstants.FORWARD_SLASH).append(LANG_DESCRIPTION).append(ISMLConstants.DOUBLE_QUOTE).append(ISMLConstants.CLOSE_ANGLE_BRACKET)
		.append(ISMLConstants.nl).append(ISMLConstants.tab).append(ISMLConstants.tab).append(ISMLConstants.tab).append(identity.getDescription())
		.append(ISMLConstants.nl).append(ISMLConstants.tab).append(ISMLConstants.tab).append(SMLValidatorUtil.endTagFor(ISMLConstants.SMLIF_NAMESPACE_PREFIX, ISMLConstants.DESCRIPTION_ELEMENT))
		.append(ISMLConstants.nl).append(ISMLConstants.tab).append(ISMLConstants.tab).append(SMLValidatorUtil.beginTagFor(ISMLConstants.SMLIF_NAMESPACE_PREFIX, ISMLConstants.BASE_URI_ELEMENT)).append(identity.getBaseURI()).append(SMLValidatorUtil.endTagFor(ISMLConstants.SMLIF_NAMESPACE_PREFIX, ISMLConstants.BASE_URI_ELEMENT))
		.append(ISMLConstants.nl).append(ISMLConstants.tab).append(SMLValidatorUtil.endTagFor(ISMLConstants.SMLIF_NAMESPACE_PREFIX, ISMLConstants.IDENTITY_ELEMENT)).append(ISMLConstants.nl);
	}
	
	
	/**
	 * A document can have two alternative IDs: one relative to the context 
	 * root directory or an absolute path from the the context directory.
	 * 
	 * @param repository The repository
	 * @param id The id of the document
	 * @return The alternative id that can be used to reference the document
	 */
	public static String getAlternativeId(ISMLRepository repository, String id)
	{
		String contextDirectory = repository.getProperty(IFileSystemSMLProperties.ROOT_DIRECTORY, "");
		if (id.startsWith(contextDirectory))
			return id.substring(contextDirectory.length() + 1);
		return contextDirectory + (contextDirectory.endsWith("/") || contextDirectory.endsWith("\\") ? "" : "/") + id;
	}


	/**
	 * Returns the document's id relative to the context root directory
	 * 
	 * @param document The document
	 * @return The path of the document relative to the context root directory
	 */
	public static String getDocumentName(ISMLDocument document)
	{
		return findDocumentLocation(document, false);
	}
	
	
	/**
	 * Returns the full document's id (i.e. an absolute path)
	 * 
	 * @param document The document
	 * @return The absolute path of the document
	 */
	public static String getDocumentPath(ISMLDocument document)
	{
		return findDocumentLocation(document, true);
	}


	private static String findDocumentLocation(ISMLDocument document, boolean absolute)
	{
		ISMLRepository repository = ((SMLFileDocument)document).getRepository();
		String contextDirectory = repository.getProperty(IFileSystemSMLProperties.ROOT_DIRECTORY, "");
		String id = document.getMetadata().getId();
		boolean startsWithContext = id.startsWith(contextDirectory);
		
		return absolute ?	startsWithContext ? id : getAlternativeId(repository, id) : 
							startsWithContext ? getAlternativeId(repository, id) : id;
	}
	
	
	/**
	 * Tokenizes the qualified name passed in into two tokens:
	 * the prefix and the local name.
	 * 
	 * @param name The qualified name (expected to be in the format <prefix>:<localname>)
	 * @return The tokens of the qualified name
	 */
	public static String[] tokenizeQualifiedName(String name)
	{
		if (name == null)
			return new String[2];
		
		int colonInx = -1;
		String prefix = (colonInx = name.lastIndexOf(':')) < 0 ? null : name.substring(0, colonInx);
		name = colonInx < 0 ? name : name.substring(colonInx + 1);   
		
		return new String[]{prefix, name};
	}
	
	
	/**
	 * Retrieves the first alias of the document passed in.
	 * The method will first attempt to retrieve the alias from the aliases
	 * map.  The id of the document is returned if none is found in the aliases
	 * map.
	 * 
	 * @param document The document
	 * @param aliases Document aliases indexed by document id
	 * @return The first alias of the document
	 */
	public static String retrieveAlias(ISMLDocument document, Map<String, String[]> aliases)
	{			
		String id = document.getMetadata().getId();
		
		// Add the first alias or the document name
		String[] documentAliases = (String[])aliases.get(id);
		return 	documentAliases == null || documentAliases.length < 0 ? 
				SMLRepositoryUtil.getDocumentName(document) : documentAliases[0];
	}
	
	
	/**
	 * Converts the map into an String array.  It assumes that the map's
	 * values are {@link List} of {@link String}
	 * 
	 * @param collection The map to be converted
	 * @return A string array representation
	 */
	public static String[] toStringArray(Collection<List<String>> collection)
	{
		if (collection == null || collection.size() <= 0)
			return new String[0];
		
		List<String> finalList = new ArrayList<String>();
		for (Iterator<List<String>> ruleLists = collection.iterator(); ruleLists.hasNext();)
		{
			List<String> ruleList = ruleLists.next();
			for (int i = 0, ruleListCount = ruleList.size(); i < ruleListCount; i++)
			{
				String rule = (String)ruleList.get(i);
				if (!finalList.contains(rule))
					finalList.add(rule);
			}						
		}		
		
		return (String[])finalList.toArray(new String[finalList.size()]);
	}
}
