/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.repository.application.impl;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.cosmos.rm.internal.repository.ISMLRepositoryConstants;
import org.eclipse.cosmos.rm.internal.repository.application.ISMLResourceProperty;
import org.eclipse.cosmos.rm.internal.repository.reference.SMLQualifiedReference;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidatorUtil;
import org.eclipse.cosmos.rm.provisional.repository.core.ISMLRepository;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * An implementation of a resource property.  A property is usually strongly-typed
 * and the client is expected to be aware of how to interpret a property belonging
 * to a facet.  {@link #load(Node)} should be used to instantiate an instance of this class.
 * 
 * @author Ali Mehregani
 */
public class SMLResourceProperty extends AbstractSMLFragment implements ISMLResourceProperty
{
	/**
	 * The repository
	 */
	private ISMLRepository repository;
	
	/**
	 * The root node
	 */
	private Node root;
	
	/**
	 * Indicates whether this is a reference or not
	 */
	private boolean reference;
	
	
	/**
	 * Instantiate an instance of this class based on the DOM
	 * node passed in.
	 * 
	 * @param node The property node
	 * @return An instance of this class if node represents a property node;
	 * null otherwise
	 */
	public static ISMLResourceProperty load(ISMLRepository repository, Node root)
	{		
		if (root == null || root.getNodeType() != Node.ELEMENT_NODE)
			return null;
		
		return new SMLResourceProperty(repository, root, 
				ISMLRepositoryConstants.URI_ITEM.equals(root.getNamespaceURI()) &&
				ISMLRepositoryConstants.PROPERTY_REF_ELEMENT.equals(root.getLocalName()));
	}

	
	/**
	 * Constructor 
	 * 
	 * @param repository The repository
	 * @param root The root node
	 * @param reference true if this is a property reference; false otherwise
	 */
	private SMLResourceProperty(ISMLRepository repository, Node root, boolean reference)
	{
		this.repository = repository;
		this.root = root;
		this.reference = reference;
	}

	/**
	 * @see org.eclipse.cosmos.rm.internal.repository.application.ISMLResourceProperty#resolveReference()
	 */
	public ISMLResourceProperty resolveReference()
	{
		if (!reference)
			return this;
		
		return load(repository, new SMLQualifiedReference(repository, SMLValidatorUtil.extractReference(root)).resolveReference());
	}

	
	/**
	 * @see org.eclipse.cosmos.rm.internal.repository.application.ISMLResourceBase#isReference()
	 */
	public boolean isReference()
	{
		return reference;
	}

	/**
	 * @see org.eclipse.cosmos.rm.internal.repository.application.impl.AbstractSMLFragment#getRootNode()
	 */
	protected Node getRootNode()
	{
		return root;
	}

	
	/**
	 * @see org.eclipse.cosmos.rm.internal.repository.application.ISMLResourceProperty#retrieveAttributes()
	 */
	public String[] retrieveAttributes()
	{
		NamedNodeMap attributes = root.getAttributes();
		if (attributes == null)
			return new String[0];
		
		int attributeCount = attributes.getLength();
		List<String> attributeList = new ArrayList<String>(attributeCount);
		for (int i = 0; i < attributeCount; i++)
		{
			attributeList.add(attributes.item(i).getNodeName());
		}
		
		return attributeList.toArray(new String[attributeCount]);
	}

	
	/**
	 * @see org.eclipse.cosmos.rm.internal.repository.application.ISMLResourceProperty#retrieveChildren()
	 */
	public ISMLResourceProperty[] retrieveChildren()
	{
		NodeList children = root.getChildNodes();
		if (children == null)
			return new ISMLResourceProperty[0];
		
		int childCount = children.getLength();
		List<ISMLResourceProperty> properties = new ArrayList<ISMLResourceProperty>(childCount);
		for (int i = 0; i < childCount; i++)
		{
			Node node = children.item(i);
			if (node.getNodeType() != Node.ELEMENT_NODE)
				continue;
			
			ISMLResourceProperty property = SMLResourceProperty.load(repository, node);
			if (property != null)
				properties.add(property);
		}
		
		return properties.toArray(new ISMLResourceProperty[properties.size()]);
	}

	
	/**
	 * @see org.eclipse.cosmos.rm.internal.repository.application.ISMLResourceProperty#retrievePropertyName()
	 */
	public String retrievePropertyName()
	{
		return root.getLocalName();
	}


	/** 
	 * @see org.eclipse.cosmos.rm.internal.repository.application.ISMLResourceProperty#retrieveValue(java.lang.String)
	 */
	public String retrieveValue(String attribute)
	{
		NamedNodeMap attributes = root.getAttributes();
		Node attributeNode = attributes == null ? null : attributes.getNamedItem(attribute);
		return attributeNode == null ? null : attributeNode.getNodeValue();
	}
}
