/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.repository.core;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.StringTokenizer;

import org.eclipse.core.runtime.content.IContentDescriber;
import org.eclipse.cosmos.rm.internal.repository.RepositoryMessages;
import org.eclipse.cosmos.rm.internal.repository.SMLRepositoryPlugin;
import org.eclipse.cosmos.rm.internal.repository.SMLRepositoryUtil;
import org.eclipse.cosmos.rm.internal.repository.application.impl.SMLFileResourceInstance;
import org.eclipse.cosmos.rm.internal.repository.operations.AbstractListenerManager;
import org.eclipse.cosmos.rm.internal.repository.operations.FileExportOperation;
import org.eclipse.cosmos.rm.internal.repository.operations.FileImportOperation;
import org.eclipse.cosmos.rm.internal.repository.operations.FileValidateOperation;
import org.eclipse.cosmos.rm.internal.repository.operations.OperationEvent;
import org.eclipse.cosmos.rm.internal.repository.operations.cmdbf.SMLQueryOperation;
import org.eclipse.cosmos.rm.internal.repository.resource.SMLFileDefinitionDocument;
import org.eclipse.cosmos.rm.internal.repository.resource.SMLFileDocument;
import org.eclipse.cosmos.rm.internal.repository.resource.SMLFileInstanceDocument;
import org.eclipse.cosmos.rm.internal.repository.resource.SMLFileMetadata;
import org.eclipse.cosmos.rm.internal.validation.common.ISMLConstants;
import org.eclipse.cosmos.rm.internal.validation.content.AbstractXMLContentDescriber;
import org.eclipse.cosmos.rm.provisional.repository.core.ISMLRepository;
import org.eclipse.cosmos.rm.provisional.repository.exception.RepositoryConnectionException;
import org.eclipse.cosmos.rm.provisional.repository.exception.RepositoryOperationException;
import org.eclipse.cosmos.rm.provisional.repository.operations.IProgressListener;
import org.eclipse.cosmos.rm.provisional.repository.operations.ISMLExportOperation;
import org.eclipse.cosmos.rm.provisional.repository.operations.ISMLImportOperation;
import org.eclipse.cosmos.rm.provisional.repository.operations.ISMLOperation;
import org.eclipse.cosmos.rm.provisional.repository.operations.ISMLValidateOperation;
import org.eclipse.cosmos.rm.provisional.repository.operations.cmdbf.ICMDBfQueryOperation;
import org.eclipse.cosmos.rm.provisional.repository.resource.ISMLDocument;
import org.eclipse.cosmos.rm.provisional.repository.resource.ISMLMetadata;

/**
 * A file system implementation of an SML repository.  The repository works
 * from a root context folder that is provided as a connection attribute
 * when connecting to this repository. 
 * 
 * @author Ali Mehregani
 */
public class FileSystemSMLRepository extends AbstractListenerManager implements ISMLRepository, Serializable
{
	/**
	 * The serial version UID
	 */
	private static final long serialVersionUID = -7789205443658521889L;
	
	
	/**
	 * Prefixes used for each property type
	 */
	private static final String PROPERTY_PREFIX_OBJECT = "object.";			// Objects //$NON-NLS-1$
	private static final String PROPERTY_PREFIX_STRING = "string.";			// Strings //$NON-NLS-1$
	private static final String PROPERTY_PREFIX_BOOLEAN = "boolean.";		// Booleans //$NON-NLS-1$
	private static final String PROPERTY_PREFIX_INT = "int.";				// Ints //$NON-NLS-1$
	
	
	/**
	 * The attribute name of the configuration property file
	 */
	public static final String ATTRIBUTE_CONFIGURATION_PROPERTY = SMLRepositoryPlugin.PLUGIN_ID + ".ATTRIBUTE_CONFIGURATION_PROPERTY"; //$NON-NLS-1$
	
	/**
	 * The properties file used for the configuration for this repository
	 */
	private Properties configurationFile;
		
	/**
	 * The metadata processor
	 */
	private MetadataProcessor metadataProcessor;

	/**
	 * Active documents of this repository
	 */
	private ISMLDocument[] activeDocuments;

	/**
	 * The operations supported by this repository
	 */
	private Map<String, ISMLOperation> operations;

	/**
	 * Indicates the connection state of this repository
	 */
	private boolean connected;

	/**
	 * The properties of this repository
	 */
	private Map<String, Object> properties;
	
		
	/** 
	 * @see org.eclipse.cosmos.rm.provisional.repository.core.ISMLRepository#connect(java.util.Map)
	 */
	public void connect(Map<String, Object> attributes) throws RepositoryConnectionException
	{
		properties = new Hashtable<String, Object>();
		
		// First attempt to retrieve the context root directory and node from the attributes map
		String rootContextDirectory = (String)attributes.get(IFileSystemSMLProperties.ROOT_DIRECTORY);
		String rootContextNode = (String)attributes.get(IFileSystemSMLProperties.ROOT_CONTEXT_NODE);
		
			
		// If the root context directory is missing, then try to load it from the configuration
		// file
		boolean suppressErrors = attributes.get(IFileSystemSMLProperties.SUPPRESS_CONNECT_ERRORS) == null ? false : ((Boolean)attributes.get(IFileSystemSMLProperties.SUPPRESS_CONNECT_ERRORS)).booleanValue();
		if (rootContextDirectory == null)
		{
			String propertyName = (String)attributes.get(ATTRIBUTE_CONFIGURATION_PROPERTY);
			if (propertyName == null)
			{
				if (!suppressErrors)
					throw new RepositoryConnectionException(RepositoryMessages.repositoryMissingConfiguration);
			}
			else
			{
						
				configurationFile = new Properties();
				InputStream is;
				try
				{
					is = new FileInputStream(propertyName);
				} 
				catch (FileNotFoundException e1)
				{
					/* Try to load it using the class loader */
					is = this.getClass().getClassLoader().getResourceAsStream(propertyName);
				}
				
				try
				{
					configurationFile.load(is);
				} 
				catch (IOException e)
				{
					throw new RepositoryConnectionException(e.getLocalizedMessage(), e);
				}
				finally
				{
					try
					{
						if (is != null)
							is.close();
					}
					catch (IOException e)
					{
						/* Ignore the exception */
					}
				}
				
				rootContextDirectory = configurationFile.getProperty(IFileSystemSMLProperties.ROOT_DIRECTORY);
				rootContextNode = configurationFile.getProperty(IFileSystemSMLProperties.ROOT_CONTEXT_NODE);
			}
		}
		
		if (!suppressErrors && (rootContextDirectory == null || !new File(rootContextDirectory).exists()))
			throw new RepositoryConnectionException(rootContextDirectory == null ? 
				RepositoryMessages.repositoryMissingContextDirectory : 
				RepositoryMessages.repositoryBadRootDirectory);
		
		rootContextDirectory = rootContextDirectory == null ? rootContextDirectory : rootContextDirectory.replace('\\', '/');
		setProperty(IFileSystemSMLProperties.ROOT_DIRECTORY, rootContextDirectory);		
		setProperty(IFileSystemSMLProperties.ROOT_CONTEXT_NODE, rootContextNode);
		
		initializeOperations();		
		connected = true;
	}

	
	private void initializeOperations()
	{
		operations = new Hashtable<String, ISMLOperation>();
		operations.put(ISMLExportOperation.ID, new FileExportOperation(this));
		operations.put(ISMLImportOperation.ID, new FileImportOperation(this));
		operations.put(ISMLValidateOperation.ID, new FileValidateOperation(this));
		operations.put(ICMDBfQueryOperation.ID, new SMLQueryOperation(this));
	}


	/**
	 * The id of the meta-data is used as the file path.  The id is expected to be 
	 * specified relative to the root context directory.  If the id ends with a '/', 
	 * then the document is stored in the folder structure passed in.  If however 
	 * the id does not end with '/', then the last segment is used as the filename.  
	 * If no id is specified, then an arbitrary name is used and the document is stored 
	 * at the root context directory.  
	 * 
	 * @see org.eclipse.cosmos.rm.provisional.repository.core.ISMLRepository#addDocument(org.eclipse.cosmos.rm.provisional.repository.resource.ISMLMetadata, java.io.InputStream)
	 */
	public void addDocument(ISMLMetadata metadata, InputStream input) throws RepositoryOperationException
	{
		String fileName = metadata == null ? null : metadata.getId();
		String rootDirectory = getProperty(IFileSystemSMLProperties.ROOT_DIRECTORY, null);
		if (fileName != null && fileName.length() > 0)
		{
			fileName = fileName.startsWith("/") ? fileName.substring(1) : fileName; //$NON-NLS-1$
			fileName = rootDirectory + "/" + fileName; //$NON-NLS-1$
			fileName = fileName.endsWith("/") ? generateTempFileName(fileName, metadata) : fileName;  //$NON-NLS-1$
			
			// Create the directory
			int lastInx = fileName.lastIndexOf('/');
			if (lastInx >= 0)
			{
				File file = new File(fileName.substring(0, lastInx));
				
				// Create the required structure only if create structure is set to true
				boolean createStructure = getProperty(IFileSystemSMLProperties.CREATE_STRUCTURE, true);								
				if (!file.exists())
				{
					if (!createStructure)
					{
						// Notify the listeners						
						notifyListeners(OperationEvent.TYPE_MISSING_STRUCTURE, 
										file.getAbsolutePath());
						return;
					}
					file.mkdirs();					
				}
					
			}
		}
		else
		{
			fileName = rootDirectory + "/" + generateTempFileName(rootDirectory, metadata); //$NON-NLS-1$
		}
		
		
		FileOutputStream fos = null;
		try
		{
			// Notify the listeners if the file already exists and overwrite without 
			// notification is set to false
			boolean overwrite = getProperty(IFileSystemSMLProperties.OVERWRITE_WITHOUT_NOTIFICATION, false);
			if (new File(fileName).exists() && !overwrite)
			{
				boolean doit = notifyListeners(OperationEvent.TYPE_OVERWRITE, fileName);
				if (!doit)
					return;
			}
				
			
			fos = new FileOutputStream(fileName);
			byte[] buffer = new byte[1024];
			int length = 0;
			while (input.available() > 0)
			{
				length = input.read(buffer);
				fos.write(buffer, 0, length);
			}
			
			// Add the aliases to the meta-information of the repository
			String[] aliases = metadata.getAliases();
			if (aliases != null)
			{												
				metadataProcessor.addAliases(fileName.substring(rootDirectory.length() + 1), aliases);
				
				// Add the rule bindings (if there are any).  Notice that rule bindings will
				// only be added if there are aliases defined for the document being added.
				if (metadata.getBoundRules() != null)
				{
					Map<String, List<String>> rules = metadata.getBoundRules();
					for (Iterator<String> keys = rules.keySet().iterator(); keys.hasNext();)
					{
						String currentKey = (String) keys.next();
						List<String> ruleList = rules.get(currentKey);
						metadataProcessor.addRuleBinding(currentKey, ruleList.toArray(new String[ruleList.size()]));
					}
				}
			}
		} 
		catch (FileNotFoundException e)
		{
			throw new RepositoryOperationException(e.getLocalizedMessage(), e);
		} 
		catch (IOException e)
		{
			throw new RepositoryOperationException(e.getLocalizedMessage(), e);
		}	
		finally 
		{			
			try
			{
				if (fos != null)
					fos.close();
				input.close();
			} 
			catch (IOException e)
			{
				// Ignore the exception
			}			
		}
	}


	private boolean notifyListeners(byte type, Object data)
	{
		IProgressListener[] listeners = getProgressListeners();		
		if (listeners == null)
			return false;
		
		OperationEvent importEvent = OperationEvent.opeartionEvent;
		importEvent.type = type;
		importEvent.data = data;
		boolean doit = false;
		for (int i = 0; i < listeners.length; i++)
		{
			listeners[i].operationInterrupted(importEvent);
			doit = doit || importEvent.doit;
		}
		
		return doit;
	}


	private String generateTempFileName(String folder, ISMLMetadata metadata)
	{
		final String PREFIX_INSTANCE = "instance"; //$NON-NLS-1$
		final String PREFIX_DEFINITION = "definition"; //$NON-NLS-1$
		
		File fileFolder = new File (folder);
		String prefix = metadata.getDocumentType() == ISMLMetadata.DOCUMENT_TYPE_DEFINITION ? PREFIX_DEFINITION : PREFIX_INSTANCE;
		if (!shouldProcessFolder(fileFolder))
			return prefix;
		
		int counter = 0;
		String[] children = fileFolder.list();
		String fileName = prefix + counter;
		for (int i = 0; i < children.length; i++)
		{
			int extensionInx = -1;
			if ((extensionInx = children[i].indexOf('.')) > 0)
				children[i] = children[i].substring(0, extensionInx);
			
			if (children[i].equals(fileName))
				fileName = prefix + ++counter;			
		}
		fileName += fileName.startsWith(PREFIX_INSTANCE) ? ".xml" : ".xsd"; //$NON-NLS-1$ //$NON-NLS-2$
		return fileName;
	}


	/** 
	 * @see org.eclipse.cosmos.rm.provisional.repository.core.ISMLRepository#disconnect()
	 */
	public void disconnect()
	{				
		connected = false;
		metadataProcessor.write();
	}

	/**
	 * @see org.eclipse.cosmos.rm.provisional.repository.core.ISMLRepository#fetchDocuments(org.eclipse.cosmos.rm.provisional.repository.resource.ISMLMetadata)
	 */
	public ISMLDocument[] fetchDocuments(ISMLMetadata metadata) throws RepositoryOperationException
	{
		return fetchDocuments(metadata, false);
	}
	
	
	/**
	 * The following fields of the meta-data is used in this order:
	 * <ol>
	 *  <li> id </li>  
	 *  <li> alias </li>
	 *  <li> rule bindings </li>
	 *  <li> document type </li>
	 *  <li> root element name </li>
	 * </ol>
	 * @see org.eclipse.cosmos.rm.provisional.repository.core.ISMLRepository#fetchDocuments(org.eclipse.cosmos.rm.provisional.repository.resource.ISMLMetadata)
	 */
	private ISMLDocument[] fetchDocuments(ISMLMetadata metadata, boolean rootContext) throws RepositoryOperationException
	{
		List<ISMLDocument> documents = new ArrayList<ISMLDocument>();
		boolean processed = fetchIdRetrieval(documents, metadata.getId());			
		processed = fetchDataRetrieval (processed, documents, arrayToMap (metadata.getAliases()), true);
		
		Collection<List<String>> rules = metadata.getBoundRules() == null ? null : metadata.getBoundRules().values();
		String[] rulesStr = SMLRepositoryUtil.toStringArray(rules);
		processed = fetchDataRetrieval(processed, documents, arrayToMap (rulesStr), false);
		processed = fetchDocumentTypeRetrieval(processed, documents, metadata.getDocumentType());
		fetchElementNameRetrieval(processed, documents, metadata.getRootElementName(), rootContext);
		
		
		return documents.toArray(new ISMLDocument[documents.size()]);
	}	


	private boolean fetchIdRetrieval(List<ISMLDocument> documents, String id) throws RepositoryOperationException
	{			
		if (id != null && id.length() > 0)
		{
			String fileName = id.replace('\\', '/');
			String rootDirectory = getProperty(IFileSystemSMLProperties.ROOT_DIRECTORY, null);

			if (!fileName.startsWith(rootDirectory))
			{
				fileName = rootDirectory + (rootDirectory.endsWith("\\") || rootDirectory.endsWith("/") ? "" : "/") + fileName; //$NON-NLS-1$
			}
			
			// If the id is a directory, then load the entire content of the directory
			if (new File(fileName).isDirectory())
			{
				loadDocuments(documents, fileName, false);
			}
			// Otherwise attempt to load a file pointing to the same ID
			else
			{
				ISMLDocument smlDocument = loadDocument (fileName, false);
				if (smlDocument != null)
					documents.add(smlDocument);
			}
			
			return true;
		}
		
		return false;
	}


	private ISMLDocument loadDocument(String fileName, boolean rootContext) throws RepositoryOperationException
	{
		fileName = fileName.replace('\\', '/');
		File documentFile = new File(fileName);
		if (!documentFile.exists())
			return null;
		
		SMLFileMetadata metadata = new SMLFileMetadata();		
		metadata.setId(fileName);
		class RootElementExtractor extends AbstractXMLContentDescriber
		{
			private String rootElementName;
			
			public int determineType(String line)
			{		
				rootElementName = retrieveElementName(line);
				return IContentDescriber.INVALID;
			}
		}
		
		RootElementExtractor rootElementExtractor = new RootElementExtractor();
		FileInputStream fis = null;
		try
		{
			fis = new FileInputStream(fileName);
			rootElementExtractor.describe(fis, null);
			metadata.setDocumentType(ISMLConstants.SCHEMA_ELEMENT.equals(rootElementExtractor.rootElementName) ?
					ISMLMetadata.DOCUMENT_TYPE_DEFINITION : ISMLMetadata.DOCUMENT_TYPE_INSTANCE);
			metadata.setRootElementName(rootElementExtractor.rootElementName);
			metadata.setRepositoryMetadata(metadataProcessor);
			
						
			SMLFileDocument document = rootContext ? new SMLFileResourceInstance((ISMLRepository)this, (String)null) : (metadata.getDocumentType() == ISMLMetadata.DOCUMENT_TYPE_DEFINITION ?
									   (SMLFileDocument)new SMLFileDefinitionDocument(this) : new SMLFileInstanceDocument(this));
			document.setMetadata(metadata);
			
			return document;
		} 
		catch (FileNotFoundException e)
		{
			throw new RepositoryOperationException(e.getLocalizedMessage(), e);
		} 
		catch (IOException e)
		{
			throw new RepositoryOperationException(e.getLocalizedMessage(), e);
		}	
		finally 
		{
			
			try
			{
				if (fis != null)
					fis.close();
			} 
			catch (IOException e)
			{
				/* Ignore the exception */
			}
		}
	}


	private boolean fetchDataRetrieval(boolean processed, List<ISMLDocument> documents, Map<String, Boolean> data, boolean alias)
	{
		if (data == null || data.size() <= 0)
			return processed;
		
		if (!processed)
		{
			documents.addAll(metadataProcessor.retrieveDocuments(data, alias));
		}
		
		boolean foundAlias = false;
		
		/* For each document */
		List<ISMLDocument> finalList = new ArrayList<ISMLDocument>();
		for (int i = 0, documentCount = documents.size(); i < documentCount; i++)
		{
			ISMLDocument document = (ISMLDocument)documents.get(i);
			String[] documentData;
			if (alias)
			{
				documentData = document.getMetadata().getAliases();
			}
			else
			{
				Collection<List<String>> rules = document.getMetadata().getBoundRules() == null ? null : document.getMetadata().getBoundRules().values();
				documentData = SMLRepositoryUtil.toStringArray(rules);
			}
			
			
			foundAlias = false;
			
			if (documentData != null)
			{
				/* For each document aliases */
				for (int j = 0; j < documentData.length; j++)
				{
					if (data.get(documentData[j]) != null)
						foundAlias = true;
				}
			}
			
			if (foundAlias)
				finalList.add(document);
		}
		
		documents.removeAll(documents);
		documents.addAll(finalList);
		return true;
	}
	
	private Map<String, Boolean> arrayToMap(String[] data)
	{		
		Map<String, Boolean> dataMap = new Hashtable<String, Boolean>();
		if (data == null)
			return dataMap;
		for (int i = 0; i < data.length; i++)
		{
			dataMap.put(data[i], Boolean.TRUE);
		}
		
		return dataMap;
	}
	
	
	private boolean fetchDocumentTypeRetrieval(boolean processed, List<ISMLDocument> documents, int documentType) throws RepositoryOperationException
	{
		if (documentType < 0) {
			return processed;
		}
		
		if (!processed) {
			loadDocuments(documents, getProperty(IFileSystemSMLProperties.ROOT_DIRECTORY, null), false);
		}
				
		List<ISMLDocument> finalArrayList = new ArrayList<ISMLDocument>();
		for (int i = 0, documentsCount=documents.size(); i < documentsCount; i++)
		{
			ISMLDocument document = documents.get(i);
			if (document.getMetadata().getDocumentType() == documentType) {
				finalArrayList.add(document);
			}
		}
		
		documents.removeAll(documents);
		documents.addAll(finalArrayList);
		return true;
	}
	
	
	private void loadDocuments(List<ISMLDocument> documents, String folder, boolean rootContext) throws RepositoryOperationException
	{		
		File folderFile = new File(folder);
		if (!shouldProcessFolder(folderFile)) {
			return;
		}

		File[] children = folderFile.listFiles();
		for (int i = 0; i < children.length; i++)
		{
			if (children[i].isDirectory())
			{
				loadDocuments(documents, children[i].getAbsolutePath(), rootContext);
			}
			else if (!children[i].getName().startsWith(".")) //$NON-NLS-1$
			{				
				ISMLDocument document = loadDocument(children[i].getAbsolutePath(), rootContext);
				if (document != null) {
					documents.add(document);
				}
			}			
		}
	}

	/* Determine whether the folder's children should be traversed */
	private boolean shouldProcessFolder(File folderFile) {
		return folderFile.exists() && folderFile.isDirectory() && !folderFile.getName().startsWith(".") || !folderFile.getName().equals("CVS");
	}


	private void fetchElementNameRetrieval(boolean processed, List<ISMLDocument> documents, String rootElementName, boolean rootContext) throws RepositoryOperationException
	{
		if (rootElementName == null || rootElementName.length() <= 0)
			return;
		
		if (!processed)
			loadDocuments(documents, getProperty(IFileSystemSMLProperties.ROOT_DIRECTORY, null), rootContext);
		
		List<ISMLDocument> finalList = new ArrayList<ISMLDocument>();
		for (int i = 0, documentsCount=documents.size(); i < documentsCount; i++)
		{
			ISMLDocument document = (ISMLDocument)documents.get(i);
			if(rootElementName.equals(document.getMetadata().getRootElementName()))
				finalList.add(document);
		}			
		
		documents.removeAll(documents);
		documents.addAll(finalList);
	}	
	
	/**
	 * @see org.eclipse.cosmos.rm.provisional.repository.core.ISMLRepository#getActiveDocuments()
	 */
	public ISMLDocument[] getActiveDocuments()
	{
		return activeDocuments;
	}
	
	/**
	 * @see org.eclipse.cosmos.rm.provisional.repository.core.ISMLRepository#setActiveDocuments(org.eclipse.cosmos.rm.provisional.repository.resource.ISMLMetadata[])
	 */
	public void setActiveDocuments(ISMLDocument[] documents)
	{
		this.activeDocuments = documents;
	}
		
	/**
	 * @see org.eclipse.cosmos.rm.provisional.repository.core.ISMLRepository#getOperation(java.lang.String)
	 */
	public ISMLOperation getOperation(String id)
	{
		return id == null ? null : (ISMLOperation)operations.get(id);
	}

	/**
	 * @see org.eclipse.cosmos.rm.provisional.repository.core.ISMLRepository#getSupportedOperations()
	 */
	public String[] getSupportedOperations()
	{
		return (String[])operations.keySet().toArray(new String[operations.size()]);
	}

	/**
	 * @see org.eclipse.cosmos.rm.provisional.repository.core.ISMLRepository#isConnection()
	 */
	public boolean isConnection()
	{
		return connected;
	}

	/**
	 * @see org.eclipse.cosmos.rm.provisional.repository.core.ISMLRepository#removeDocuments(org.eclipse.cosmos.rm.provisional.repository.resource.ISMLMetadata)
	 */
	public void removeDocuments(ISMLMetadata metadata) throws RepositoryOperationException
	{
		ISMLDocument[] documents = fetchDocuments(metadata);
		for (int i = 0; i < documents.length; i++)
		{
			File file = new File(documents[i].getMetadata().getId());
			if (file.exists())
				file.delete();
		}
	}

	/**
	 * @see org.eclipse.cosmos.rm.provisional.repository.core.ISMLRepository#retrieveRootDocuments()
	 */
	public ISMLDocument[] retrieveRootDocuments() throws RepositoryOperationException
	{		
		String rootContextNode = getProperty(IFileSystemSMLProperties.ROOT_CONTEXT_NODE, null);
		if (rootContextNode == null)
			throw new RepositoryOperationException(RepositoryMessages.repositoryMissingContextNode);
		
		StringTokenizer contextNodes = new StringTokenizer (rootContextNode, ","); //$NON-NLS-1$
		List<ISMLDocument> documents = new ArrayList<ISMLDocument>();
		while (contextNodes.hasMoreTokens())
		{
			SMLFileMetadata metadata = new SMLFileMetadata();
			metadata.setRootElementName(contextNodes.nextToken().trim());			
			ISMLDocument[] fetchedDocuments = fetchDocuments(metadata, true);
			for (int i = 0; i < fetchedDocuments.length; i++)
			{
				documents.add(fetchedDocuments[i]);
			}
		}
		
		return documents.toArray(new ISMLDocument[documents.size()]);
	}


	/**
	 * @return the metadataProcessor
	 */
	public MetadataProcessor getMetadataProcessor()
	{
		return metadataProcessor;
	}

	
	private Object getProperty(String prefix, String name, Object defaultValue)
	{
		if (properties == null)
			return null;
		
		Object prop = properties.get(prefix + name);
		return prop == null ? defaultValue : prop;		
	}

	
	/**
	 * @see org.eclipse.cosmos.rm.provisional.repository.core.ISMLRepository#getProperty(java.lang.String, java.lang.Object)
	 */
	public Object getProperty(String name, Object defaultValue)
	{
		return getProperty(PROPERTY_PREFIX_OBJECT, name, defaultValue);		
	}

	/**
	 * @see org.eclipse.cosmos.rm.provisional.repository.core.ISMLRepository#getProperty(java.lang.String, java.lang.String)
	 */
	public String getProperty(String name, String defaultValue)
	{
		return (String)getProperty(PROPERTY_PREFIX_STRING, name, defaultValue);		
	}

	/**
	 * @see org.eclipse.cosmos.rm.provisional.repository.core.ISMLRepository#getProperty(java.lang.String, boolean)
	 */
	public boolean getProperty(String name, boolean defaultValue)
	{
		return ((Boolean)getProperty(PROPERTY_PREFIX_BOOLEAN, name, new Boolean(defaultValue))).booleanValue();		
	}

	/**
	 * @see org.eclipse.cosmos.rm.provisional.repository.core.ISMLRepository#getProperty(java.lang.String, int)
	 */
	public int getProperty(String name, int defaultValue)
	{
		return ((Integer)getProperty(PROPERTY_PREFIX_INT, name, new Integer(defaultValue))).intValue();		
	}

	
	private void setProperty (String prefix, String name, Object value)
	{
		if (properties == null || name == null || value == null)
			return;
		
		properties.put(prefix + name, value);
	}	
	
	/**
	 * @see org.eclipse.cosmos.rm.provisional.repository.core.ISMLRepository#setProperty(java.lang.String, java.lang.Object)
	 */
	public void setProperty(String name, Object value)
	{
		setProperty(PROPERTY_PREFIX_OBJECT, name, value);
	}

	/**
	 * @see org.eclipse.cosmos.rm.provisional.repository.core.ISMLRepository#setProperty(java.lang.String, java.lang.String)
	 */
	public void setProperty(String name, String value)
	{	
		String oldValue = getProperty(name, null);
		setProperty(PROPERTY_PREFIX_STRING, name, value);
		
		// We need to update the meta-data processor if the context root
		// directory changes
		if ((value == null ? oldValue != null : !value.equals(oldValue)) && IFileSystemSMLProperties.ROOT_DIRECTORY.equals(name))
			metadataProcessor = new MetadataProcessor(this);
	}

	/**
	 * @see org.eclipse.cosmos.rm.provisional.repository.core.ISMLRepository#setProperty(java.lang.String, boolean)
	 */
	public void setProperty(String name, boolean value)
	{
		setProperty(PROPERTY_PREFIX_BOOLEAN, name, new Boolean(value));
	}

	/**
	 * @see org.eclipse.cosmos.rm.provisional.repository.core.ISMLRepository#setProperty(java.lang.String, int)
	 */
	public void setProperty(String name, int value)
	{
		setProperty(PROPERTY_PREFIX_INT, name, new Integer(value));
	}
}
