/**********************************************************************
 * Copyright (c) 2006, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.repository.operations;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.cosmos.rm.internal.repository.core.FileSystemSMLRepository;
import org.eclipse.cosmos.rm.internal.repository.core.IFileSystemSMLProperties;
import org.eclipse.cosmos.rm.internal.validation.artifacts.ElementModel;
import org.eclipse.cosmos.rm.internal.validation.artifacts.ElementSchemaModel;
import org.eclipse.cosmos.rm.internal.validation.artifacts.SMLDocuments;
import org.eclipse.cosmos.rm.internal.validation.artifacts.SchemaBindings;
import org.eclipse.cosmos.rm.internal.validation.artifacts.ValidationSet;
import org.eclipse.cosmos.rm.internal.validation.artifacts.SchemaBindings.SchemaBinding;
import org.eclipse.cosmos.rm.internal.validation.common.AbstractValidationOutput;
import org.eclipse.cosmos.rm.internal.validation.common.FileOutput;
import org.eclipse.cosmos.rm.internal.validation.common.ISMLConstants;
import org.eclipse.cosmos.rm.internal.validation.common.IValidationConstants;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidationMessages;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidatorUtil;
import org.eclipse.cosmos.rm.internal.validation.common.SystemOutput;
import org.eclipse.cosmos.rm.internal.validation.common.AbstractValidationOutput.ValidationMessage;
import org.eclipse.cosmos.rm.internal.validation.common.AbstractValidationOutput.ValidationMessageFactory;
import org.eclipse.cosmos.rm.internal.validation.core.FoundationBuilder;
import org.eclipse.cosmos.rm.internal.validation.core.ISMLValidator;
import org.eclipse.cosmos.rm.internal.validation.core.ISchematronValidation;
import org.eclipse.cosmos.rm.internal.validation.core.IValidationListener;
import org.eclipse.cosmos.rm.internal.validation.core.IValidator;
import org.eclipse.cosmos.rm.internal.validation.core.ValidationEvent;
import org.eclipse.cosmos.rm.internal.validation.core.ValidationFactory;
import org.eclipse.cosmos.rm.internal.validation.databuilders.DataBuilderRegistry;
import org.eclipse.cosmos.rm.internal.validation.databuilders.DocumentCacheBuilder;
import org.eclipse.cosmos.rm.internal.validation.databuilders.ElementSchematronCacheBuilder;
import org.eclipse.cosmos.rm.internal.validation.databuilders.IdentityDataBuilder;
import org.eclipse.cosmos.rm.internal.validation.databuilders.SchemaBindingsBuilder;
import org.eclipse.cosmos.rm.internal.validation.smlvalidators.AcyclicValidator;
import org.eclipse.cosmos.rm.internal.validation.smlvalidators.IdentityConstraintValidator;
import org.eclipse.cosmos.rm.internal.validation.smlvalidators.ReferenceValidator;
import org.eclipse.cosmos.rm.internal.validation.smlvalidators.SchemaValidator;
import org.eclipse.cosmos.rm.internal.validation.smlvalidators.SchematronValidator;
import org.eclipse.cosmos.rm.internal.validation.smlvalidators.TargetValidator;
import org.eclipse.cosmos.rm.provisional.repository.core.ISMLRepository;
import org.eclipse.cosmos.rm.provisional.repository.core.SMLRepositoryFactory;
import org.eclipse.cosmos.rm.provisional.repository.exception.MissingRepositoryException;
import org.eclipse.cosmos.rm.provisional.repository.exception.RepositoryConnectionException;
import org.eclipse.cosmos.rm.provisional.repository.exception.RepositoryOperationException;
import org.eclipse.cosmos.rm.provisional.repository.operations.ISMLOperation;
import org.eclipse.cosmos.rm.provisional.repository.operations.ISMLValidateOperation;


/**
 * This is the starting point for a validation process.  Clients are encouraged to re-use the same instance
 * with the {@link this#initialize(Map)} and/or {@link this#setAttribute(String, String)} methods 
 * to configure an instance based on a specific setting.  
 * 
 * @author Ali Mehregani
 */
public class SMLMainValidator implements IValidationListener
{	
	private static final String OPTION_SML = "-sml";
	private static final String OPTION_SCHEMATRON = "-schematron";
	private static final String OPTION_FILE = "-output";
	private static final String OPTION_CONTEXT = "-context";
	private static final String OPTION_CONFIGURATION = "-configuration";
	private static final String OPTION_SUPPRESS_WARNINGS = "-sw";
	
	/**
	 * The SML-IF schema location
	 */
	private static final String SML_IF_SCHEMA = "validation-resources/smlif.xsd";
	
	/**
	 * Stores the user specified context directory
	 */
	private static String contextDirectory;
	
	/**
	 * Stores the user specified configuration file
	 */
	private static String configurationFile;
	
	/**
	 * Indicates if warnings should be suppressed
	 */
	private static boolean suppressWarnings;
	private static IValidationOutputFactory validationOutputFactory;
	private static IValidationOutputFactory defaultValidationOutputFactory = new IValidationOutputFactory() {
		public AbstractValidationOutput createEclipseOutput() {
			return new SystemOutput();
		}
	};
	
	/**
	 * The attributes of this validation class
	 */
	private Map<String, Object> attributes;
	
	/**
	 * The progress monitor
	 */
	private IProgressMonitor monitor;
	
	
	/**
	 * The constructor
	 */
	public SMLMainValidator()
	{
		this.attributes = new Hashtable<String, Object>();
		setDefaultAttributes();
	}
	
	
	/**
	 * Sets default values for attributes
	 */
	protected void setDefaultAttributes()
	{
		// Standalone environment
		attributes.put(IValidator.ATTRIBUTE_ENV, IValidator.VALUE_ENV_STANDALONE);
		
		// Default SML validators
		List<String> smlValidators = new ArrayList<String>();
		smlValidators.add(SchemaValidator.class.getName());
		smlValidators.add(AcyclicValidator.class.getName());
		smlValidators.add(IdentityConstraintValidator.class.getName());
		
		smlValidators.add(TargetValidator.class.getName());
		smlValidators.add(SchematronValidator.class.getName());
		
		smlValidators.add(ReferenceValidator.class.getName());
		attributes.put(IValidator.ATTRIBUTE_VALIDATION_SML, smlValidators);
		
		/* The SML-IF schema */
		attributes.put(IValidator.ATTRIBUTE_SML_IF_SCHEMA, SML_IF_SCHEMA);

		/* By default, continue processing after an error */
		attributes.put(IValidator.ATTRIBUTE_FAST_FAIL_POLICY, "false");
	}


	/**
	 * The initialize method is used to pass in multiple attributes
	 * at the same time
	 * 
	 * @param attributes The attributes.  The key and value pairs are expected 
	 * to be of type String
	 */
	public void initialize(Map<String, Object> attributes)
	{	
		for (Iterator<String> keys = attributes.keySet().iterator(); keys.hasNext();)
		{
			String currentKey = keys.next();
			setAttribute(currentKey, attributes.get(currentKey));
		}
		
		// Register the required top level data builders
		DataBuilderRegistry.getInstanceLevelRegistry().clear();
		DataBuilderRegistry topLevelRegistry = DataBuilderRegistry.getTopLevelRegistry(); 
		topLevelRegistry.clear();
		
		topLevelRegistry.registerDataStructureBuilder(DocumentCacheBuilder.ID, new DocumentCacheBuilder());
		topLevelRegistry.registerDataStructureBuilder(SchemaBindingsBuilder.ID, new SchemaBindingsBuilder());
		topLevelRegistry.registerDataStructureBuilder(IdentityDataBuilder.ID, new IdentityDataBuilder());
		topLevelRegistry.registerDataStructureBuilder(ElementSchematronCacheBuilder.ID, new ElementSchematronCacheBuilder());
	}


	/**
	 * Set an attribute of this type.  The key must be one of the ATTRIBUTE_* 
	 * constants defined by this class.
	 * 
	 * @param key The attribute key (see the ATTRIBUTE_* constants of this class)
	 * @param value The value of the attribute
	 */
	public void setAttribute(String key, Object value)
	{
		attributes.put(key, value);
	}

	
	/**
	 * Start the validation process
	 */
	public boolean start()
	{	
		// Setup a common output stream if none has been setup 
		AbstractValidationOutput validationOutput = (AbstractValidationOutput)attributes.get(IValidator.ATTRIBUTE_OUTPUT);
		if (validationOutput == null)
		{
			validationOutput = createValidationOutput();
			attributes.put(IValidator.ATTRIBUTE_OUTPUT, validationOutput);			
		}
		if (validationOutput != null)
		{
			validationOutput.initialize(attributes);
		}
		
		monitor = (IProgressMonitor)attributes.get(IValidator.ATTRIBUTE_PROGRESS_MONITOR);
		monitor = monitor == null ? new NullProgressMonitor() : monitor; 
		
		try
		{
			// Register the top-level data builders and parse the 
			// document once.		
			FoundationBuilder foundationBuilder = new FoundationBuilder(DataBuilderRegistry.TOP_LEVEL_MODE);
			foundationBuilder.initialize(attributes);
			foundationBuilder.validate();
			
			// Create the validators at the instance level
			List<IValidator> validators = createValidators();
			validators.add(0, new FoundationBuilder(DataBuilderRegistry.INSTANCE_LEVEL_MODE));
			
			// Form the validation sets and parse through each set
			List<ValidationSet> validationSets = createValidationSets();
			int validationSetCount = validationSets.size();
			monitor.beginTask(SMLValidationMessages.validationTask, validationSetCount * validators.size());
			
			
			// Iterate through validation set and invoke each
			// validator
			for (int i = 0; i < validationSetCount; i++)
			{
				// Reset the validators
				reset(validators);
				
				// Add the set to the attributes
				ValidationSet validationSet = validationSets.get(i);
				attributes.put(IValidator.ATTRIBUTE_VALIDATION_SET, validationSet);
				
				// Initialize the validators
				initialize(validators);
				
				// Run the validators
				execute(validators);
			}
			
			return validationOutput.getErrorCount() == 0;
		} 
		catch (Exception e)
		{			
			e.printStackTrace();
			String message = e.getMessage();
			validationOutput.reportMessage(ValidationMessageFactory.createErrorMessage(ValidationMessage.NO_LINE_NUMBER, SMLValidationMessages.errorValidatorInstantiation + (message == null ? IValidationConstants.EMPTY_STRING : message)));			
		} 		
		finally
		{
			
			validationOutput.close();
		}
		
		return false;
	}	
	
	
	/**
	 * Resets the validators passed in
	 * 
	 * @param validators The validators to be reseted
	 */
	private void reset(List<IValidator> validators)
	{
		for (int i = 0, validatorCount = validators.size(); i < validatorCount; i++)
		{
			validators.get(i).reset();
		}
	}


	/**
	 * Initialize the validators passed in
	 * 
	 * @param validators The validators to be initialized
	 */
	private void initialize(List<IValidator> validators)
	{
		for (int i = 0, validatorCount = validators.size(); i < validatorCount; i++)
		{
			IValidator validator = validators.get(i);
			validator.initialize(attributes);
			validator.addValidationListener(this);
		}
	}

	
	/**
	 * Execute the validators passed in, updating the progress
	 * monitor after the execution of each validator.
	 * 
	 * @param validators A list of validators
	 */
	private void execute(List<IValidator> validators)
	{		
		int currentIndex, validatorCount;
		for (currentIndex = 0, validatorCount = validators.size(); currentIndex < validatorCount; currentIndex++)
		{
			IValidator validator = validators.get(currentIndex);
			if (monitor.isCanceled())
			{
				break;
			}
			
			if (!validator.validate() && fastFailPolicy()) 
			{				
				break;
			}
						
			monitor.worked(1);			
		}		
		
		int remainingWork = validators.size() - currentIndex;
		if (remainingWork > 0)
		{
			monitor.worked(remainingWork);
		}
	}


	/**
	 * Assuming the existence of the schema bindings and the document
	 * cache builder structure, this method will return a set requiring
	 * validation
	 * 
	 * @return A set of validation sets
	 */
	private List<ValidationSet> createValidationSets()
	{
		List<ValidationSet> validationSets = new ArrayList<ValidationSet>();
		
		// Retrieve the required data structures
		SchemaBindings schemaBindings = (SchemaBindings)SMLValidatorUtil.retrieveDataStructure(DataBuilderRegistry.TOP_LEVEL_MODE, SchemaBindingsBuilder.ID);
		SMLDocuments smlDocuments = (SMLDocuments)SMLValidatorUtil.retrieveDataStructure(DataBuilderRegistry.TOP_LEVEL_MODE, DocumentCacheBuilder.ID);
		
		// Schema bindings is not present
		if (!schemaBindings.isPresent())
		{
			ValidationSet validationSet = new ValidationSet();
			ElementSchemaModel[] definitions = smlDocuments.getDefinitions();
			
			for (int i = 0; i < definitions.length; i++)
			{
				validationSet.addDefinitionDocument(definitions[i]);
			}
			
			ElementModel[] instances = smlDocuments.getInstances();
			for (int i = 0; i < instances.length; i++)
			{
				validationSet.addInstanceDocument(instances[i]);
			}			
			validationSets.add(validationSet);
		}
		
		// Otherwise it's present
		else
		{
			SchemaBinding[] bindings = schemaBindings.getBindings();
			List<ElementSchemaModel> addedDefinitions = new ArrayList<ElementSchemaModel>();
			List<ElementModel> addedInstances = new ArrayList<ElementModel>();
			
			// Add the default bindings to the 'addedDefinitions' list
			SchemaBinding defaultBinding = schemaBindings.getDefaultBinding();
			ElementSchemaModel[] defaultSchemaElements = null;
			if (defaultBinding != null)
			{
				String[] defaultSchemas = defaultBinding.getSchemaAliases();				
				for (int i = 0; i < defaultSchemas.length; i++)
				{
					ElementSchemaModel defaultSchema = smlDocuments.getDefinitionByAlias(defaultSchemas[i]);
					if (defaultSchemas != null)
					{
						addedDefinitions.add(defaultSchema);						
					}
				}
				
				if (!addedDefinitions.isEmpty())
				{
					defaultSchemaElements = addedDefinitions.toArray(new ElementSchemaModel[addedDefinitions.size()]);
				}
			}
			
			for (int i = 0; i < bindings.length; i++)
			{
				String[] aliases = bindings[i].getDocumentAliases();
				ValidationSet validationSet = new ValidationSet();
				for (int j = 0; j < aliases.length; j++)
				{
					ElementModel instanceDocument = smlDocuments.getInstanceByAlias(aliases[j]);
					if (instanceDocument != null)
					{
						validationSet.addInstanceDocument (instanceDocument);
						addedInstances.add(instanceDocument);
					}
				}
				
				aliases = bindings[i].getSchemaAliases();
				for (int j = 0; j < aliases.length; j++)
				{
					ElementSchemaModel elementSchemaModel = smlDocuments.getDefinitionByAlias(aliases[j]);
					if (elementSchemaModel != null)
					{
						validationSet.addDefinitionDocument(elementSchemaModel);
						addedDefinitions.add(elementSchemaModel);
					}
				}
				
				// Add the default bindings
				if (defaultSchemaElements != null)
				{
					validationSet.addDefinitionDocuments(defaultSchemaElements);
				}
				
				if (!validationSet.isEmpty())
				{
					validationSets.add(validationSet);
				}
			}
			
			// If there is a default binding, then walk through
			// each instance that hasn't yet been bounded and bind it to 
			// the default schemas
			if (defaultSchemaElements != null)
			{
				ElementModel[] instances = smlDocuments.getInstances();
				ValidationSet defaultValidationSet = new ValidationSet(); 
				boolean foundUnboundInstance = false;
				for (int i = 0; i < instances.length; i++)
				{
					if (!addedInstances.contains(instances[i]))
					{
						foundUnboundInstance = true;
						defaultValidationSet.addInstanceDocument(instances[i]);
					}
				}
				
				if (foundUnboundInstance)
				{
					defaultValidationSet.addDefinitionDocuments(defaultSchemaElements);
					validationSets.add(defaultValidationSet);
				}
			}
			
			// Add a dummy validation set for definition documents that
			// have not yet been included
			ElementSchemaModel[] definitions = smlDocuments.getDefinitions();
			ValidationSet dummySet = new ValidationSet();
			for (int i = 0; i < definitions.length; i++)
			{
				if (!addedDefinitions.contains(definitions[i]))
				{
					dummySet.addDefinitionDocument(definitions[i]);
				}
			}
			
			if (!dummySet.isEmpty())
			{
				validationSets.add(dummySet);
			}
		}
		
		return validationSets;
	}


	/**
	 * Creates a set of validators
	 * 
	 * @return The complete set of validators required in
	 * validating an SML-IF document
	 * 
	 * @throws ClassNotFoundException 
	 * @throws IllegalAccessException 
	 * @throws InstantiationException 
	 */
	private List<IValidator> createValidators() throws InstantiationException, IllegalAccessException, ClassNotFoundException
	{
		List<IValidator> validators = new ArrayList<IValidator>();
		addEntries(validators, ValidationFactory.createValidator(ISMLValidator.class, attributes.get(IValidator.ATTRIBUTE_VALIDATION_SML)));
		addEntries(validators, ValidationFactory.createValidator(ISchematronValidation.class, attributes.get(IValidator.ATTRIBUTE_VALIDATION_SCHEMATRON)));
		return validators;
	}


	private void addEntries(List<IValidator> validators, IValidator[] entries)
	{
		for (int i = 0; i < entries.length; i++)
		{
			validators.add(entries[i]);
		}
	}

	
	private boolean fastFailPolicy() {
		String fastFail = (String) attributes.get(IValidator.ATTRIBUTE_FAST_FAIL_POLICY);
		if (fastFail != null) {
			return Boolean.valueOf(fastFail).booleanValue();
		}
		return false;
	}


	/**
	 * Create a validation output based on the environment that the validation 
	 * is being performed in.
	 * 
	 * @return An output stream
	 */
	protected AbstractValidationOutput createValidationOutput()
	{
		String environment = (String)attributes.get(IValidator.ATTRIBUTE_ENV);
		AbstractValidationOutput validationOutput = null;
		/* Eclipse */
		if (IValidator.VALUE_ENV_ECLIPSE.equals(environment))
		{
			validationOutput = createEclipseOutput();
		}
		/* Standalone */
		else if (IValidator.VALUE_ENV_STANDALONE.equals(environment))
		{
			validationOutput =  new FileOutput();
		}
		/* Unspecified - system output by default */
		else
		{
			validationOutput =  new SystemOutput();
		}
		return validationOutput;
	}


	protected AbstractValidationOutput createEclipseOutput() {
		return getEclipseValidationOutputFactory().createEclipseOutput();
	}


	public static IValidationOutputFactory getEclipseValidationOutputFactory() {
		if (validationOutputFactory == null) {
			return defaultValidationOutputFactory;
		}
		return validationOutputFactory;
	}
	
	public static void setEclipseValidationOutputFactory(IValidationOutputFactory factory) {
		validationOutputFactory = factory;
	}


	/**
	 * Returns the attributes
	 * 
	 * @return attributes
	 */
	public Map<String, Object> getAttributes()
	{
		return attributes;
	}
	
	
	/**
	 * The main method - Used to set the appropriate attributes based on the
	 * program arguments.  Expected format:
	 * org.eclipse.cosmos.rm.internal.validation.core.MainValidator ([option][option argument(s)])* <comma separated path(s) to SML-IF/SML document(s)>,
	 * where available option and arguments are:
	 * <ul>
	 * 	<li> -xml <a fully qualified class name> </li>
	 * 	<li> -sml <fully qualified class names separated by a comma> </li>
	 * 	<li> -schematron <a fully qualified class name> </li>
	 *  <li> -output <path to the output file> </li>
	 *  <li> -context <the context directory of the repository>  (the current working
	 *  director will be used as the context directory if this option is not specified)</li>
	 *  <li> -configuration <the path to the configuration file of the repository> (this
	 *  option takes precedence over the context option)</li>
	 * </ul>
	 * 
	 * @param args The program arguments
	 */
	public static void main (String[] args)
	{
		if (args.length <= 0)
		{
			printUsage();
			return;
		}
		
		int lastInx = -1, currentInx = 0;
		Map<String, Object> validationAttributes = new Hashtable<String, Object>();
		try
		{
			while (lastInx != currentInx)
			{
				lastInx = currentInx;
				currentInx = processOption(validationAttributes, args, currentInx);
			}
			
			
			if (currentInx != args.length - 1)
			{
				printUsage();
				return;
			}
			
			validationAttributes.put(IValidator.ATTRIBUTE_ENV, IValidator.VALUE_ENV_STANDALONE);
			if (System.getProperty("org.eclipse.cosmos.rm.repository") == null)
			{
				System.setProperty("org.eclipse.cosmos.rm.repository", FileSystemSMLRepository.class.getName());
			}
			
			ISMLRepository repository = SMLRepositoryFactory.createRepository();
			Map<String, Object> repositoryAttributes = new Hashtable<String, Object>();
			
			// The configuration file takes the highest precedence
			if (configurationFile != null)
			{
				repositoryAttributes.put(FileSystemSMLRepository.ATTRIBUTE_CONFIGURATION_PROPERTY, configurationFile);
			}
			// Use the context directory if the configuration file is not specified
			else if (contextDirectory != null)
			{
				repositoryAttributes.put(IFileSystemSMLProperties.ROOT_DIRECTORY, contextDirectory);
			}
			// Resort to the current working directory if the context directory is not specified
			else
			{
				repositoryAttributes.put(IFileSystemSMLProperties.ROOT_DIRECTORY, System.getProperty("user.dir"));
			}
			
			repository.connect(repositoryAttributes);
			ISMLOperation validationOperation = repository.getOperation(ISMLValidateOperation.ID);
			
			
			StringTokenizer resources = new StringTokenizer(args[currentInx], ",");
			StringBuffer smlModelUnits = new StringBuffer();
			
			if (suppressWarnings)
			{
				validationAttributes.put(IValidator.ATTRIBUTE_SUPPRESS_WARNINGS, Boolean.TRUE);
			}
			
			validationAttributes.put(IValidator.ATTRIBUTE_INPUT_TYPE, IValidator.VALUE_SML_IF);			
			while (resources.hasMoreTokens())
			{
				String currentResource = resources.nextToken();
				int documentType;
				try
				{
					String fullPath = currentResource;
					String contextDirectory = repository.getProperty(IFileSystemSMLProperties.ROOT_DIRECTORY, "");
					if (!currentResource.startsWith(contextDirectory))
					{
						fullPath = contextDirectory + (contextDirectory.endsWith("/") || contextDirectory.endsWith("\\") ? "" : "/") + 
									currentResource;
					}
					documentType = SMLValidatorUtil.identifyDocumentType(new File(fullPath));
					if (documentType == ISMLConstants.TYPE_SMLIF)
					{
						validationAttributes.put(IValidator.ATTRIBUTE_INSTANCE, fullPath);		
						validationOperation.setArguments(new Object[]{validationAttributes});
						validationOperation.run();		
					}
					else if (documentType != ISMLConstants.TYPE_UNKNOWN)
					{
						smlModelUnits.append(currentResource).append(",");
					}
				} 
				catch (IOException e)
				{					
					e.printStackTrace();
				} 
				catch (CoreException e)
				{					
					e.printStackTrace();
				}				
			}
			
			if (smlModelUnits.length() > 0)
			{
				smlModelUnits.deleteCharAt(smlModelUnits.length() - 1);
				validationAttributes.put(IValidator.ATTRIBUTE_INPUT_TYPE, IValidator.VALUE_SML_UNIT);
				validationAttributes.put(IValidator.ATTRIBUTE_INSTANCE, smlModelUnits.toString());
				validationOperation.setArguments(new Object[]{validationAttributes});
				validationOperation.run();		
			}
				
		}
		catch (IllegalArgumentException e)
		{
			printUsage();
		} 
		catch (MissingRepositoryException e)
		{			
			e.printStackTrace();
		} 
		catch (RepositoryConnectionException e)
		{		
			e.printStackTrace();
		} 
		catch (RepositoryOperationException e)
		{
			e.printStackTrace();
		}		
	}


	private static int processOption(Map<String, Object> attributes, String[] args, int inx) throws IllegalArgumentException
	{
		if (inx < 0 || inx >= args.length)
			return inx;
		
		String currentOption = args[inx];
		
		if (OPTION_SCHEMATRON.equals(currentOption))
		{	
			inx = addAttribute(attributes, args, ++inx, IValidator.ATTRIBUTE_VALIDATION_SCHEMATRON);			
		}
		else if (OPTION_SML.equals(currentOption))
		{
			List<String> smlValidators = new ArrayList<String>();
			while (++inx < args.length - 1 && !args[inx].startsWith("-"))
			{
				StringTokenizer st = new StringTokenizer(args[inx]);
				while (st.hasMoreTokens())
				{
					smlValidators.add(st.nextToken());
				}
			}
			
			if (smlValidators.isEmpty())
				throw new IllegalArgumentException();
			
			attributes.put(IValidator.ATTRIBUTE_VALIDATION_SML, smlValidators);
		}
		else if (OPTION_FILE.equals(currentOption))
		{
			inx = addAttribute(attributes, args, ++inx, IValidator.ATTRIBUTE_FILE_OUTPUT);						
		}
		else if (OPTION_CONTEXT.equals(currentOption))
		{
			contextDirectory = args[++inx];
			inx++;
		}
		else if (OPTION_CONFIGURATION.equals(currentOption))
		{
			configurationFile = args[++inx];
			inx++;
		}
		else if (OPTION_SUPPRESS_WARNINGS.equals(currentOption))
		{
			suppressWarnings = true;
			inx++;
		}
		
		return inx;
	}


	private static int addAttribute(Map<String, Object> attributes, String[] args, int inx, String attribute) throws IllegalArgumentException
	{
		if (inx >= args.length || args[inx].startsWith("-"))
			throw new IllegalArgumentException();
		
		attributes.put(attribute, args[inx]);
		return inx + 1;
	}


	private static void printUsage()
	{
		System.err.println(SMLValidationMessages.errorIllegalArguments);
	}


	public void validationErrorOccurred(ValidationEvent event) {
		event.continueOnError = !fastFailPolicy();
	}
}
