/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.rm.internal.smlif.editor;


import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;

import javax.xml.parsers.ParserConfigurationException;

import org.eclipse.core.filesystem.URIUtil;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.cosmos.rm.internal.smlif.SMLPlugin;
import org.eclipse.cosmos.rm.internal.smlif.common.SMLCommonUtil;
import org.eclipse.cosmos.rm.internal.smlif.common.SMLMessages;
import org.eclipse.cosmos.rm.internal.validation.common.ISMLConstants;
import org.eclipse.cosmos.rm.internal.validation.common.IValidationConstants;
import org.eclipse.cosmos.rm.internal.validation.common.XMLInternalUtility;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.IPageLayout;
import org.eclipse.ui.IPartListener2;
import org.eclipse.ui.IPathEditorInput;
import org.eclipse.ui.IURIEditorInput;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IWindowListener;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchPartReference;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.forms.editor.FormEditor;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

/**
 * The SML-IF editor part used to create three sub-pages: overview,
 * documents, and source (raw SML-IF)
 * 
 * @author Ali Mehregani
 * @modified David Whiteman
 */
public class SMLIFEditor extends FormEditor implements IResourceChangeListener
{
	/**
	 * The id of the SML-IF editor
	 */
	private static final String ID = "org.eclipse.cosmos.rm.internal.smlif.editor.smlifEditor"; //$NON-NLS-1$
	
	
	public static final int TYPE_IDENTITY = 0;
	public static final int BASE_IDENTITY = 4; //xml:base identity attribute

	/**
	 * The index of the source page
	 */
	private static final int SMLIF_TEXT_INDEX = 2;
	
	/**
	 * The overview page
	 */
	private OverviewPage overviewPage;

	/**
	 * The documents pages
	 */
	private DocumentsPage documentsPage;

	
	/**
	 * The root DOM node of the SML-IF file selected 
	 */
	private Node rootNode;

	/**
	 * The identity node
	 */
	private Node identityNode;

	
	/**
	 * Indicates whether an error has been reported
	 */
	private boolean reportedError;
	
	/**
	 * An extended toolkit
	 */
	private SMLFormToolkit extendedToolkit;
	
	/**
	 * The form pages contained by this editor
	 */
	private AbstractFormPage[] formPages;

	/**
	 * The DOM document representing the SML-IF content
	 */
	private Document document;
	
	/**
	 * The form part shared amongst the form pages
	 */
	private SMLIFFormPart smlifFormPart;
	
	/**
	 * Indicates whether the underlying file has been changed and that it
	 * needs to be reloaded
	 */
	private boolean needsReload;
	
	/**
	 * Disables the resource listener if set to false
	 */
	private boolean enableResourceListener;
	
	/**
	 * The last modified date of the underlying file
	 */
	private long lastModifiedDate;
	
	/**
	 * The activate listener
	 */
	private static ActivateListener activateListener;
	
	/**
	 * Last window
	 */
	private static IWorkbenchWindow lastWindow;
	
	/**
	 * The workspace file associated with the editor input
	 */
	private IFile workspaceFile;
	
	/**
	 * The local file associated with the editor input (doesn't
	 * necessarily have to be under the workspace)
	 */
	private File localFile;

	/**
	 * The file name corresponding to the editor input
	 */
	private String fileName;

	/**
	 * The SML-IF text page
	 */
	private SMLIFTextPage smlIFSourcePage;

	/**
	 * ID of the Eclipse search view
	 */
	private static final String ID_ECLIPSE_SEARCH_VIEW = "org.eclipse.search.ui.views.SearchView"; //$NON-NLS-1$
		
	/**
	 * Create the multi-page editor for SML-IF documents
	 */
	public SMLIFEditor() 
	{
		super();
		ResourcesPlugin.getWorkspace().addResourceChangeListener(this, IResourceChangeEvent.PRE_CLOSE | IResourceChangeEvent.POST_CHANGE);		
		extendedToolkit = new SMLFormToolkit (SMLPlugin.getDefault().getWorkbench().getDisplay());
		smlifFormPart = new SMLIFFormPart(this);
		enableResourceListener = true;		
	}

	
	/**
	 * @see org.eclipse.ui.part.MultiPageEditorPart#init(org.eclipse.ui.IEditorSite, org.eclipse.ui.IEditorInput)
	 */
	public void init(IEditorSite site, IEditorInput input) throws PartInitException
	{		
		workspaceFile = null;
		localFile = null;
		
		if (input instanceof IFileEditorInput) {
			workspaceFile = ((IFileEditorInput)input).getFile();
		} else if (input instanceof IPathEditorInput) {
			localFile = new File (((IPathEditorInput)input).getPath().toString());
		} else if (input instanceof IURIEditorInput) {
			URI uri = ((IURIEditorInput)input).getURI();
			if (uri.isAbsolute()) {
				IPath path = URIUtil.toPath(((IURIEditorInput)input).getURI());
				localFile = path.toFile();
			}
		}
		
		if (workspaceFile == null && localFile == null) {
			throw new PartInitException (SMLMessages.errorEditorWrongInput);
		}
		super.init(site, input);
		fileName = workspaceFile == null ? localFile.getName() : workspaceFile.getName();
		setPartName(fileName);
		
		if (site != null && lastWindow != site.getWorkbenchWindow())
		{
			if (activateListener != null)
				activateListener.dispose();
			activateListener = new ActivateListener(site);
		}
	}

	/**
	 * @see org.eclipse.ui.forms.editor.FormEditor#addPages()
	 */
	protected void addPages() 
	{
		overviewPage = new OverviewPage(this);
		documentsPage = new DocumentsPage(this);
		formPages = new AbstractFormPage[]{overviewPage, documentsPage};
		
		try
		{
			for (int i = 0; i < formPages.length; i++)
			{
				addPage(formPages[i], getEditorInput());
			}
			
			smlIFSourcePage = new SMLIFTextPage();
			addPage(smlIFSourcePage, getEditorInput());
			setPageText(formPages.length, SMLMessages.editorSourceTitle);
			getSite().setSelectionProvider(null);
		} 
		catch (PartInitException e)
		{
			SMLCommonUtil.openErrorWithDetail(SMLMessages.errorEditorTitle, SMLMessages.errorEditorPages, e);
		}
		
//		PlatformUI.getWorkbench().getHelpSystem().setHelp(this.getContainer(), "org.eclipse.cosmos.rm.smlif.editor");

	}
	
	protected void pageChange(int newPageIndex)
	{		
		if (newPageIndex == SMLIF_TEXT_INDEX)
		{
			smlIFSourcePage.activated();
		}
		else
		{
			smlIFSourcePage.deactivated();
		}
		super.pageChange(newPageIndex);
	}
	
	/**
	 * @see org.eclipse.ui.part.MultiPageEditorPart#dispose()
	 */
	public void dispose() 
	{
		ResourcesPlugin.getWorkspace().removeResourceChangeListener(this);
		activateListener.dispose();
		super.dispose();
	}
	
	
	/**
	 * @see org.eclipse.ui.part.EditorPart#doSave(org.eclipse.core.runtime.IProgressMonitor)
	 */
	public void doSave(IProgressMonitor monitor) 
	{	
		enableResourceListener = false;
		smlifFormPart.commit(true);		
		if (!isDirty())
			editorDirtyStateChanged();

		for (int i = 0; i < formPages.length; i++)
		{
			formPages[i].doSave(monitor);
		}
		smlIFSourcePage.doSave(monitor);
		enableResourceListener = true;
	}
	
	
	/**
	 * @see org.eclipse.ui.part.EditorPart#doSaveAs()
	 */
	public void doSaveAs() 
	{
	}
	

	/**
	 * @see org.eclipse.ui.part.EditorPart#isSaveAsAllowed()
	 */
	public boolean isSaveAsAllowed() 
	{
		return true;
	}

	
	/**
	 * Closes all project files on project close.
	 */
	public void resourceChanged(final IResourceChangeEvent event)
	{
		if (!enableResourceListener || workspaceFile == null)
			return;
		
		switch (event.getType())
		{
			case IResourceChangeEvent.PRE_CLOSE:
			{				
				closeAllEditors(event);
				break;
			}
			case IResourceChangeEvent.POST_CHANGE:
			{
				IEditorInput input = getEditorInput();
				if (input == null)
					return;
				
				String inputPath = workspaceFile.getFullPath().toString();
				if (inputPath == null || inputPath.length() <= 0)
					return;
				
				IResourceDelta[] affectedChildren = event.getDelta().getAffectedChildren();
				List<String> affectedResourcePaths = new ArrayList<String>();
				findAffectedResourcePaths(affectedChildren, affectedResourcePaths);
				
				for (int i = 0, resourceCount = affectedResourcePaths.size(); i < resourceCount; i++)
				{
					if (inputPath.equals(affectedResourcePaths.get(i)))
					{
						needsReload = true;
						break;
					}
				}				
				break;
			}
		}
	}


	/**
	 * Close all open editors.  If the resourceChangeEvent is not null, then
	 * only close the editor if the resource changed is the parent project.
	 * 
	 * @param resourceChangeEvent optional event that indicates relevant project
	 */
	private void closeAllEditors(final IResourceChangeEvent resourceChangeEvent) {
		Runnable closeEditors = new Runnable()
		{
			public void run()
			{
				IWorkbenchPage[] pages = getSite().getWorkbenchWindow().getPages();
				for (int i = 0; i<pages.length; i++)
				{
					if (resourceChangeEvent != null) {
						if(!(workspaceFile.getProject().equals(resourceChangeEvent.getResource()))) {
							continue;
						}
					}
					IEditorPart editorPart = pages[i].findEditor(getEditorInput());
					pages[i].closeEditor(editorPart,true);
				}
			}    
		};
		
		Display.getDefault().asyncExec(closeEditors);
	}


	private void findAffectedResourcePaths(IResourceDelta[] affectedChildren, List<String> list)
	{		
		for (int i = 0; i < affectedChildren.length; i++)
		{
			IResourceDelta[] affected = affectedChildren[i].getAffectedChildren();
			if (affected == null || affected.length <= 0)
			{
				list.add(affectedChildren[i].getFullPath().toString());
			}
			else
			{
				findAffectedResourcePaths(affected, list);
			}
		}		
	}


	public String retrieveField(int type, String name)
	{
		Node rootDocumentNode = getDocumentNode();
		if (rootDocumentNode == null) {
			return null;
		}
		
		/* The identity type */
		if (type == TYPE_IDENTITY)
		{
			identityNode = identityNode == null ? SMLCommonUtil.retrieveChildNode(rootDocumentNode, ISMLConstants.SMLIF_URI, ISMLConstants.IDENTITY_ELEMENT) : identityNode;			
			return SMLCommonUtil.retrieveTextNode(SMLCommonUtil.retrieveChildNode(identityNode, ISMLConstants.SMLIF_URI, name));
		}
//		else if (type == BASE_IDENTITY)
//		{
//			identityNode = identityNode == null ? SMLCommonUtil.retrieveChildNode(rootDocumentNode, ISMLConstants.SMLIF_URI, ISMLConstants.IDENTITY_ELEMENT) : identityNode;						
//			return SMLCommonUtil.retrieveAttributeNode(identityNode, ISMLConstants.SMLIF_URI, ISMLConstants.IDENTITY_ELEMENT);			
//		}
		
		return null;
	}
	
	/**
	 * Equivalent to calling retrieveChildNode(Node parent, String uri, String localName),
	 * where parent is the root node of the associated document
	 * 
	 * @param uri The URI
	 * @param localName The localName of the element to be retrieved
	 */
	public Node retrieveChildNode(String uri, String localName)
	{
		return SMLCommonUtil.retrieveChildNode(getDocumentNode(), uri, localName);
	}
	
	/**
	 * Parse and return the DOM document of the SML-IF document
	 * 
	 * @return The DOM document node of the SML-IF file
	 */
	protected Node getDocumentNode()
	{
		if (rootNode != null)
			return rootNode;
		
		if (getEditorInput() == null)
			return null;
					
		if (workspaceFile == null && localFile == null)
			return null;

		if ((workspaceFile != null) && (!workspaceFile.exists())) {
			// The workspace file disappeared out from under us
			
			closeAllEditors(null);
			return null;
		}

		InputStream is = null;
		Exception exception = null;
		int lineNumber = -1;
		try
		{
			is = workspaceFile == null ? new FileInputStream(localFile) : workspaceFile.getContents();
			document = XMLInternalUtility.domParseDocument(is, false, false);
			NodeList children = document.getChildNodes();			
			for (int i = 0, childCount = children.getLength(); i < childCount; i++)
			{
				if (children.item(i).getNodeType() == Node.ELEMENT_NODE)
				{
					rootNode = children.item(i);
					break;
				}
			}
			if (rootNode == null) {
				throw new SMLIFEditorException(SMLMessages.errorNoRootNode);
			}
			if (!ISMLConstants.SMLIF_URI.equals(rootNode.getNamespaceURI())) {
				throw new SMLIFEditorException(SMLMessages.errorUnsupportedNamespace);
			}
			if (!ISMLConstants.MODEL_ELEMENT.equals(rootNode.getLocalName())) {
				throw new SMLIFEditorException(SMLMessages.errorWrongRootNode);
			}
						
			return rootNode;
			
		} 
		catch (CoreException e)
		{
			exception = e;
		} 
		catch (ParserConfigurationException e)
		{
			exception = e;
		} 
		catch (SAXParseException e)
		{
			exception = e;
			lineNumber = e.getLineNumber();
		}
		catch (SAXException e)
		{
			exception = e;
		} 
		catch (IOException e)
		{
			exception = e;
		}		
		catch (SMLIFEditorException e)
		{
			exception = e;
		}
		finally
		{			
			try
			{
				if (is != null)
					is.close();
			} catch (IOException e)
			{
				/* Ignore the exception */
			}
		}
		
		if (exception != null && !reportedError)
		{
			reportedError = true;
			SMLCommonUtil.openErrorWithDetail(SMLMessages.errorOpeningTitle, 
					SMLMessages.errorOpening + (lineNumber >= 0 ? IValidationConstants.LINE_SEPARATOR + 
					NLS.bind(SMLMessages.errorOpeningFileLine, String.valueOf(lineNumber)) : IValidationConstants.EMPTY_STRING), 
					exception);
			
			for (int i = 0; i < formPages.length; i++)
			{
				formPages[i].setEnabled(false);
			}
		}
		
		return null;
	}

	public Document getDocument()
	{
		if (document == null)
			getDocumentNode();
		return document;
	}
	
	
	/**
	 * @return the extendedToolkit
	 */
	public SMLFormToolkit getExtendedToolkit()
	{
		return extendedToolkit;
	}


	/**
	 * @return the smlifFormPart
	 */
	public SMLIFFormPart getSmlifFormPart()
	{
		return smlifFormPart;
	}
	
	public SMLIFTextPage getSmlifSourcePage()
	{
		return smlIFSourcePage;
	}
	
	
	/**
	 * The activate listener registered for updating the content of the editor if the
	 * underlying file has changed by another eclipse-based editor or outside Eclipse
	 * 
	 * @author Ali Mehregani
	 */
	private class ActivateListener implements IPartListener2, IWindowListener
	{
		
		public ActivateListener(IEditorSite site)
		{
			site.getWorkbenchWindow().getPartService().addPartListener((IPartListener2) this);
			PlatformUI.getWorkbench().addWindowListener(this);
		}
		
		
		public void dispose()
		{
			getEditorSite().getWorkbenchWindow().getPartService().removePartListener((IPartListener2) this);
			PlatformUI.getWorkbench().removeWindowListener(this);
		}
		
		// Make the raw XML page the active page
		private void turnToTextPage() {
			SMLIFEditor.this.setActivePage(SMLIF_TEXT_INDEX);
		}

		/**
		 * @see org.eclipse.ui.IWindowListener#windowActivated(org.eclipse.ui.IWorkbenchWindow)
		 */
		public void windowActivated(IWorkbenchWindow window)
		{
			handleActivation();
		}


		private void handleActivation()
		{
			long currentModifiedDate = 0;
			if (workspaceFile == null)
				return;
			
			currentModifiedDate = new File(workspaceFile.getRawLocation().toString()).lastModified();
			if (currentModifiedDate != 0)
			{
				if (currentModifiedDate != lastModifiedDate && !workspaceFile.isSynchronized(IResource.DEPTH_ZERO))
				{
					lastModifiedDate = currentModifiedDate;
																			
					// Do not use default parent shell, because it causes an infinite loop with activation of ths window
					if (!needsReload && !MessageDialog.openQuestion(null, 
							SMLMessages.editorFileChangedTitle, SMLMessages.editorFileChanged))
					{					
						return;
					}
					
					try
					{
						workspaceFile.refreshLocal(IResource.DEPTH_ZERO, null);
					} 
					catch (CoreException e)
					{
						// Do not use default parent shell, because it causes an infinite loop with activation of ths window
						SMLCommonUtil.openErrorWithDetail(SMLMessages.errorSynchronizeTitle, SMLMessages.errorSynchronize, e, false);
						return;
					}
					
					needsReload = true;
				}
			}									
			
			if (needsReload)
			{
				rootNode = null; 
				identityNode = null;
				document = null;
						
				for (int i = 0; i < formPages.length; i++)
				{
					formPages[i].updateContent();
				}
			}
			needsReload = false;
		}


		/**
		 * @see org.eclipse.ui.IWindowListener#windowClosed(org.eclipse.ui.IWorkbenchWindow)
		 */
		public void windowClosed(IWorkbenchWindow window)
		{
		}

		/**
		 * @see org.eclipse.ui.IWindowListener#windowDeactivated(org.eclipse.ui.IWorkbenchWindow)
		 */
		public void windowDeactivated(IWorkbenchWindow window)
		{
		}

		/**
		 * @see org.eclipse.ui.IWindowListener#windowOpened(org.eclipse.ui.IWorkbenchWindow)
		 */
		public void windowOpened(IWorkbenchWindow window)
		{
		}

		public void partActivated(IWorkbenchPartReference partRef) 
		{
			IWorkbenchPage activePage = SMLPlugin.getDefault().getWorkbench().getActiveWorkbenchWindow().getActivePage();
			if ((activePage == null) || !ID.equals(activePage.getActivePart().getSite().getId()))
				return;
			
			handleActivation();
			
			
			// If I double click on a relevant problem in the problems view,
			// and this editor is already open, I want to turn to the editor page
			if (IPageLayout.ID_PROBLEM_VIEW.equals(partRef.getId())) 
			{
				turnToTextPage();
			}
		}

		public void partBroughtToTop(IWorkbenchPartReference partRef) 
		{
		}

		public void partClosed(IWorkbenchPartReference partRef) 
		{
		}

		public void partDeactivated(IWorkbenchPartReference partRef) 
		{
		}

		public void partHidden(IWorkbenchPartReference partRef) 
		{
		}

		public void partInputChanged(IWorkbenchPartReference partRef) 
		{
		}

		public void partOpened(IWorkbenchPartReference partRef) 
		{
			// This is being opened from another view
			// If it's the problems or search view, we'll turn to the source page
			IWorkbenchPart workbenchPart = partRef.getPage().getActivePart(); 
			if (!(workbenchPart instanceof IViewPart)) return;
			String partId = ((IViewPart)workbenchPart).getViewSite().getId();
			if (IPageLayout.ID_PROBLEM_VIEW.equals(partId) || SMLIFEditor.ID_ECLIPSE_SEARCH_VIEW.equals(partId)) {
				turnToTextPage();
			}
		}

		public void partVisible(IWorkbenchPartReference partRef) 
		{
		}
		
	}
}
