/**********************************************************************
 * Copyright (c) 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.artifacts;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.xerces.xs.XSComplexTypeDefinition;

/**
 * Keys references of acyclic elements based on their type.
 * For example, assuming &lt;optionCourse&gt; is of type optionalCourseType,
 * then the following entry: <br/> <br/>
 * 
 * <pre>
 * &lt;optionalCourse&gt;
 *  &lt;sml:uri&gt;Course1&lt;/sml:uri&gt;
 * &lt;/optionalCourse&gt;
 * </pre>
 * 
 * is stored as: [optionalCourseType, Course1] 
 * 
 * <p>
 * It's expected that if an instance is  acyclic by derivation, then the most 
 * senior acyclic parent is used as the key of the reference.  For example, 
 * assuming the following inheritance graduateCourseType &gt; secondYearCourseType 
 * &gt; optionalCourseType, then any reference contained in an instance of 
 * graduateCourseType should be stored using optionalCourseType.
 * </p> 
 * 
 * @author Ali Mehregani
 */
public class AcyclicStructure
{
	/**
	 * Stores the acyclic element entries
	 * key = uri+localname of the acyclic type
	 * value = a list of acyclic elements
	 */
	private Map<String, List<AcyclicEntry>> entries;
	
	
	/**
	 * Constructor
	 */
	public AcyclicStructure()
	{
		entries = new Hashtable<String, List<AcyclicEntry>>();	
	}
	
	
	/**
	 * Add an entry to this structure
	 * 
	 * @param type The associated type of the acyclic entry
	 * @param acyclicEntry The acyclic entry
	 */
	public void add(XSComplexTypeDefinition type, AcyclicEntry acyclicEntry)
	{	
		String typeQName = type.getNamespace() + "/" + type.getName();
		List<AcyclicEntry> acyclicEntries = entries.get(typeQName);
		if (acyclicEntries == null)
		{
			acyclicEntries = new ArrayList<AcyclicEntry>();			
			entries.put(typeQName, acyclicEntries);
		}
		
		acyclicEntries.add(acyclicEntry);
	}
	
	
	/**
	 * Retrieve the keys of this data structure.  The keys corresponds
	 * to the qualified names (i.e. uri/localName) of acyclic types
	 * 
	 * @return The keys of this structure
	 */
	public String[] keys()
	{
		Iterator<String> keySet = entries.keySet().iterator();
		String[] keys = new String[entries.size()];
		int i = 0;
		while (keySet.hasNext())
		{
			keys[i++] = keySet.next();
		}
		
		return keys;
	}


	/**
	 * Retrieve the value of based on the key passed in
	 * 
	 * @param key The key to lookup by
	 * @return The value associated with key
	 */
	public AcyclicEntry[] get(String key)
	{
		List<AcyclicEntry> list = entries.get(key);
		return list == null ? null : list.toArray(new AcyclicEntry[list.size()]);
	}
	
	
	/**
	 * Retrieve all acyclic entries that belong to type CT and have
	 * are contained in a document with the alias passed in
	 * 
	 * @param alias The alias of the owning document
	 * @param type The acyclic type
	 * @return The acyclic entries based on the type and alias passed in
	 */
	public AcyclicEntry[] get(XSComplexTypeDefinition type, String alias)
	{
		AcyclicEntry[] entries = get(type.getNamespace() + "/" + type.getName());
		if (entries == null)
		{
			return new AcyclicEntry[0];
		}
		
		List<AcyclicEntry> foundEntries = new ArrayList<AcyclicEntry>();
		for (int i = 0; i < entries.length; i++)
		{
			String[] aliases = entries[i].getAliases();
			for (int j = 0; j < aliases.length; j++)
			{
				if (alias.equals(aliases[j]))
				{
					foundEntries.add(entries[i]); 
					break;
				}
			}
		}
		
		return foundEntries.toArray(new AcyclicEntry[foundEntries.size()]);
	}

	
	/**
	 * Stores information about the acyclic entry
	 * 
	 * @author Ali Mehregani
	 */
	public static class AcyclicEntry
	{
		/**
		 * The URI of the element is an acyclic entry
		 */
		private String uri;
		
		/**
		 * The local name of the element that is an acyclic entry
		 */
		private String localName;
		
		/**
		 * The reference of the acyclic entry
		 */
		private String reference;
		
		/**
		 * The type of this acyclic entry.  The most senior parent
		 * is always selected as the type.  For example if R2 is
		 * an extension of R1 and both have acyclic = true, then instances
		 * of R2 will have their type set to R1.
		 */
		private XSComplexTypeDefinition type;
		
		/**
		 * The alias of the document owning this acyclic entry.  If
		 * the document happens to have more than one alias, then the
		 * first one is selected
		 */
		private String ownerAlias;
		
		/**
		 * The line number for this acyclic entry
		 */
		private int lineNumber;
		
		/**
		 * The aliases associated with this acyclic entry
		 */
		private String[] aliases;
		
		/**
		 * The indices leading to the context node
		 */
		private int[] nodePath;
		
		/**
		 * The orphan index - valid if this acyclic entry
		 * is contained in a document without an alias 
		 */
		private int documentPosition;
		
		/**
		 * @return the uri
		 */
		public String getURI()
		{
			return uri;
		}
			

		/**
		 * @param uri the uri to set
		 */
		public void setURI(String uri)
		{
			this.uri = uri;
		}
		
		/**
		 * @return the localName
		 */
		public String getLocalName()
		{
			return localName;
		}
		
		/**
		 * @param localName the localName to set
		 */
		public void setLocalName(String localName)
		{
			this.localName = localName;
		}
		
		/**
		 * @return the reference
		 */
		public String getReference()
		{
			return reference;
		}
		
		/**
		 * @param reference the reference to set
		 */
		public void setReference(String reference)
		{
			this.reference = reference;
		}
		
		/**
		 * @return the type
		 */
		public XSComplexTypeDefinition getType()
		{
			return type;
		}
		
		/**
		 * @param type the type to set
		 */
		public void setType(XSComplexTypeDefinition type)
		{
			this.type = type;
		}
		
		/**
		 * @return the ownerAlias
		 */
		public String getOwnerAlias()
		{
			return ownerAlias;
		}

		/**
		 * @return the lineNumber
		 */
		public int getLineNumber()
		{
			return lineNumber;
		}

		/**
		 * @param lineNumber the lineNumber to set
		 */
		public void setLineNumber(int lineNumber)
		{
			this.lineNumber = lineNumber;
		}


		/**
		 * @return the aliases
		 */
		public String[] getAliases()
		{
			return aliases;
		}


		/**
		 * @param aliases the aliases to set
		 */
		public void setAliases(String[] aliases)
		{
			this.aliases = aliases;
			this.ownerAlias = aliases == null || aliases.length <= 0 ? null : aliases[0];
		}


		/**
		 * @return the nodePath
		 */
		public int[] getNodePath()
		{
			return nodePath;
		}


		/**
		 * @param nodePath the nodePath to set
		 */
		public void setNodePath(int[] nodePath)
		{
			this.nodePath = nodePath;
		}


		/**
		 * @return the documentPosition
		 */
		public int getDocumentPosition()
		{
			return documentPosition;
		}


		/**
		 * @param documentPosition the documentPosition to set
		 */
		public void setDocumentPosition(int documentPosition)
		{
			this.documentPosition = documentPosition;
		}
	}
}
