/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.common;

import java.io.IOException;
import java.io.InputStream;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

/**
 * An internal utility class
 * 
 * @author Ali Mehregani
 */
public class XMLInternalUtility
{
	private static Document document;
	
	private static Document getDocument()
	{
		if (document == null)
		{
			try
			{
				document = DocumentBuilderFactory.newInstance().newDocumentBuilder().newDocument();
			}
			catch (ParserConfigurationException e)
			{				
				// Ignore
				e.printStackTrace();
			}
		}
		
		return document;
	}
	
	public static Node createElement(String namespaceURI, String qualifiedName)
	{
		return createElement(namespaceURI, qualifiedName, null);
	}
	
	public static Node createElement(String namespaceURI, String qualifiedName, String[][] attributes)
	{
		Document doc = getDocument();
		if (doc == null)
		{
			return null;
		}
		
		Element node = doc.createElementNS(namespaceURI, qualifiedName);
		if (attributes != null)
		{
			for (int i = 0; i < attributes.length; i++)
			{
				node.setAttribute(attributes[i][0], attributes[i][1]);
			}
		}
		
		return node;
	}
	
	public static Node appendNode (Node parent, Node child)
	{
		if (parent == null || child == null)
		{
			return null;
		}
		
		parent.appendChild(child);
		return parent;
	}
	
	public static Node createTextNode (String value)
	{
		Document doc = getDocument();
		if (doc == null)
		{
			return null;
		}
		return doc.createTextNode(value);
	}

	public static Document domParseDocument(InputStream resource) throws ParserConfigurationException, SAXException, IOException 
	{
		return domParseDocument(resource, true, true);
	}

	/**
	 * Uses a DOM parser to parse the XML file passed in
	 * 
	 * @param resource The resource to be parsed
	 * @return The document representing the parsed resource
	 */
	public static Document domParseDocument(InputStream resource, boolean ignoreComments, boolean ignoreWhitespace) throws ParserConfigurationException, SAXException, IOException 
	{
		DocumentBuilderFactory docBuilderFactory = DocumentBuilderFactory.newInstance();
	    docBuilderFactory.setNamespaceAware(true);
	    docBuilderFactory.setIgnoringElementContentWhitespace(ignoreWhitespace);
	    docBuilderFactory.setIgnoringComments(ignoreComments);
	    DocumentBuilder docBuilder = docBuilderFactory.newDocumentBuilder();
	    
		return docBuilder.parse(resource);
	}

	/**
	 * Equivalent to serializeNode(buffer, node, -1, false)
	 */
	public static void serializeNode(StringBuffer buffer, Node node)
	{
		serializeNode(buffer, node, -1, false);
	}

	/**
	 * Equivalent to serializeNode(buffer, node, indent, null, null)
	 */
	public static void serializeNode(StringBuffer buffer, Node node, int indent, boolean autoFormat)
	{
		serializeNode(buffer, node, indent, null, null, autoFormat);
	}

	/**
	 * Used to serialize a node into string
	 * 
	 * @param buffer The buffer that the string will be written to
	 * @param node The starting node
	 * @param indent The starting indent level (use 0 if starting at root)
	 * @param currentIndex The current index of the node (use null if not interested to store indices)
	 * @param indices The indices map (currentIndex must not be null if indices is used)
	 * @param autoFormat Indicates whether the XML content should be formatted
	 */
	public static void serializeNode(StringBuffer buffer, Node node, int indent, String currentIndex, Map<String, int[]> indices, boolean autoFormat)
	{		
		if (node == null)
			return;
		
		switch (node.getNodeType())
		{		
			case Node.TEXT_NODE:
			{
				if (!autoFormat)
				{
					buffer.append(node.getNodeValue());
					break;
				}
				
				StringBuffer textValueBuffer = new StringBuffer();
				String textValue = node.getNodeValue();
				StringTokenizer st = new StringTokenizer(textValue, "\n");
				
				while(st.hasMoreTokens())
				{
					String currentToken = st.nextToken().trim();
					if (currentToken.length() > 0)
					{
						textValueBuffer.append(ISMLConstants.nl);							
						addIndent(textValueBuffer, indent);
						textValueBuffer.append(currentToken);
					}
				}
				
				buffer.append(textValueBuffer);
				break;
			}
			case Node.COMMENT_NODE:
			{
				if (node.getNodeValue() == null)
					return;
				
				addIndent(buffer, indent);				
				buffer.append("<!--" + node.getNodeValue() + "-->");
				buffer.append(ISMLConstants.nl);
				
				break;
			}			
			case Node.ELEMENT_NODE:
			{
				if (autoFormat)
				{
					boolean needsLineSep = false;
					for (int i = buffer.length() - 1; i >= 0; i--)
					{
						char currentChar = buffer.charAt(i);
						if (currentChar == '\n')
							break;
						else if (currentChar == '\t' || currentChar == ' ' || currentChar == '\r')
							continue;
						else
						{
							needsLineSep = true;
							break;
						}
					}
					
					if (needsLineSep)
						buffer.append(ISMLConstants.nl);
				}
				NamedNodeMap attributes = node.getAttributes();
				Node[] children = retrieveChildNodes(node);
				boolean hasAttributes = attributes != null && attributes.getLength() > 0;
				boolean hasChildren = children != null && children.length > 0;
				
				int[] elementIndices = new int[2];
				elementIndices[0] = buffer.length();	
				
				if (autoFormat)
				{
					addElement(buffer, indent, node.getNodeName(), false, false);
				}
				else
				{
					addIndent(buffer, indent);
					buffer.append(ISMLConstants.OPEN_ANGLE_BRACKET).append(node.getNodeName());
				}
				
				if (hasAttributes)
				{
					if (autoFormat)
						addAttribute(buffer, attributes, !hasChildren, true);
					else
					{
						for (int i = 0, attCount = attributes.getLength(); i < attCount; i++)
						{
							Node currentAttribute = attributes.item(i);
							buffer.append(ISMLConstants.SINGLE_SPACE + currentAttribute.getNodeName() + ISMLConstants.EQUAL_SIGN + ISMLConstants.DOUBLE_QUOTE+currentAttribute.getNodeValue()+ISMLConstants.DOUBLE_QUOTE);
						}
						buffer.append(hasChildren ? ISMLConstants.CLOSE_ANGLE_BRACKET : ISMLConstants.FORWARD_SLASH+ISMLConstants.CLOSE_ANGLE_BRACKET);
					}
				}
				else
				{
					if (hasChildren) {
						buffer.append(ISMLConstants.CLOSE_ANGLE_BRACKET);
					} else {
						buffer.append(ISMLConstants.FORWARD_SLASH).append(ISMLConstants.CLOSE_ANGLE_BRACKET);
					}
				}
				
				if (!hasChildren)
				{
					elementIndices[1] = buffer.length();
					if (currentIndex != null) {
						indices.put(currentIndex, elementIndices);
					}
					return;
				}
				
				int counter = -1;
				if (autoFormat)
				{
					for (int i = 0; i < children.length; i++)
					{
						Node child = children[i];
						if (isWhitespace(child))
						{
							continue;				
						}
						else
						{
							buffer.append(ISMLConstants.nl);
						}
						
						if (child.getNodeType() == Node.ELEMENT_NODE)
							counter++;
						
						serializeNode(buffer, child, indent + 1, currentIndex == null ? null : currentIndex + "," + counter, indices, autoFormat); //$NON-NLS-1$
					}
				}
				else
				{				
					NodeList children1 = node.getChildNodes();					
					for (int i=0, nodeCount=children1.getLength(); i<nodeCount; i++)
					{
						Node currentChild = children1.item(i);
						if (currentChild.getNodeType() == Node.ELEMENT_NODE)
							counter++;
						
						serializeNode(buffer, currentChild, 0, currentIndex == null ? null : currentIndex + "," + counter, indices, autoFormat); //$NON-NLS-1$
					}					
				}
				if (autoFormat)
				{
					buffer.append(ISMLConstants.nl);
					addElement(buffer, indent, node.getNodeName(), true, true);
				}
				else
				{
					buffer.append(ISMLConstants.OPEN_ANGLE_BRACKET).append(ISMLConstants.FORWARD_SLASH).append(node.getNodeName()).append(ISMLConstants.CLOSE_ANGLE_BRACKET);
				}
				elementIndices[1] = buffer.length();
				if (currentIndex != null)
					indices.put(currentIndex, elementIndices);
				break;
			}
			case Node.DOCUMENT_NODE:
			{
				buffer.append("<?xml version=\"1.0\" encoding=\"" + ISMLConstants.UTF_8 + "\"?>" + ISMLConstants.nl);				
				NodeList children = node.getChildNodes();
				if (children == null) {
					break;
				}
				
				for (int i = 0, childCount = children.getLength(); i < childCount; i++)
				{
					serializeNode(buffer, children.item(i), indent, currentIndex, indices, autoFormat);
				}
			}
			default: 
				break;
		}
		
	}

	/**
	 * Adds XML attributes to the string buffer passed in.  Attributes with a null value will
	 * not be written.
	 * 
	 * Pre-condition:
	 * <ul>
	 * 	<li> attributes and attributeValues must be valid (i.e. non-null) </li>
	 * 	<li> attributes and attributeValues length must be the same </li>
	 * </ul>
	 *  
	 * @param stringBuffer The string buffer that the attributes will be written to
	 * @param attributes attributes
	 * @param close If set, a forward slash is added after the attributes.  
	 * @param end If set, a closed angle bracket is added after the attributes
	 */
	public static void addAttribute(StringBuffer stringBuffer, NamedNodeMap attributes, boolean close, boolean end)
	{
		for (int i = 0, attCount = attributes.getLength(); i < attCount; i++)
		{
			Node currentAttribute = attributes.item(i);
			stringBuffer.append(ISMLConstants.SINGLE_SPACE);
			stringBuffer.append(currentAttribute.getNodeName());
			stringBuffer.append(ISMLConstants.EQUAL_SIGN);
			stringBuffer.append(ISMLConstants.DOUBLE_QUOTE);
			stringBuffer.append(currentAttribute.getNodeValue());
			stringBuffer.append(ISMLConstants.DOUBLE_QUOTE);
		}
		
		stringBuffer.append(close ? ISMLConstants.FORWARD_SLASH : ISMLConstants.EMPTY_STRING);
		stringBuffer.append(end ? ISMLConstants.CLOSE_ANGLE_BRACKET + (close ? ISMLConstants.nl : ISMLConstants.EMPTY_STRING) : ISMLConstants.EMPTY_STRING);
	}

	/**
	 * Adds an XML element to the string buffer passed in.
	 * 
	 * @param stringBuffer The string buffer that the element will be written to
	 * @param indent The number of indents that should be prefixed to the element written
	 * @param elementName The element name
	 * @param close Indicates whether this element should be closed or not.  If set, 
	 * a forward slash is added before theelement is ended.  
	 * @param end Indicates if the element should be ended with a closed angle bracket.  
	 * For example elements requring attributes are not ended.
	 */
	public static void addElement (StringBuffer stringBuffer, int indent, String elementName, boolean close, boolean end)
	{
		if (stringBuffer == null)
			return;
		
		if (indent > 0)
			addIndent(stringBuffer, indent);
		stringBuffer.append(ISMLConstants.OPEN_ANGLE_BRACKET);
		stringBuffer.append(close ? ISMLConstants.FORWARD_SLASH : ISMLConstants.EMPTY_STRING);		
		stringBuffer.append(elementName);
		stringBuffer.append(end ? ISMLConstants.CLOSE_ANGLE_BRACKET + 
				(close ? ISMLConstants.nl : ISMLConstants.EMPTY_STRING) : 
					ISMLConstants.EMPTY_STRING);		
	}

	public static void addIndent (StringBuffer stringBuffer, int indent)
	{
		for (int i = 0; i < indent; i++)
		{
			stringBuffer.append("\t");
		}
	}
	
	/**
	 * Add <tt>indent</tt> number of tabs to the string managed by
	 * the StringWriter
	 * 
	 * @param stringWriter
	 * @param indent
	 */
	public static void addIndent(StringWriter stringWriter, int indent) {
		for (int i = 0; i < indent; i++) {
			stringWriter.append('\t');
		}
	}
	

	private static Node[] retrieveChildNodes(Node node)
	{
		if (node == null) {
			return null;
		}
		
		NodeList children = node.getChildNodes();	
		List<Node> finalChildrenList = new ArrayList<Node>();
		for (int i = 0, childCount = children.getLength(); i < childCount; i++)
		{
			Node child = children.item(i);			
			if (!isWhitespace(child))
			{
				finalChildrenList.add(child);
			}
		}
		
		return (Node[])finalChildrenList.toArray(new Node[finalChildrenList.size()]);
	}

	public static boolean isWhitespace(Node node)
	{
		return node != null && node.getNodeType() == Node.TEXT_NODE && (node.getNodeValue() == null || node.getNodeValue().trim().length() <= 0);
	}
		
	
	public static class NodeXMLWritable //implements IXMLWritable
	{
		private List<Node> rootNode;
		public NodeXMLWritable(Node node)
		{
			this.rootNode = new ArrayList<Node>();
			addNode(node);
		}

		public void addNode (Node node)
		{
			if (node == null)
				return;
			
			this.rootNode.add(node);
		}
		
		
		public void toXML(StringWriter writer, int indentLevel)
		{			
			for (int i = 0, nodeCount = rootNode.size(); i < nodeCount; i++)
			{
				StringBuffer sb = new StringBuffer();
				sb.append(ISMLConstants.nl);
				XMLInternalUtility.serializeNode(sb, rootNode.get(i), indentLevel, false);
				sb.append(ISMLConstants.nl);
				writer.write(sb.toString());
			}
		}
	}
	
	
	public static class XMLWritableString {//implements IXMLWritable {

		private String value;
		
		public XMLWritableString(String value) {
			super();
			this.value = value;
		}

		public void toXML(StringWriter writer, int indentLevel) {
			addIndent(writer, indentLevel);
			writer.write(valueWithIndent(indentLevel+1)+ISMLConstants.nl);
		}
		
		private String valueWithIndent(int i) {
			String tempValue = value;
			StringWriter tabsWriter = new StringWriter();		
			tabsWriter.append('\n');
			addIndent(tabsWriter, i);				
			tempValue = tempValue.replace("\n", tabsWriter.toString());
			tempValue = tabsWriter.toString() + tempValue;
			return tempValue;
		}

		@Override
		public int hashCode() {
			final int prime = 31;
			int result = 1;
			result = prime * result + ((value == null) ? 0 : value.hashCode());
			return result;
		}

		@Override
		public boolean equals(Object obj) {
			if (this == obj)
				return true;
			if (obj == null)
				return false;
			if (getClass() != obj.getClass())
				return false;
			final XMLWritableString other = (XMLWritableString) obj;
			if (value == null) {
				if (other.value != null)
					return false;
			} else if (!equalsSpaceIgnore(value, other.value))
				return false;
			return true;
		}

		/**
		 * A crude comparison between the XML source found in the <record>
		 * element.  This is a simplistic implementation that doesn't consider
		 * spaces in the tag, etc.
		 * 
		 * @param string1
		 * @param string2
		 * @return
		 */
		private boolean equalsSpaceIgnore(String string1, String string2) {
			String cleanedString1 = string1.replaceAll("[\\t\\r\\n]", "");
			String cleanedString2 = string2.replaceAll("[\\t\\r\\n]", "");
			return cleanedString1.equals(cleanedString2);
		}

	}
}
