/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.databuilders;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.cosmos.rm.internal.validation.SMLActivator;
import org.eclipse.cosmos.rm.internal.validation.artifacts.ReferenceDescriptor;
import org.eclipse.cosmos.rm.internal.validation.common.ISMLConstants;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidatorUtil;
import org.eclipse.cosmos.rm.internal.validation.core.IFoundationBuilder;
import org.eclipse.cosmos.rm.internal.validation.reference.ReferenceContext;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;

/**
 * A reference extractor is used to retrieve the references of an 
 * SML document.  The data structure returned is a {@link List} of 
 * reference URIs in the form of {@link String}. 
 * 
 * @author Ali Mehregani
 */
public class ReferenceExtractor extends AbstractTrackerDataBuilder<List<ReferenceDescriptor>>
{
	/**
	 * The ID of this data builder
	 */
	public static final String ID = SMLActivator.PLUGIN_ID + ".ReferenceExtractor"; //$NON-NLS-1$
	
	/**
	 * Indicates when the URI element has been hit
	 */
	private boolean uriElementHit;
	
	/**
	 * Stores the current reference
	 */
	private StringBuffer currentReference;
	
	/**
	 * Stores each reference retrieved
	 */
	private List<ReferenceDescriptor> references;
	
	/**
	 * Stores all SML references; an SML reference is identified by sml:ref = true
	 */
	private List<ReferenceContext> contextReferences;
	
	/**
	 * set to true when an sml:ref attribute is found
	 * For an element that has sml:ref = true, all childern under that reference must be
	 * processed as being SML reference schemes; ( default reference scheme is sml:uri )
	 * If an sml:uri element is located and it's parent does not have the sml:ref set to true,
	 * than that element should not be processed as an sml reference
	 */
	private boolean isSMLRef;
		
	/**
	 * The SML reference containing a set of SML reference schemes
	 * An SML reference is identified by the sml:ref attribute
	 * while an SML reference scheme are all the children of such element;
	 * in particular, a reference scheme is identified by the sml:uri element
	 */
	private ReferenceContext refContext;

	private int refContextStack;

	public ReferenceExtractor()
	{
		currentReference = new StringBuffer();
		isSMLRef = false;
		refContextStack = 0;
		
		references = new ArrayList<ReferenceDescriptor>();
		contextReferences = new ArrayList<ReferenceContext>();
		super.addEvent(IFoundationBuilder.EVENT_CHARACTER);
	}
	
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.AbstractDataBuilder#startElement(java.lang.String, java.lang.String, java.lang.String, org.xml.sax.Attributes)
	 */
	public void startElement(String uri, String localName, String name, Attributes attributes) throws SAXException
	{	
		super.startElement(uri, localName, name, attributes);
		Object value = attributes.getValue(ISMLConstants.SML_URI, ISMLConstants.REF_ATTRIBUTE);
		if(value != null)
		{
			isSMLRef= SMLValidatorUtil.isTrue(value.toString());
			
			if(isSMLRef == true)
			{//sml reference starts here
			
				refContext = new ReferenceContext(getFilePath(), getLocator().getLineNumber());
				refContext.setNamespace(uri);
				refContext.setElementName(localName);				
				refContext.setNodePath(getNodePath());					
				contextReferences.add(refContext);
				
				String[] aliases = getAliases();				
				refContext.setAliases(aliases);
				
				if (aliases == null || aliases.length <= 0)
				{
					refContext.setDocumentPosition(getOrphanInx());
				}
				
				
				value = attributes.getValue(ISMLConstants.SML_URI, ISMLConstants.URI_NILREF);
				if(value != null)
				{
					refContext.setNilref(SMLValidatorUtil.isTrue(value.toString()));
				}				 								
				refContextStack= 0;
			}
		}
		
		if(isSMLRef && uri.equals(refContext.getNamespace()) && localName.equals(refContext.getElementName()))
		{
			refContextStack++;
		}
				
		else if (ISMLConstants.SML_URI.equals(uri) && ISMLConstants.URI_ELEMENT.equals(localName))
		{
			if(isSMLRef)
				uriElementHit = true;			
		}
		
	}
	
	public void endElement(String uri, String localName, String name) throws SAXException
	{
		super.endElement(uri, localName, name);
		if(isSMLRef && uri.equals(refContext.getNamespace()) && localName.equals(refContext.getElementName()))
		{
			refContextStack--;
			
			if(refContextStack == 0)
			{//the SML reference ends here
				
				isSMLRef = false;
				refContext = null;				
			}
		}
		
		
		else if (ISMLConstants.SML_URI.equals(uri) && ISMLConstants.URI_ELEMENT.equals(localName))
		{
			if(refContext != null)
			{//this is a real reference scheme since it is defined within an element with sml:ref = true
			
				uriElementHit = false;
				ReferenceDescriptor ref = new ReferenceDescriptor(getFilePath(), getContextLineNumber() - 1, currentReference.toString());
				
				references.add(ref);
				currentReference = new StringBuffer();
				
				refContext.addReferenceScheme(ref);
				ref.setContext(refContext);
			}
		}
	}
			
	/**
	 * @see org.xml.sax.helpers.DefaultHandler#characters(char[], int, int)
	 */
	public void characters(char[] ch, int start, int length) throws SAXException
	{		
		if (uriElementHit) {
			String string = new String(ch, start, length);
			if (currentReference.length() > 0) {
				// joining multi-line references with a space
				// a \n in the first character indicates this
				if ((string.charAt(0) == '\n') && (string.trim().length() > 0)) {
					currentReference.append(' ');
				}
			}
			// the trim() call trims all tabs and new line characters
			currentReference.append(string.trim());
		}
	}
	
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.IDataBuilder#getDataStructure()
	 */
	public List<ReferenceDescriptor> getDataStructure()
	{
		return references;
	}

	/**
	 * @return the contextReferences
	 */
	public List<ReferenceContext> getContextReferences() {
		return contextReferences;
	}
}
