/**********************************************************************
 * Copyright (c) 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.databuilders;

import java.util.Map;
import java.util.StringTokenizer;

import org.eclipse.cosmos.rm.internal.validation.SMLActivator;
import org.eclipse.cosmos.rm.internal.validation.artifacts.SchemaBindings;
import org.eclipse.cosmos.rm.internal.validation.artifacts.SchemaBindings.SchemaBinding;
import org.eclipse.cosmos.rm.internal.validation.common.ISMLConstants;
import org.eclipse.cosmos.rm.internal.validation.common.IValidationConstants;
import org.eclipse.cosmos.rm.internal.validation.core.FoundationBuilder;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;

/**
 * Constructs a data structure to store information about schema
 * bindings.  The structure returned is of type {@link SchemaBindings}.
 * 
 * @author Ali Mehregani
 */
public class SchemaBindingsBuilder extends AbstractDataBuilder<SchemaBindings>
{
	/**
	 * The ID of this builder
	 */
	public static final String ID = SMLActivator.PLUGIN_ID + ".SchemaBindingDataBuilder";
	
	/**
	 * The data structure used in storing the schema bindings
	 */
	private SchemaBindings schemaBindings;
	
	/**
	 * The current schema binding
	 */
	private SchemaBinding schemaBinding;

	/**
	 * Indicates when the &lt;documentAlias&gt; element is 
	 * hit
	 */
	private boolean documentAliasElement;

	/**
	 * The string buffer used to store a document alias
	 */
	private StringBuffer documentAliasBuffer;
	
	/**
	 * Indicates the end of the header data
	 */
	private boolean endOfHeader;
	
	
	/**
	 * Constructor
	 */
	public SchemaBindingsBuilder()
	{
		schemaBindings = new SchemaBindings();		
	}
	
	
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.AbstractDataBuilder#initialize(java.util.Map)
	 */
	public void initialize(Map<String, Object> init)
	{
		super.initialize(init);
		addEvent(FoundationBuilder.EVENT_CHARACTER);
	}
	
	
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.AbstractDataBuilder#startElement(java.lang.String, java.lang.String, java.lang.String, org.xml.sax.Attributes)
	 */
	public void startElement(String uri, String localName, String qname, Attributes attributes) throws SAXException
	{
		if (endOfHeader)
		{
			return;
		}
		
		super.startElement(uri, localName, qname, attributes);		
		if (!isSMLIFElement())
		{
			return;
		}
		
	
		// <schemaBindings> element
		if (ISMLConstants.SCHEMA_BINDINGS_ELEMENT.equals(localName))
		{
			schemaBindings.setPresent(true);
		}
		
		// <schemaBinding>/<defaultSchema>/ or <noSchemaBinding> element
		else if (ISMLConstants.SCHEMA_BINDING_ELEMENT.equals(localName) || 
				ISMLConstants.DEFAULT_SCHEMA_ELEMENT.equals(localName) ||
				ISMLConstants.NO_SCHEMA_BINDING_ELEMENT.equals(localName))
		{
			schemaBinding = new SchemaBinding();
		}
	
		
		// <namespaceBinding> element
		else if (ISMLConstants.NAMESPACE_BINDING_ELEMENT.equals(localName))
		{
			// Find the aliases of the definition documents
			String aliases = attributes.getValue(IValidationConstants.EMPTY_STRING, ISMLConstants.ALIASES_ATTRIBUTE);
			aliases = isNull(aliases) ? attributes.getValue(ISMLConstants.SMLIF_URI, ISMLConstants.ALIASES_ATTRIBUTE) : aliases;
			aliases = isNull(aliases) ? attributes.getValue(ISMLConstants.ALIASES_ATTRIBUTE) : aliases;
			
			if (isNull(aliases))
			{
				return;
			}
			
			StringTokenizer st = new StringTokenizer (aliases);
			while (st.hasMoreTokens())
			{
				schemaBinding.addSchemaAlias(st.nextToken());
			}
		}
		
		// <documentAlias> element
		else if (ISMLConstants.DOCUMENT_ALIAS_ELEMENT.equals(localName))
		{
			documentAliasElement = true;
			documentAliasBuffer = new StringBuffer();
		}
		
		// <definition>/<instance> element
		else if (ISMLConstants.DEFINITIONS_ELEMENT.equals(localName) || 
				 ISMLConstants.INSTANCES_ELEMENT.equals(localName))
		{
			endOfHeader = true;
		}
	}
	
	
	/**
	 * Returns true if str is either null or 
	 * an empty string
	 * 
	 * @param str The string to be checked
	 * @return true if str is either null or empty; false
	 * otherwise
	 */
	private boolean isNull(String str)
	{		
		return str == null || str.length() <= 0;
	}

	
	/**
	 * @see org.xml.sax.helpers.DefaultHandler#characters(char[], int, int)
	 */
	public void characters(char[] ch, int start, int length) throws SAXException
	{
		if (documentAliasElement)
		{
			documentAliasBuffer.append (new String(ch, start, length).trim());
		}
	}

	
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.AbstractDataBuilder#endElement(java.lang.String, java.lang.String, java.lang.String)
	 */
	public void endElement(String uri, String localName, String name) throws SAXException
	{		
		if (endOfHeader)
		{
			return;
		}
		
		super.endElement(uri, localName, name);
		if (!isSMLIFElement() || schemaBinding == null)
		{
			return;
		}
		
		// </defaultSchema> element
		if (ISMLConstants.DEFAULT_SCHEMA_ELEMENT.equals(localName))
		{
			schemaBindings.setDefaultBinding(schemaBinding);
		}
		
		// </documentAlias> element
		else if (ISMLConstants.DOCUMENT_ALIAS_ELEMENT.equals(localName))
		{
			documentAliasElement = false;
			schemaBinding.addDocumentAlias(documentAliasBuffer.toString());
		}		
		
		// </schemaBinding> element
		else if (ISMLConstants.SCHEMA_BINDING_ELEMENT.equals(localName))
		{
			schemaBindings.addBinding(schemaBinding);
			schemaBinding = null;
		}
		
		// </noSchemaBinding> element
		else if (ISMLConstants.NO_SCHEMA_BINDING_ELEMENT.equals(localName))
		{
			schemaBindings.setNoSchemaBindings(schemaBinding);
			schemaBinding = null;
		}
	}
	
	
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.IDataBuilder#getDataStructure()
	 */
	public SchemaBindings getDataStructure()
	{		
		return schemaBindings;
	}
}
