/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.smlvalidators;

import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintStream;
import java.io.PrintWriter;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.xml.namespace.NamespaceContext;
import javax.xml.transform.ErrorListener;
import javax.xml.transform.Result;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.TransformerFactoryConfigurationError;
import javax.xml.transform.URIResolver;
import javax.xml.transform.dom.DOMResult;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpression;
import javax.xml.xpath.XPathExpressionException;

import org.apache.xerces.dom.PSVIElementNSImpl;
import org.apache.xerces.xs.XSComplexTypeDefinition;
import org.apache.xerces.xs.XSTypeDefinition;
import org.eclipse.cosmos.rm.internal.validation.artifacts.DOMStructure;
import org.eclipse.cosmos.rm.internal.validation.artifacts.ElementLocation;
import org.eclipse.cosmos.rm.internal.validation.artifacts.ElementNode;
import org.eclipse.cosmos.rm.internal.validation.artifacts.ElementTypeMap;
import org.eclipse.cosmos.rm.internal.validation.artifacts.RuleBindings;
import org.eclipse.cosmos.rm.internal.validation.artifacts.Schematron;
import org.eclipse.cosmos.rm.internal.validation.artifacts.Schematron.Pattern;
import org.eclipse.cosmos.rm.internal.validation.artifacts.Schematron.Rule;
import org.eclipse.cosmos.rm.internal.validation.common.ISMLConstants;
import org.eclipse.cosmos.rm.internal.validation.common.IValidationConstants;
import org.eclipse.cosmos.rm.internal.validation.common.IValidationMessage;
import org.eclipse.cosmos.rm.internal.validation.common.IValidationOutput;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidationMessages;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidatorUtil;
import org.eclipse.cosmos.rm.internal.validation.common.AbstractValidationOutput.ValidationMessage;
import org.eclipse.cosmos.rm.internal.validation.common.AbstractValidationOutput.ValidationMessageFactory;
import org.eclipse.cosmos.rm.internal.validation.core.AbstractSMLValidator;
import org.eclipse.cosmos.rm.internal.validation.databuilders.DataBuilderRegistry;
import org.eclipse.cosmos.rm.internal.validation.databuilders.DocumentDOMBuilder;
import org.eclipse.cosmos.rm.internal.validation.databuilders.ElementSchematronCacheBuilder;
import org.eclipse.cosmos.rm.internal.validation.databuilders.ElementTypeMapDataBuilder;
import org.eclipse.cosmos.rm.internal.validation.databuilders.NamespaceContextBuilder;
import org.eclipse.cosmos.rm.internal.validation.reference.DerefXPathFunction;
import org.eclipse.cosmos.rm.internal.validation.util.FileHelper;
import org.eclipse.cosmos.rm.internal.validation.util.ParserHelper;
import org.eclipse.osgi.util.NLS;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * This validator will apply schematrons to elements whose schema
 * has a schematron. 
 * 
 * @author sleeloy
 * @author Ali Mehregani
 */
public class SchematronValidator extends AbstractSMLValidator 
{
	// The style sheet to extract the schematron from the schema file
	private final static String ISO_SKELETON = "validation-resources/customized_skeleton.xsl"; //$NON-NLS-1$
		
	protected IValidationOutput<String, Object> validationLogger;
	protected ElementTypeMap elementTypeMapBuilderStructure;
	protected RuleBindings schemaCacheBuilderStructure;	
	protected DocumentDOMBuilder documentDOMBuilder;

	/* Setup the instance level builders used by the validator
	 * (non-Javadoc)
	 * @see org.eclipse.cosmos.rm.internal.validation.core.AbstractValidator#initialize(java.util.Map)
	 */
	public void initialize(Map<String, Object> validationAttribute)
	{
		super.initialize(validationAttribute);

		DataBuilderRegistry builderRegistry = DataBuilderRegistry.getInstanceLevelRegistry();
		
		builderRegistry.registerDataStructureBuilder(ElementTypeMapDataBuilder.ID, new ElementTypeMapDataBuilder());
		builderRegistry.registerDataStructureBuilder(DocumentDOMBuilder.ID, new DocumentDOMBuilder());				
		builderRegistry.registerDataStructureBuilder(NamespaceContextBuilder.ID, new NamespaceContextBuilder());			
	}

	public boolean validate()
	{			
		setTaskName(SMLValidationMessages.validationSchematron);				
		validationLogger = getValidationOutput();
		documentDOMBuilder = (DocumentDOMBuilder) DataBuilderRegistry.getInstanceLevelRegistry().getDataStructureBuilder(DocumentDOMBuilder.ID);
		elementTypeMapBuilderStructure = (ElementTypeMap)SMLValidatorUtil.retrieveDataStructure(ElementTypeMapDataBuilder.ID);
		DOMStructure domDocuments = (DOMStructure)SMLValidatorUtil.retrieveDataStructure(DocumentDOMBuilder.ID);
		schemaCacheBuilderStructure = (RuleBindings)SMLValidatorUtil.retrieveDataStructure(DataBuilderRegistry.TOP_LEVEL_MODE, ElementSchematronCacheBuilder.ID);	
		List<Node> docWithNoAlias = domDocuments.getOrphanedDocuments();

		// Walk through documents and determine which elements to apply the schematron 
		Iterator<String> iter = domDocuments.getDocuments().keySet().iterator();
		while (iter.hasNext()){
			String alias = iter.next();
			Node rootElement = domDocuments.get(alias);			
			if (rootElement instanceof Element)
			{
				if (!processNode((Element)rootElement, rootElement, alias) && shouldAbortOnError())
				{
					return false;
				}
			}
		}
		
		//Walk through documents with no alias and determine which elements to apply the schematron 
		Iterator<Node> orphanIterator = docWithNoAlias.iterator();
		while (orphanIterator.hasNext()){
			Object rootElement = orphanIterator.next();
			if (rootElement instanceof Element)
			{
				if (!processNode((Element)rootElement, (Node)rootElement, null) && shouldAbortOnError())
					return false;
			}
		}
		
		return true;
	}
	
	/**
	 * Processes a node to determine if a schematron should be applied.  
	 * @param currentElement node to process
	 * @param rootElement root document element associated with the current element
	 * @param alias alias name of the root document.  This is needed if rule binding is defined in document
	 * @return true if node was processed successfully otherwise false is returned.
	 */
	protected boolean processNode(Node currentElement, Node rootElement, Object alias){
		
		if (currentElement.getNodeType() != Node.ELEMENT_NODE)
			return true;
		
		// Allow all rules to be evaluated before exiting
		boolean allPass = true;

		// Rules bound to specific documents
		if (schemaCacheBuilderStructure.isRuleBindingPresent())
		{
			// Check to see if this element has a schematron
			String nodeNamespace = currentElement.getNamespaceURI();
			List<Schematron> schematronRules = schemaCacheBuilderStructure.getBoundRules(nodeNamespace, currentElement.getLocalName());			
			
			// Check alias
			if ((alias != null) && (schematronRules == null))
			{
				schematronRules = schemaCacheBuilderStructure.getBoundRules((String)alias);
			}
			
			// Check to see if schematron is associated with the type
			String targetType = null; 
			if (schematronRules == null && (targetType = elementTypeMapBuilderStructure.getType(currentElement.getNamespaceURI(), currentElement.getLocalName())) != null)
			{				
				schematronRules = schemaCacheBuilderStructure.getBoundRules(nodeNamespace, targetType+ISMLConstants.TYPEDELIM);
				
				// Check inheritance tree
				if (schematronRules == null) {
					schematronRules = retrieveRulesFromHierarchy(currentElement, targetType, nodeNamespace);
				}										
			}
			
			// Keep track of xpath in builder
			if (schematronRules != null) 
			{
				for (Iterator<Schematron> iter = schematronRules.iterator(); iter.hasNext();)
				{
					Schematron rule = iter.next();
					if (!(validateSchematron(rule, currentElement, rootElement)) && shouldAbortOnError()) 
					{
						allPass = false;
					}
				}
			}
		}
		// Globally bound rules
		List<Schematron> globallyBoundRules = schemaCacheBuilderStructure.getGloballyBoundRules();
		for (int i = 0, ruleCount = globallyBoundRules.size(); i < ruleCount; i++)
		{
			if (!(validateSchematron(globallyBoundRules.get(i), currentElement, rootElement)) && shouldAbortOnError()) {
				allPass = false;
			}
		}
		
		// Check children elements to see if a schematron should be applied
		NodeList nodeLists = currentElement.getChildNodes();
		for (int x = 0; x <nodeLists.getLength(); x++){
			Node node = nodeLists.item(x);
			if (node.getNodeType() == Node.ELEMENT_NODE)
			{					
				if (!processNode(node, rootElement, null) && shouldAbortOnError()) {
					allPass = false;
				}
			}
		}
		
		
		return allPass;				
	}
	
	/**
	 * Traverse the hierarchy to find schematron rules matching the type of the currentElement
	 * 
	 * @param currentElement
	 * @param targetType
	 * @param nodeNamespace
	 * @return list of schematron rules to be applied
	 */
	protected List<Schematron> retrieveRulesFromHierarchy(Node currentElement, String targetType, String nodeNamespace) {
		List<Schematron> schematronRules = null;
		PSVIElementNSImpl psviElementNode = (PSVIElementNSImpl) currentElement;
		XSTypeDefinition type = (XSTypeDefinition) psviElementNode.getTypeDefinition();
		while ((type instanceof XSComplexTypeDefinition) && (((XSComplexTypeDefinition) type).getContentType() == XSComplexTypeDefinition.CONTENTTYPE_ELEMENT)) {
			schematronRules = schemaCacheBuilderStructure.getBoundRules(nodeNamespace, (ParserHelper.removeNameSpace(type.getName())+ISMLConstants.TYPEDELIM));
			if (schematronRules != null)
			{
				break;
			}
			type = type.getBaseType();
		}
		return schematronRules;
	}
	
	protected boolean validateSchematron(Schematron schematronNode, Node currentElement, Node rootElement)
	{
		ErrorCaptureStream errorCaptureStream = new ErrorCaptureStream();
		try 
		{						
	    	StringBuffer buffer = schematronNode.getFragment();

	    	//valid if there's no schema to validate against
			if (buffer == null) 
				return true;

			String fragment = buffer.toString(); 
			int schemaInx = fragment.lastIndexOf("</");
			if (schemaInx < 0)
				return true;
			
			List<Pattern> patterns = schematronNode.getPatterns();
			
			/* For every pattern */
			for (int i = 0, patternCount = patterns.size(); i < patternCount; i++)
			{
				Pattern pattern = patterns.get(i);
				List<Rule> rules = pattern.getRules();
				StringBuffer schematronFragmentBuffer = new StringBuffer().append(buffer);
				// https://bugs.eclipse.org/bugs/show_bug.cgi?id=180809  
				schematronFragmentBuffer.insert(schemaInx, pattern.getFragment().toString());		
				
				/* For every rule of the pattern */
				for (int j = 0, ruleCount = rules.size(); j < ruleCount; j++)
				{
					int patternInx = schematronFragmentBuffer.substring(0, schemaInx + pattern.getFragment().length()).lastIndexOf("</");
					if (patternInx < 0)
						continue;
					
					StringBuffer fragmentWithRule = new StringBuffer().append(schematronFragmentBuffer); 
					Rule rule = rules.get(j);
					
					/* If the context does not contain a deref function and it is not an absolute path, then define it relative
					 * to the document node. We need to define the context path relative to the root document node to allow user
					 * to use the parent axes. */					
					String ruleFragment = null;
					if (!rule.getMatchContainsDeref() && !rule.getContext().startsWith(ISMLConstants.FORWARD_SLASH))
					{
						ruleFragment = rule.getFragment().toString();
						StringBuffer pathToRoot = new StringBuffer();
						findPathToRoot(pathToRoot, currentElement);
						String ruleContext = rule.getContext();
						if (ruleContext.startsWith(ISMLConstants.PERIOD))
						{
							int forwardSlashInx = ruleContext.indexOf(ISMLConstants.FORWARD_SLASH);
							ruleContext = forwardSlashInx >= 0 ? ruleContext.substring(forwardSlashInx) : IValidationConstants.EMPTY_STRING;											
						}
						ruleContext = ruleContext.length() > 0 ? ISMLConstants.FORWARD_SLASH + ruleContext : ruleContext;
						ruleFragment = SMLValidatorUtil.stringReplace(ruleFragment, "\""+rule.getContext()+"\"", "\""+pathToRoot.toString().substring(0, pathToRoot.toString().length() - 1)+ruleContext+"\"");
					}
					
					fragmentWithRule.insert(patternInx, ruleFragment == null ? rule.getFragment().toString() : ruleFragment);																			
					StringReader sr = new StringReader(fragmentWithRule.toString());

					//extract xsl from schematron
					DOMResult schematron = new DOMResult();
					// TODO handle the null case when the schematron skeleton can't be found
					transform(new StreamSource(sr), new StreamSource(FileHelper.getResourceAsStream(ISO_SKELETON)), schematron, errorCaptureStream);
					
					final Map<String, String> prefixMap = schematronNode.getPrefixMap();
					String ruleContext = rule.getContext();
					if (rule.getMatchContainsDeref() && ruleContext.length() > 0)
					{
						NamespaceContext namespaceContext = new NamespaceContext(){
			
							public String getNamespaceURI(String prefix)
							{
								String uri = prefixMap.get(prefix);
								return uri == null ? IValidationConstants.EMPTY_STRING : uri;
							}
			
							public String getPrefix(String arg0)
							{
								return null;
							}
			
							public Iterator<?> getPrefixes(String arg0)
							{
								return null;
							}};
							
						SMLValidatorUtil.xpath.setNamespaceContext(namespaceContext);					
						XPathExpression xpathExp = SMLValidatorUtil.xpath.compile(ruleContext);					
						NodeList nodeList = (NodeList)xpathExp.evaluate(currentElement, XPathConstants.NODESET);
						
						// If the context of the Schematron rule can't be determined, then we should report a failure
						if (nodeList == null || nodeList.getLength() <= 0)
						{		
							ElementLocation elementLocation = currentElement instanceof Element ? 
									documentDOMBuilder.getLocation((Element)currentElement) : 
									null;
							
							IValidationMessage message = elementLocation == null ? 
									ValidationMessageFactory.createErrorMessage(rule.getLineNumber(), NLS.bind(SMLValidationMessages.schematronMissingContext, ruleContext)) :
									ValidationMessageFactory.createErrorMessage(elementLocation.getFilePath(), elementLocation.getLineNumber(), NLS.bind(SMLValidationMessages.schematronMissingContext, ruleContext));
							validationLogger.reportMessage(message);
							if (shouldAbortOnError()) {
								return false;
							}							
						}
						
						//apply schematron on the nodes returned by evaluating the rule context 
						for (int x = 0; x < nodeList.getLength(); x++)
						{												
							if (!applySchematron(errorCaptureStream, nodeList.item(x), schematron.getNode()) && shouldAbortOnError())
							{
								return false;
							}
						}
					}
					else
					{
						if (ruleFragment != null)
							currentElement = currentElement.getOwnerDocument().getFirstChild();
						
						//apply schematron on the current element
						if (!applySchematron(errorCaptureStream, currentElement, schematron.getNode()) && shouldAbortOnError())
							return false;
					}
				}
			}

			return true;
		}
		catch (XPathExpressionException xe) 
		{			
			printStackTrace(xe, schematronNode.getLineNumber());
			handleError (errorCaptureStream, xe.getLocalizedMessage(), schematronNode.getLineNumber());
		}
		catch (TransformerException xe) 
		{
			printStackTrace(xe, schematronNode.getLineNumber());
			handleError (errorCaptureStream, xe.getLocalizedMessage());
		}
		catch (Exception e) 
		{
			printStackTrace(e, schematronNode.getLineNumber());
			handleError (errorCaptureStream, e.getLocalizedMessage());
		}
		return false;
	}

	private void findPathToRoot(StringBuffer pathToRoot, Node currentElement)
	{
		pathToRoot.insert(0, currentElement.getNodeName()+'['+findIndex(currentElement)+']'+ISMLConstants.FORWARD_SLASH);
		Node parentNode = currentElement.getParentNode();
		if (parentNode == null || parentNode.getParentNode() == null || parentNode.getParentNode().getNodeType() == Node.DOCUMENT_NODE)
			return;
		findPathToRoot(pathToRoot, parentNode);
	}

	private int findIndex(Node currentElement)
	{
		Node[] children = SMLValidatorUtil.retrieveChildElements(currentElement.getParentNode());
		for (int i = 0; i < children.length; i++)
		{
			if (children[i].equals(currentElement))
				return i+1;
		}
		
		return -1;		
	}

	private void handleError(ErrorCaptureStream errorCaptureStream, String msg)
	{
		handleError(errorCaptureStream, msg, ValidationMessage.NO_LINE_NUMBER);
	}
	
	private void handleError(ErrorCaptureStream errorCaptureStream, String msg, int lineNumber)
	{
		if (!errorCaptureStream.isEmpty())
		{
			String[] lines = errorCaptureStream.getLines();
			for (int i = 0; i < lines.length && lines[i] != null; i++)
			{
				validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(ValidationMessage.NO_LINE_NUMBER, lines[i]));
			}
		}
		validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(lineNumber, msg));
	}
	
	private void printStackTrace(Exception xe, int lineNumber)
	{
		StringWriter sw = new StringWriter();
		xe.printStackTrace(new PrintWriter(sw));
		
		validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(lineNumber, sw.toString()));
	}

	private boolean applySchematron(ErrorCaptureStream errorCaptureStream, Node currentElement, Node schematron)
	{
		StringWriter sw = new StringWriter();
		StreamResult output = new StreamResult(sw);
		DOMSource schematronSource =  new DOMSource(schematron);
		
		// apply schematron to main document
		try
		{
			DerefXPathFunction.instance().setDocumentNode(currentElement);
			transform(createStreamSource(currentElement), schematronSource, output, errorCaptureStream);
			DerefXPathFunction.instance().setDocumentNode(null);
		} 
		catch (Exception e)
		{
			handleError (errorCaptureStream, NLS.bind(SMLValidationMessages.schematronTransformation, e.getLocalizedMessage() == null ? IValidationConstants.EMPTY_STRING : e.getLocalizedMessage()));
			if (shouldAbortOnError()) {
				return false;
			}
		}
		
		// check to see if this is an empty string
		String result = sw.getBuffer().toString().trim();
		if (result.startsWith("<?xml")) {
			// remove xml prolog
			int ind = result.indexOf("?>")+2;
			result = result.substring(ind).trim();
		}
		
		if (result.trim().length() > 0 )
		{
			validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(ValidationMessage.NO_LINE_NUMBER, result));
			if (shouldAbortOnError()) {
				return false;
			}
		}
		return true;
	}

	private StreamSource createStreamSource(Node node) {
		int lineNumber = 0;
		if (node instanceof ElementNode) {
			lineNumber = ((ElementNode) node).getLocation().getLineNumber();
		}
		try
		{
			Transformer transformer = TransformerFactory.newInstance().newTransformer();

			//	initialize StreamResult with File object to save to file
			StreamResult result = new StreamResult(new StringWriter());
			DOMSource source = new DOMSource(node);
			transformer.transform(source, result);

			return new StreamSource(new StringReader(result.getWriter().toString()));
		} 
		catch (TransformerConfigurationException e) 
		{
			validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(lineNumber, e.getLocalizedMessage()));
		} 
		catch (IllegalArgumentException e) 
		{
			validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(lineNumber, e.getLocalizedMessage()));
		} 
		catch (TransformerFactoryConfigurationError e) 
		{
			validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(lineNumber, e.getLocalizedMessage()));
		} 
		catch (TransformerException e)
		{
			validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(lineNumber, e.getLocalizedMessage()));
		}
		
		return null;
	}
	
	/**
	 * Helper method that will apply a XSLT transform to a given xml source and write the results
	 * to a Result object
	 * @param source xml source 
	 * @param xslt xslt transform that will be applied to the source
	 * @param result stores the resulting transform
	 * @param captureError Used to capture errors written to the error stream
	 * @throws TransformerException 
	 */
	protected void transform(Source source, Source xslt, Result result, ErrorCaptureStream captureError) throws TransformerException 
	{
		Transformer transformer = null;

		// Instantiate a TransformerFactory
		TransformerFactory tFactory = TransformerFactory.newInstance();
		tFactory.setURIResolver(new XslUriResolver());
		// Use the TransformerFactory to process the stylesheet source and
		// generate a Transformer.
		transformer = tFactory.newTransformer(xslt);
		// TODO what does a null transformer mean?  bogus schematron code?
		transformer.setErrorListener(new ErrorListener()
		{
			public void error(TransformerException e) throws TransformerException 
			{
				throw e;				
			}

			public void fatalError(TransformerException e) throws TransformerException 
			{
				throw e;
			}

			public void warning(TransformerException e) throws TransformerException 
			{
				throw e;
			}
		}
		);
				
		// Use the Transformer to transform an XML Source and send the
		// output to a Result object.
		System.setErr(new PrintStream(captureError));
		transformer.transform(source, result);		
		System.setErr(System.err);
	}

	private static class ErrorCaptureStream extends OutputStream
	{
		private String[] lines;
		private int currentInx;
		private StringBuffer buffer;
		
		public ErrorCaptureStream()
		{
			lines = new String[5];
			currentInx = 0;
			buffer = new StringBuffer();
		}
		public void write(int b) throws IOException
		{
			if (currentInx >= lines.length) {
				return;
			}
			
			if (b == '\n') {
				if (currentInx == 0 || !buffer.toString().equals(lines[currentInx - 1])) {
					lines[currentInx++] = buffer.toString();
				}
				buffer = new StringBuffer();
				return;
			}
			
			if (b != '\r') {
				buffer.append((char)b);
			}
		}
		
		public boolean isEmpty()
		{
			return lines[0] == null;
		}
		
		public String[] getLines()
		{
			return lines;
		}
	}

	/**
	 * Private class for handling references in an XSLT
	 * Relative paths in href attribute will be loaded by classloader. 
	 */
	private class XslUriResolver implements URIResolver {

		public Source resolve(String href, String base) throws TransformerException {
			return new StreamSource(FileHelper.getResourceAsStream(href));
		}
		
	}


}
