/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.requirement;

import java.util.Collection;

import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.SPISessionImpl;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.exception.InvalidInputException;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.requirement.AlternativeRequirement;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.requirement.CapacityConstraint;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.requirement.ConsumptionConstraint;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.requirement.InternalDependency;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.requirement.PropertyConstraint;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.requirement.RelationshipConstraint;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.requirement.Requirement;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.requirement.RequirementFactory;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.requirement.RequirementResourceConstraint;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.requirement.UniquenessConstraint;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.requirement.VersionConstraint;
import org.w3c.dom.Element;

public class RequirementFactoryImpl implements RequirementFactory {
	private SPISessionImpl session;
	private static final String CAPACITY_CONSTRAINT_TYPE = "sdd-dd:CapacityConstraintType";
	private static final String CONSUMPTION_CONSTRAINT_TYPE = "sdd-dd:ConsumptionConstraintType";
	private static final String PROPERTY_CONSTRAINT_TYPE = "sdd-dd:PropertyConstraintType";
	private static final String PROPERTY_VALUE_LIST_TYPE = "sdd-dd:PropertyValueListType";
	
	public RequirementFactoryImpl (SPISessionImpl session) {
		this.session = session;
	}

	public AlternativeRequirement createAlternativeRequirement(String alternativeRequirementID) {
		session.testParameter(alternativeRequirementID, 1, null);

		Element alternativeRequirementElement = session.createDeploymentElement("Alternative");
		alternativeRequirementElement.setAttribute("id", alternativeRequirementID);
		
		return new AlternativeRequirementImpl(alternativeRequirementElement, session);
	}

	public CapacityConstraint createCapacityConstraint(String propertyName) {
		session.testParameter(propertyName, 1, null);

		Element capacityConstraintElement = session.createDeploymentElement("CapacityConstraint");
		
		Element propertyNameElement = session.createDeploymentElement("PropertyName");
		propertyNameElement.setTextContent(propertyName);
		
		session.insertNode(capacityConstraintElement, propertyNameElement, CAPACITY_CONSTRAINT_TYPE);
		
		return new CapacityConstraintImpl(capacityConstraintElement, session);
	}

	public ConsumptionConstraint createConsumptionConstraint(String propertyName) {
		session.testParameter(propertyName, 1, null);

		Element consumptionConstraintElement = session.createDeploymentElement("ConsumptionConstraint");
		
		Element propertyNameElement = session.createDeploymentElement("PropertyName");
		propertyNameElement.setTextContent(propertyName);

		session.insertNode(consumptionConstraintElement, propertyNameElement, CONSUMPTION_CONSTRAINT_TYPE);
		
		return new ConsumptionConstraintImpl(consumptionConstraintElement, session);
	}

	public PropertyConstraint createPropertyConstraintMultiValue(String propertyName, Collection<String> propertyValues) {
		session.testParameter(propertyName, 1, null);
		session.testParameter(propertyValues, 2, null);
		
		Element propertyNameElement = session.createDeploymentElement("PropertyName");
		propertyNameElement.setTextContent(propertyName);
		
		Element propertyValueListElement = session.createDeploymentElement("ListOfValues");
		
		for (String propertyValue : propertyValues) {
			Element valueElement = session.createDeploymentElement("Value");
			valueElement.setTextContent(propertyValue);
			
			session.insertNode(propertyValueListElement, valueElement, PROPERTY_VALUE_LIST_TYPE);
		}
		
		Element pcElement = session.createDeploymentElement("PropertyConstraint");
        session.insertNode(pcElement, propertyNameElement, PROPERTY_CONSTRAINT_TYPE);
        session.insertNode(pcElement, propertyValueListElement, PROPERTY_CONSTRAINT_TYPE);
		
		return new PropertyConstraintImpl(pcElement, session);
	}

	public PropertyConstraint createPropertyConstraintSingleValue(String propertyName, String value) {
		session.testParameter(propertyName, 1, null);
		session.testParameter(value, 2, null);
		
		Element propertyNameElement = session.createDeploymentElement("PropertyName");
		propertyNameElement.setTextContent(propertyName);
		
		Element valueElement = session.createDeploymentElement("Value");
		valueElement.setTextContent(value);
		
		Element pcElement = session.createDeploymentElement("PropertyConstraint");
        session.insertNode(pcElement, propertyNameElement, PROPERTY_CONSTRAINT_TYPE);
        session.insertNode(pcElement, valueElement, PROPERTY_CONSTRAINT_TYPE);
		
		return new PropertyConstraintImpl(pcElement, session);
	}

	public Requirement createRequirement(String requirementID) {
		session.testParameter(requirementID, 1, null);
		
		Element requirementElement = session.createDeploymentElement("Requirement");
		requirementElement.setAttribute("id", requirementID);
		
		return new RequirementImpl(requirementElement, session);
	}

	public RequirementResourceConstraint createRequirementResourceConstraint(String resourceConstraintID,
			String resourceReference) {
		session.testParameter(resourceConstraintID, 1, null);
		session.testParameter(resourceReference, 2, null);

		Element rrcElement = session.createDeploymentElement("ResourceConstraint");
		rrcElement.setAttribute("id", resourceConstraintID);
		rrcElement.setAttribute("resourceRef", resourceReference);
		 
		return new RequirementResourceConstraintImpl(rrcElement, session);
	}

	public VersionConstraint createVersionConstraint() {
		Element vcElement = session.createDeploymentElement("VersionConstraint");
		
		return new VersionConstraintImpl(vcElement, session);
	}

	public InternalDependency createInternalDependency(String contentRef) throws InvalidInputException {
        session.testParameter(contentRef, 1, null);

        Element internalDepElement = session.createDeploymentElement("InternalDependency");
        internalDepElement.setAttribute("contentElementRef", contentRef);
		        
        return new InternalDependencyImpl(internalDepElement, session);
    }
	
    public RelationshipConstraint createRelationshipConstraint(String type) throws InvalidInputException {
        session.testParameter(type, 1, null);

        Element rcElement = session.createDeploymentElement("RelationshipConstraint");
        rcElement.setAttribute("type", type);
        
        return new RelationshipConstraintImpl(rcElement, session);
    }

    public UniquenessConstraint createUniquenessConstraint(String distinctResourceRef) throws InvalidInputException {
        session.testParameter(distinctResourceRef, 1, null);
         
        Element ucElement = session.createDeploymentElement("UniquenessConstraint");
        ucElement.setAttribute("distinctResourceRef", distinctResourceRef);
        
        return new UniquenessConstraintImpl(ucElement, session);
    }
}