/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi;

import java.util.Collection;

/**
 * If the package provides maintenance for deployed software, MaintenanceInformation declares information about the fix or fixes 
 * provided. If the package content is a single fix, MaintenanceInformation describes the information about that one fix. If the 
 * content is a collection of fixesfor example, a fix packMaintenanceInformation describes each of the fixes provided by the 
 * fix pack.
 */
public interface MaintenanceInformation {
	
    /**
     * Gets the Severity.
     * 
     * This value SHOULD correspond to a severity value used within the SDD providers support system. It serves as a hint to the deployer 
     * about the urgency of applying the described maintenance.
     * 
     * @return A TranslatableString that contains severity of the maintenance content.
     */
    TranslatableString getSeverity();
    
    /**
     * Gets the Categories.
     * 
     * These values SHOULD correspond to maintenance categories within the SDD providers support system.
     * 
     * @return A collection of TranslatableStrings that contain categories of the maintenance content.
     */
    Collection<TranslatableString> getCategory();
    
    /**
     * Gets the Supersedes.
     * 
     * Superseded fixes are ones that fix a problem also fixed by the superseding maintenance package or content element and therefore 
     * need not be applied.<br><br>
     * This element does not indicate whether or not the superseded fix needs to be removed. To indicate that the previous fix must be 
     * removed before the superseding maintenance can be applied successfully; the SDD author can create a requirement stating that the 
     * fix must not be present.<br><br>
     * Superseded fixes MAY include all the information defined in MaintenanceInformationType. At a minimum, a superseded fix MUST 
     * include at least one Fix element with the name of the superseded fix defined.
     * 
     * @return A collection of MaintenanceInformation objects that contain previously released fixes that are superseded by application of this maintenance.
     */
    Collection<MaintenanceInformation> getSupersedes();
    
    /**
     * Gets the FixIdentity.
     * 
     * Fix elements provide information about individual fixes provided by the maintenance content. 
     * 
     * @return A collection of FixIdentity objects that contain included fixes.
     */
    Collection<FixIdentity> getFixIdentityDefinitions();
    
    /**
     * Sets the Severity.
     * 
     * This value SHOULD correspond to a severity value used within the SDD providers support system. It serves as a hint to the deployer 
     * about the urgency of applying the described maintenance.
     * 
     * @param severity A TranslatableString that contains severity of the maintenance content.
     */
    void setSeverity(TranslatableString severity);
    
    /**
     * Sets the Category.
     * 
     * These values SHOULD correspond to maintenance categories within the SDD providers support system.
     * 
     * @param category A collection of TranslatableStrings that contain categories of the maintenance content.
     */
    void setCategory(Collection<TranslatableString> category);
    
    /**
     * Sets the Supersedes.
     * 
     * Superseded fixes are ones that fix a problem also fixed by the superseding maintenance package or content element and therefore 
     * need not be applied.<br><br>
     * This element does not indicate whether or not the superseded fix needs to be removed. To indicate that the previous fix must be 
     * removed before the superseding maintenance can be applied successfully; the SDD author can create a requirement stating that the 
     * fix must not be present.<br><br>
     * Superseded fixes MAY include all the information defined in MaintenanceInformationType. At a minimum, a superseded fix MUST 
     * include at least one Fix element with the name of the superseded fix defined.
     * 
     * @param maintenanceInformation A collection of MaintenanceInformation objects that contain previously released fixes that are 
     * superseded by application of this maintenance.
     */
    void setSupersedes(Collection<MaintenanceInformation> maintenanceInformation);
    
    /**
     * Sets the FixIdentity.
     * 
     * Fix elements provide information about individual fixes provided by the maintenance content. 
     * 
     * @param fixIdentityDefinitions A collection of FixIdentity objects that contain included fix.
     */
    void setFixIdentityDefinitions(Collection<FixIdentity> fixIdentityDefinitions);
    
	/**
	 * Creates a copy of this object.
	 * 
	 * @return A copy of this object. 
	 */
    Object clone();
}
