/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.internal.deployment.sdd.tests.common.validation;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import java.util.Collection;

import junit.framework.TestCase;

import org.eclipse.cosmos.me.internal.deployment.sdd.common.validation.SchemaValidationRule;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.validation.XMLValidationError;
import org.xml.sax.SAXParseException;



public class SchemaValidationRule_Test extends TestCase {
	private String fSep = System.getProperty("file.separator");

	protected void setUp() throws Exception {
		super.setUp();
	}

	protected void tearDown() throws Exception {
		super.tearDown();
	}

	
	/*
	 * Test constructor
	 */
	// Tests creating an instance given an existing, readable Schema file
	public void testSchemaValidationRule_ExistingSchema() throws FileNotFoundException, IOException {
		File[] files = {
				new File(AllTests.COMMON_SCHEMA)
			};
		SchemaValidationRule validationRule = new SchemaValidationRule(Arrays.asList(files));
		
		assertNotNull(validationRule);
	}
	
	// Tests create an instance given a Schema file that doesn't exist
	public void testSchemaValidationRule_MissingSchema() throws IOException {
		
		try {
			File[] files = {
					new File(AllTests.ARTIFACT_LOCATION + fSep + "sdd-common-1.0.xsd-NotHere")
				};
			SchemaValidationRule validationRule = new SchemaValidationRule(Arrays.asList(files));

			fail("Schema is missing but a FileNotFoundException was not thrown");
		} catch (FileNotFoundException e) {
			// This exception is expected
		}
	}
	
	// Tests creating an instance given a readable InputStream
	public void testSchemaValidationRule_ReadableInputStream() throws FileNotFoundException, IOException {
		InputStream[] streams = {
				new FileInputStream(new File(AllTests.COMMON_SCHEMA))
			};
		SchemaValidationRule validationRule = new SchemaValidationRule(Arrays.asList(streams));
		
		assertNotNull(validationRule);
	}
	
	// Tests creating an instance given an unreadable InputStream
	public void testSchemaValidationRule_UnreadableInputStream() throws FileNotFoundException, IOException {
		InputStream[] streams = {
				new FileInputStream(new File(AllTests.COMMON_SCHEMA))
			};
		streams[0].close();
		
		try {
			
			SchemaValidationRule validationRule = new SchemaValidationRule(Arrays.asList(streams));
			
			fail("InputStream is unreadable but an IOException was not thrown");
		}
		catch (IOException e) {
			// This exception is expected
		}
	}
	
	/*
	 * Test adding and removing Schemas
	 */
	// Tests adding from a File
	public void testSchemaValidationRule_AddFromFile() throws FileNotFoundException, IOException {
		SchemaValidationRule validationRule = new SchemaValidationRule(new File(AllTests.PD_SCHEMA));
		File schemaFile = new File(AllTests.COMMON_SCHEMA);
		validationRule.addSchemaFile(schemaFile);
		
		assertEquals(validationRule.getSchemaFiles().size(), 2);
		assertEquals(validationRule.getSchemaFiles().toArray(new File[0])[1], schemaFile);		
	}
	
	// Tests adding from an InputStream
	public void testSchemaValidationRule_AddFromInputStream() throws FileNotFoundException, IOException {
		SchemaValidationRule validationRule = new SchemaValidationRule(new File(AllTests.PD_SCHEMA));
		InputStream schemaStream = new FileInputStream(new File(AllTests.COMMON_SCHEMA));
		validationRule.addSchemaInputStream(schemaStream);
		
		assertEquals(validationRule.getSchemaFiles().size(), 2);
		assertEquals(validationRule.getSchemaInputStreams().size(), 1);
		assertEquals(validationRule.getSchemaInputStreams().toArray(new InputStream[0])[0], schemaStream);	
	}
	
	// Tests removing a File
	public void testSchemaValidationRule_RemoveFile() throws FileNotFoundException, IOException {
		SchemaValidationRule validationRule = new SchemaValidationRule(new File(AllTests.PD_SCHEMA));
		File schemaFile = new File(AllTests.COMMON_SCHEMA);
		validationRule.addSchemaFile(schemaFile);
		
		validationRule.removeSchemaFile(null);
		assertEquals(validationRule.getSchemaFiles().size(), 2);
		
		validationRule.removeSchemaFile(schemaFile);
		assertEquals(validationRule.getSchemaFiles().size(), 1);
	}
	
	// Tests removing an InputStream
	public void testSchemaValidationRule_RemoveInputStream() throws FileNotFoundException, IOException {
		SchemaValidationRule validationRule = new SchemaValidationRule(new File(AllTests.PD_SCHEMA));
		InputStream schemaStream = new FileInputStream(new File(AllTests.COMMON_SCHEMA));
		validationRule.addSchemaInputStream(schemaStream);
		
		validationRule.removeSchemaInputStream(null);
		assertEquals(validationRule.getSchemaFiles().size(), 2);
		assertEquals(validationRule.getSchemaInputStreams().size(), 1);
		
		validationRule.removeSchemaInputStream(schemaStream);
		assertEquals(validationRule.getSchemaFiles().size(), 1);
		assertEquals(validationRule.getSchemaInputStreams().size(), 0);	
	}
	
	/*
	 * Test validation
	 */
	// Tests validating a Deployment Descriptor which is valid against the SDD schema
	public void testValidate_ValidDD() throws FileNotFoundException, IOException {
		File[] files = {
				new File(AllTests.COMMON_SCHEMA),
				new File(AllTests.DD_SCHEMA)
			};
		SchemaValidationRule validationRule = new SchemaValidationRule(Arrays.asList(files));
		
		File xmlFile = new File(AllTests.VALID_DD);
		
		assertEquals(validationRule.validate(new FileInputStream(xmlFile)).size(), 0);
	}

	// Tests validating a Package Descriptor which is valid against the SDD schema	
	public void testValidate_ValidPD() throws FileNotFoundException, IOException {
		File[] files = {
				new File(AllTests.COMMON_SCHEMA),
				new File(AllTests.XMLDSIG_SCHEMA),
				new File(AllTests.PD_SCHEMA)
							
			};
		SchemaValidationRule validationRule = new SchemaValidationRule(Arrays.asList(files));
		
		File xmlFile = new File(AllTests.VALID_PD);
		
		assertEquals(validationRule.validate(new FileInputStream(xmlFile)).size(), 0);
	}

	// Tests validating a Deployment Descriptor which is NOT valid against the SDD schema
	public void testValidate_InvalidDD() throws FileNotFoundException, IOException {
		File[] files = {
				new File(AllTests.COMMON_SCHEMA),
				new File(AllTests.DD_SCHEMA)
			};
		SchemaValidationRule validationRule = new SchemaValidationRule(Arrays.asList(files));
		
		File xmlFile = new File(AllTests.INVALID_DD);
		
		Collection<XMLValidationError> validationErrors = validationRule.validate(new FileInputStream(xmlFile));
		
		assertFalse(validationErrors.size() == 0);
	}

	// Tests validating a Package Descriptor which is NOT valid against the SDD schema
	public void testValidate_InvalidPD() throws FileNotFoundException, IOException {
		File[] files = {
				new File(AllTests.COMMON_SCHEMA),
				new File(AllTests.XMLDSIG_SCHEMA),
				new File(AllTests.PD_SCHEMA)
							
			};
		SchemaValidationRule validationRule = new SchemaValidationRule(Arrays.asList(files));
		
		File xmlFile = new File(AllTests.INVALID_PD);
		
		Collection<XMLValidationError> validationErrors = validationRule.validate(new FileInputStream(xmlFile));
		
		assertFalse(validationErrors.size() == 0);
	}

		
	/*
	 * Test methods from the ErrorHandler interface
	 */
	public void testError() throws FileNotFoundException, IOException {
		File[] files = {
				new File(AllTests.COMMON_SCHEMA)
			};
		SchemaValidationRule validationRule = new SchemaValidationRule(Arrays.asList(files));
		
		validationRule.error(new SAXParseException("Test exception - error()", "publicId", "systemId", 0, 0));
	}

	public void testFatalError() throws FileNotFoundException, IOException {
		File[] files = {
				new File(AllTests.COMMON_SCHEMA)
			};
		SchemaValidationRule validationRule = new SchemaValidationRule(Arrays.asList(files));
		
		validationRule.fatalError(new SAXParseException("Test exception - fatalError()", "publicId", "systemId", 0, 0));
	}

	public void testWarning() throws FileNotFoundException, IOException {
		File[] files = {
				new File(AllTests.COMMON_SCHEMA)
			};
		SchemaValidationRule validationRule = new SchemaValidationRule(Arrays.asList(files));
		
		validationRule.warning(new SAXParseException("Test exception - warning()", "publicId", "systemId", 0, 0));
	}	
}
