/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.databuilders;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;

import org.apache.xerces.xs.ElementPSVI;
import org.apache.xerces.xs.PSVIProvider;
import org.apache.xerces.xs.XSElementDeclaration;
import org.apache.xerces.xs.XSTypeDefinition;
import org.eclipse.cosmos.rm.internal.validation.common.ISMLConstants;
import org.eclipse.cosmos.rm.internal.validation.common.IValidationConstants;
import org.eclipse.cosmos.rm.internal.validation.common.IValidationMessage;
import org.eclipse.cosmos.rm.internal.validation.common.IValidationOutput;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidatorUtil;
import org.eclipse.cosmos.rm.internal.validation.core.IFoundationBuilder;
import org.eclipse.cosmos.rm.internal.validation.core.IValidator;
import org.w3c.dom.Node;
import org.xml.sax.Attributes;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

/**
 * Abstract class that provides a default implementation of the IDataBuilder interface.
 * 
 * @author Sheldon Lee-Loy
 * @author Ali Mehregani
 */
public abstract class AbstractDataBuilder<T> extends DefaultHandler implements IDataBuilder<T> 
{		
	/**
	 * The initialization data
	 */
	private Map<String, Object> init;
	
	/** 
	 * Stores line and column information
	 */
	private Locator locator;
	
	/**
	 * The target namespace
	 */
	private String targetNamespace;
	
	/**
	 * Stores the default namespace
	 */
	private String defaultNamespace;
	
	/**
	 * The prefix-namespace mapping
	 * KEY = A {@link String} indicating a prefix
	 * VALUE = A {@link String} indicating a URL
	 */
	private Map<String, String> prefixMap;
	
	/**
	 * The validity of the structure generated.  Set to true by
	 * default.
	 */
	private boolean validity;
	
	/**
	 * The error message
	 */
	private IValidationMessage errorMessage;

	/**
	 * The file path currently being processed
	 */
	private String filePath;

	/**
	 * The supported events of this data builder
	 */
	private int supportedEvents;
	
	/**
	 * The outputter associated with this data builder
	 */
	private IValidationOutput<?,?> messageOutputter;

	/**
	 * PSVI provider of this data builder
	 */
	private PSVIProvider psvi;

	/**
	 * The base line number
	 */
	private int baseLineNumber;
	
	/**
	 * The aliases of the current document being parsed
	 */
	private String[] aliases;
	
	/**
	 * Indicates if the current element belongs to the SML namespace
	 */
	private boolean smlElement;
	
	/**
	 * Indicates if the current element belongs to the SML-IF namespace
	 */
	private boolean smlifElement;
	
	/**
	 * Constructor
	 */
	public AbstractDataBuilder()
	{
		setStructureValidity(true);
		targetNamespace = "";
		prefixMap = new Hashtable<String, String>();
	}
	
	
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.IDataBuilder#reset()
	 */
	public void reset()
	{
		// Empty content - Subclasses are expected to
		// overwrite
	}
	
	
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.IDataBuilder#initialize(java.util.Map)
	 */
	public void initialize(Map<String, Object> init)
	{
		this.init = init;
		this.psvi = (PSVIProvider)init.get(IValidator.ATTRIBUTE_PSVI_INSTANCE);
		this.messageOutputter = (IValidationOutput<?,?>) init.get(IValidator.ATTRIBUTE_OUTPUT);
		Object tmp = init.get(IValidator.ATTRIBUTE_BASE_LINE_NUMBER);
		this.baseLineNumber = tmp == null ? 0 : (Integer) tmp;
		tmp = init.get(IValidator.ATTRIBUTE_DOCUMENT_ALIASES);
		this.aliases = tmp == null ? new String[0] : (String[])tmp;
	}
	
	
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.IDataBuilder#endDefinitions()
	 */
	public void endDefinitions() 
	{
		// no-op
	}

	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.IDataBuilder#endInstances()
	 */
	public void endInstances() 
	{
		// no-op
	}


	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.IDataBuilder#startDefinitions()
	 */
	public void startDefinitions() 
	{
	}
	
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.IDataBuilder#startInstances()
	 */
	public void startInstances() 
	{		
	}
	

	/**
	 * This is used to determine the target name space of the document currently
	 * being parsed
	 * 
	 * @see org.xml.sax.helpers.DefaultHandler#startElement(java.lang.String, java.lang.String, java.lang.String, org.xml.sax.Attributes)
	 */
	public void startElement(String uri, String localName, String qName, Attributes attributes) throws SAXException
	{
		smlifElement = ISMLConstants.SMLIF_URI.equals(uri);
		smlElement = ISMLConstants.SML_URI.equals(uri);
		
		if (ISMLConstants.SCHEMA_URI.equals(uri) && ISMLConstants.SCHEMA_ELEMENT.equals(localName))
		{
			targetNamespace = attributes.getValue(ISMLConstants.TARGET_NAMESPACE_ATTRIBUTE);
			
			for (int i = 0, attributeCount = attributes.getLength(); i < attributeCount; i++)
			{
				String[] attributeName = SMLValidatorUtil.tokenizeName(attributes.getQName(i));						
				if (ISMLConstants.XML_NS_ATTRIBUTE.equals(attributeName[0]))
				{													
					prefixMap.put(attributeName[1], attributes.getValue(i));			
				}
				else if (ISMLConstants.XML_NS_ATTRIBUTE.equals(attributeName[1]))
				{
					defaultNamespace = attributes.getValue(i);
				}
			}
		}
	
	}
	
	
	/**
	 * @see org.xml.sax.helpers.DefaultHandler#endElement(java.lang.String, java.lang.String, java.lang.String)
	 */
	public void endElement(String uri, String localName, String qName) throws SAXException
	{
		if (ISMLConstants.SCHEMA_URI.equals(uri) && ISMLConstants.SCHEMA_ELEMENT.equals(localName))
		{
			targetNamespace = "";
			prefixMap.clear();
		}
	}
	
	
	/**
	 * @see org.xml.sax.helpers.DefaultHandler#setDocumentLocator(org.xml.sax.Locator)
	 */
	public void setDocumentLocator(Locator locator) 
	{
		this.locator = locator;
	}
	
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.IDataBuilder#isStructureValid()
	 */
	public boolean isStructureValid()
	{		
		return validity;
	}
	
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.IDataBuilder#setStructureValidity(boolean)
	 */
	public void setStructureValidity(boolean validity)
	{
		this.validity = validity;
	}
	
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.IDataBuilder#getErrorMessage()
	 */
	public IValidationMessage getErrorMessage()
	{
		return errorMessage;
	}	
	
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.IDataBuilder#setErrorMessage(java.lang.String)
	 */
	public void setErrorMessage(IValidationMessage error)
	{
		this.errorMessage = error;
	}
		
	/**
	 * Appends to the error message
	 * 
	 * @param error The error, which is appended as a separate line
	 */
	protected void appendToErrorMessage(String error)
	{
		String errMsg = errorMessage.getAttribute(IValidationMessage.ATTRIBUTE_MESSAGE, null);
		if (errMsg == null) {
			errMsg = "";
		}
		
		if (errMsg.length() > 0) {
			errMsg += IValidationConstants.LINE_SEPARATOR;
		}
		
		errMsg += error;
	}

	/**
	 * @return the prefix
	 */
	public Map<String, String> getPrefixMap()
	{
		return prefixMap;
	}

	/**
	 * @param prefix the prefix to set
	 */
	public void setPrefixMap(Map<String, String> prefix)
	{
		this.prefixMap = prefix;
	}

	/**
	 * @return the targetNamespace
	 */
	public String getTargetNamespace()
	{
		return targetNamespace;
	}

	/**
	 * @param targetNamespace the targetNamespace to set
	 */
	public void setTargetNamespace(String targetNamespace)
	{
		this.targetNamespace = targetNamespace;
	}

	/**
	 * @return the defaultNamespace
	 */
	public String getDefaultNamespace()
	{
		return defaultNamespace;
	}

	/**
	 * @param defaultNamespace the defaultNamespace to set
	 */
	public void setDefaultNamespace(String defaultNamespace)
	{
		this.defaultNamespace = defaultNamespace;
	}
	

	
	/**
	 * Sets the path of the current file being processed
	 * 
	 * @param filePath The path
	 */
	public void setFilePath(String filePath)
	{
		this.filePath = filePath;
	}
	
	
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.IDataBuilder#getFilePath()
	 */
	public String getFilePath()
	{
		return filePath;
	}

	
	/**
	 * Subclasses are expected to overwrite this method to process
	 * comments if the associated event is supported.
	 * 
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.IDataBuilder#comment(char[], int, int)
	 */
	public void comment(char[] cs, int i, int j) 
	{
	}
	
	
	/**
	 * Subclasses are expected to add events for processing different
	 * sections of a document.  See a list of event constants under 
	 * {@link IFoundationBuilder} for a list of events that are supported.
	 * 
	 * @param event The event to be added.
	 */
	protected void addEvent(int event)
	{
		addEvents(new int[]{event});
	}
	
	/**
	 * Same semantics as {@link #addEvent(int)} but with the
	 * ability to add multiple events.
	 * 
	 * @param events The events to be added
	 */
	protected void addEvents(int[] events)
	{
		for (int i = 0; i < events.length; i++)
		{
			supportedEvents = supportedEvents | (1 << events[i]);
		}
	}
	
	
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.IDataBuilder#getSupportedEvents()
	 */
	public int getSupportedEvents()
	{
		return supportedEvents;	
	}
	
	
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.IDataBuilder#getMessageOutputter()
	 */
	public IValidationOutput<?, ?> getMessageOutputter()
	{	
		return messageOutputter;
	}
	
	
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.IDataBuilder#setMessageOutputter(org.eclipse.cosmos.rm.internal.validation.common.IValidationOutput)
	 */
	public void setMessageOutputter(IValidationOutput<?, ?> messageOutputter)
	{
		this.messageOutputter = messageOutputter;
	}


	/**
	 * @return the locator
	 */
	public Locator getLocator()
	{
		return locator;
	}


	/**
	 * @param locator the locator to set
	 */
	public void setLocator(Locator locator)
	{
		this.locator = locator;
	}


	/**
	 * @return the init
	 */
	public Map<String, Object> getInit()
	{
		return init;
	}


	/**
	 * @return the psvi
	 */
	public PSVIProvider getPsvi()
	{
		return psvi;
	}


	/**
	 * @param psvi the psvi to set
	 */
	public void setPsvi(PSVIProvider psvi)
	{
		this.psvi = psvi;
	}

	
	/**
	 * @return the baseLineNumber
	 */
	public int getBaseLineNumber()
	{
		return baseLineNumber;
	}


	/**
	 * @param baseLineNumber the baseLineNumber to set
	 */
	public void setBaseLineNumber(int baseLineNumber)
	{
		this.baseLineNumber = baseLineNumber;
	}


	/**
	 * @return the aliases
	 */
	public String[] getAliases()
	{
		return aliases;
	}


	/**
	 * @param aliases the aliases to set
	 */
	public void setAliases(String[] aliases)
	{
		this.aliases = aliases;
	}


	public AnnotationResult retrieveElementAnnotation(String uri, String localName) 
	{
		ElementPSVI elementPSVI = getPsvi().getElementPSVI();
		XSElementDeclaration elementDeclaration = elementPSVI == null ? null : elementPSVI.getElementDeclaration();
		return SMLValidatorUtil.retrieveAnnotation(elementDeclaration, uri, localName);
	}


	/**
	 * @return the smlElement
	 */
	public boolean isSMLElement()
	{
		return smlElement;
	}


	/**
	 * @return the smlifElement
	 */
	public boolean isSMLIFElement()
	{
		return smlifElement;
	}
	
	
	/**
	 * Returns the line number of the current context
	 * 
	 * @return The line number of current context
	 */
	public int getContextLineNumber()
	{
		return getBaseLineNumber() + getLocator().getLineNumber() - 1;
	}


	public static class AnnotationResult
	{
		private List<Node> nodes;
		private XSTypeDefinition type;
		
		public AnnotationResult()
		{
			nodes = new ArrayList<Node>();
		}
		
		/**
		 * @return the node
		 */
		public Node[] getNodes()
		{
			return nodes.toArray(new Node[nodes.size()]);
		}
		/**
		 * @param node the node to set
		 */
		public void addNode(Node node)
		{
			this.nodes.add(node);
		}
		/**
		 * @return the type
		 */
		public XSTypeDefinition getType()
		{
			return type;
		}
		/**
		 * @param type the type to set
		 */
		public void setType(XSTypeDefinition type)
		{
			this.type = type;
		}		
	}
}
