/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.internal.deployment.sdd.common.spi;

import java.util.ArrayList;
import java.util.Collection;

import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.contentunit.CompositeInstallableImpl;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.contentunit.ConfigurationUnitImpl;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.contentunit.InstallableUnitImpl;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.contentunit.LocalizationUnitImpl;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.DeploymentDescriptor;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.Resource;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.CompositeInstallable;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.ContentUnit;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.exception.InvalidInputException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

public class DeploymentDescriptorImpl extends DescriptorImpl implements DeploymentDescriptor {
	private static final String DEPLOYMENT_DESCRIPTOR_TYPE = "sdd-dd:DeploymentDescriptorType";
	private static final String TOPOLOGY_TYPE = "sdd-dd:TopologyType";
	
	public DeploymentDescriptorImpl(Document doc, SPISessionImpl session) {
		super(doc, session, DEPLOYMENT_DESCRIPTOR_TYPE);
	}
	
	public ContentUnit getContentUnit() {
		ContentUnit contentUnit = null;
		
		if (isSetElement("InstallableUnit")) {
		    Element installableUnit = getChild("InstallableUnit");
			contentUnit = new InstallableUnitImpl(installableUnit, session);
		}
		else if (isSetElement("LocalizationUnit")) {
            Element localizationUnit =  getChild("LocalizationUnit");
            contentUnit = new LocalizationUnitImpl(localizationUnit, session);
		}
		else if (isSetElement("ConfigurationUnit")) {
            Element configurationUnit = getChild("ConfigurationUnit");
            contentUnit = new ConfigurationUnitImpl(configurationUnit, session);
		}
			
		return contentUnit;
	}

	public Collection<CompositeInstallable> getCompositeInstallables() {
	    Collection<Element> composites = getChildren("CompositeInstallable");
	    
	    Collection <CompositeInstallable> compositeInstallables = new ArrayList<CompositeInstallable>();

	    for (Element composite : composites) {
	        compositeInstallables.add(new CompositeInstallableImpl(composite, session));
	    }
	    
	    return compositeInstallables;
	}
	
	public Collection<Resource> getResources() {
	    Collection<Resource> resources = new ArrayList<Resource>();

	    if (isSetElement("Topology")) {
	    	Element topology = getChild("Topology");
	        Collection<Element> tmpList = session.getChildren(topology, "Resource");

	        for (Element element : tmpList)
            {
	            resources.add(new ResourceImpl(element, session));
	        }
	    }
		
		return resources;
	}

	public void setResources(Collection<Resource> resources) {
		session.testParameter(resources, 1, null);
		
		if (resources == null) {
			unSetElement("Topology");
		}
		else {
			if (!isSetElement("Topology")) {
				Element topology = session.createDeploymentElement("Topology");
				setContent(topology);
			}
			
			Element topology = getChild("Topology");
			
			for (Resource resource : resources) {
				if (resource == null) {
					throw new InvalidInputException(1, InvalidInputException.NULL_ARRAY_ELEMENT);
				}

				session.insertNode(topology, (SPIDataObject) resource, TOPOLOGY_TYPE);
			}
			
			unSetElement("Topology");
			setContent(topology);
		}
	}

	public void setContentUnit(ContentUnit cu) {
        session.testParameter(cu, 1, null);

        unSetElement("InstallableUnit");
        unSetElement("ConfigurationUnit");
        unSetElement("LocalizationUnit");
        unSetElement("CompositeUnit");

        setContent((SPIDataObject)cu);
    }
	
	public void setCompositeInstallables(Collection<CompositeInstallable> ci) {
        session.testParameter(ci, 1, null);
        
        if (ci == null) {
        	unSetElement("CompositeInstallable");
        }
        else {
        	removeContent(getChildren("CompositeInstallable", false));
        
        	addContent(ci);
        }
    }
	
	public String toString() {
		String summary = "Deployment Descriptor\n" + super.toString();

		summary += "\nContent Unit\n------------------------------------\n";
		summary += getContentUnit().toString();

		summary += "\n\nTopology\n------------------------------------\n";
		Collection<Resource> topologyResources = getResources();
		for (Resource resource : topologyResources) {
			summary += resource.toString();
		}

		return summary;
	}
	
	public Object clone() {
		return new DeploymentDescriptorImpl((Document)document.cloneNode(true), session);
	}
}
