/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.internal.deployment.sdd.tests.common.spi;

import java.util.Collection;
import java.util.Properties;

import junit.framework.TestCase;

import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.SPIDataObject;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.SPISessionImpl;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.SPISession;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.TranslatableString;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.requirement.RelationshipConstraint;
import org.w3c.dom.Element;

public class RelationshipConstraint_Test extends TestCase {
    RelationshipConstraint relConstraint;
    RelationshipConstraint clonedRelConstraint;
    Element element;
    SPISessionImpl session;

    public RelationshipConstraint_Test(String name) {
        super(name);
    }

    protected void setUp() throws Exception {
        super.setUp();
        
        relConstraint = SPISession.DEFAULT_INSTANCE.createRequirementFactory().createRelationshipConstraint("constraintType");
        relConstraint.setDescription(new TranslatableString("desc", "descKey"));
        relConstraint.setShortDescription(new TranslatableString("shortDesc", "shortDescKey"));
        
        Properties props = new Properties();
        props.setProperty("key1", "value1");
        props.setProperty("key2", "value2");
        relConstraint.setProperties(props);
        
        relConstraint.setRelatedResourceRef("relResourceRef");
        
        clonedRelConstraint = (RelationshipConstraint)relConstraint.clone();
        element = ((SPIDataObject)relConstraint).getElement();
        session = (SPISessionImpl)SPISession.DEFAULT_INSTANCE;
    }

    public void testGetTranslatableStrings() {
        Properties props = relConstraint.getTranslatableStrings();
        Properties clonedProps = clonedRelConstraint.getTranslatableStrings();
        
        assertEquals(2, props.keySet().size());
        assertEquals(2, clonedProps.keySet().size());
        
        assertEquals("desc", props.get("descKey"));
        assertEquals("shortDesc", props.get("shortDescKey"));

        assertEquals("desc", clonedProps.get("descKey"));
        assertEquals("shortDesc", clonedProps.get("shortDescKey"));
    }

    public void testGetProperties() {
        Properties testProps = relConstraint.getProperties();
        Properties testclonedProps = clonedRelConstraint.getProperties();
        Collection<Element> testDataObjectProps = session.getChildren(element, "Property");
        
        assertEquals("value1", testProps.get("key1"));
        assertEquals("value2", testProps.get("key2"));
        assertEquals("value1", testclonedProps.get("key1"));
        assertEquals("value2", testclonedProps.get("key2"));
        
        assertEquals(2, testDataObjectProps.size());
        
        for (int i = 0; i < testDataObjectProps.size(); i++) {
            Element prop = testDataObjectProps.toArray(new Element[0])[i];
            
            String value = session.getChildren(prop, "Value").toArray(new Element[0])[0].getTextContent();
            String propertyName = session.getChildren(prop, "PropertyName").toArray(new Element[0])[0].getTextContent();
            
            assertTrue(value + " wasn't found in the properties object", testProps.contains(value));
            assertTrue(propertyName + " wasn't found in the properties object", testProps.containsKey(propertyName));
            assertEquals(value, testProps.get(propertyName));
        }
    }

    public void testGetRelatedResourceReference() {
        assertEquals("relResourceRef", relConstraint.getRelatedResourceReference());
        assertEquals("relResourceRef", clonedRelConstraint.getRelatedResourceReference());
        assertEquals("relResourceRef", element.getAttribute("relatedResourceRef"));
    }

    public void testGetType() {
        assertEquals("constraintType", relConstraint.getType());
        assertEquals("constraintType", clonedRelConstraint.getType());
        assertEquals("constraintType", element.getAttribute("type"));
    }

    public void testGetDescription() {
        assertEquals("desc", relConstraint.getDescription().getValue());
        assertEquals("descKey", relConstraint.getDescription().getTranslationKey());

        assertEquals("desc", clonedRelConstraint.getDescription().getValue());
        assertEquals("descKey", clonedRelConstraint.getDescription().getTranslationKey());
        
        assertEquals("desc", session.getChildren(element, "Description").toArray(new Element[0])[0].getTextContent());
        assertEquals("descKey", (session.getChildren(element, "Description").toArray(new Element[0])[0]).getAttribute("translationKey"));
    }

    public void testGetShortDescription() {
        assertEquals("shortDesc", relConstraint.getShortDescription().getValue());
        assertEquals("shortDescKey", relConstraint.getShortDescription().getTranslationKey());

        assertEquals("shortDesc", clonedRelConstraint.getShortDescription().getValue());
        assertEquals("shortDescKey", clonedRelConstraint.getShortDescription().getTranslationKey());
        
        assertEquals("shortDesc", session.getChildren(element, "ShortDescription").toArray(new Element[0])[0].getTextContent());
        assertEquals("shortDescKey", (session.getChildren(element, "ShortDescription").toArray(new Element[0])[0]).getAttribute("translationKey"));
    }

    public void testSetProperties() {
        Properties props = new Properties();
        props.setProperty("key3", "value3");
        relConstraint.setProperties(props);

        Properties testProps = relConstraint.getProperties();
        Collection<Element> testDataObjectProps = session.getChildren(element, "Property");
        
        assertEquals(1, testDataObjectProps.size());

        assertEquals("value3", testProps.get("key3"));
        
        for (int i = 0; i < testDataObjectProps.size(); i++) {
            Element prop = testDataObjectProps.toArray(new Element[0])[i];
            
            String value = session.getChildren(prop, "Value").toArray(new Element[0])[0].getTextContent();
            String propertyName = session.getChildren(prop, "PropertyName").toArray(new Element[0])[0].getTextContent();
            
            assertTrue(value + " wasn't found in the properties object", testProps.contains(value));
            assertTrue(propertyName + " wasn't found in the properties object", testProps.containsKey(propertyName));
            assertEquals(value, testProps.get(propertyName));
        }
    }

    public void testSetRelatedResourceRef() {
        relConstraint.setRelatedResourceRef("differentRef");
        
        assertEquals("differentRef", relConstraint.getRelatedResourceReference());
        assertEquals("differentRef", element.getAttribute("relatedResourceRef"));
    }

    public void testSetType() {
        relConstraint.setType("differentType");
        
        assertEquals("differentType", relConstraint.getType());
        assertEquals("differentType", element.getAttribute("type"));
        
        try {
            relConstraint.setType(null);
            
            fail("Type was set to null");
        }
        catch (Exception e) {
        }
    }

    public void testSetDescription() {
        relConstraint.setDescription(new TranslatableString("differentDesc", "differentDescKey"));
        
        assertEquals("differentDesc", relConstraint.getDescription().getValue());
        assertEquals("differentDescKey", relConstraint.getDescription().getTranslationKey());

        assertEquals("differentDesc", session.getChildren(element, "Description").toArray(new Element[0])[0].getTextContent());
        assertEquals("differentDescKey", (session.getChildren(element, "Description").toArray(new Element[0])[0]).getAttribute("translationKey"));
    }

    public void testSetShortDescription() {
        relConstraint.setShortDescription(new TranslatableString("differentShortDesc", "differentShortDescKey"));

        assertEquals("differentShortDesc", relConstraint.getShortDescription().getValue());
        assertEquals("differentShortDescKey", relConstraint.getShortDescription().getTranslationKey());

        assertEquals("differentShortDesc", session.getChildren(element, "ShortDescription").toArray(new Element[0])[0].getTextContent());
        assertEquals("differentShortDescKey", (session.getChildren(element, "ShortDescription").toArray(new Element[0])[0]).getAttribute("translationKey"));
    }
}
