/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit;

import java.util.Collection;

import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.artifact.Artifact;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.exception.InvalidInputException;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.requirement.Requirement;

/**
 * The ConfigurationUnit element defines artifacts that configure one or more existing resources. It also defines the 
 * requirements for applying those artifacts. It MUST NOT install, update, or uninstall resources.
 */
public interface ConfigurationUnit extends ContentUnit {
	/**
	 * Gets the target resource reference.
	 * 
	 * The targetResourceRef attribute identifies the resource in Topology that will process the ConfigurationUnits 
	 * artifacts to configure the resources identified by the ConfigurationUnits ResultingChange definition.
	 * 
	 * @return Reference to the resource that can process the ConfigurationUnits artifacts.
	 */
	String getTargetResourceReference();
	
	/**
	 * Sets the target resource reference.
	 * 
	 * The targetResourceRef attribute identifies the resource in Topology that will process the ConfigurationUnits 
	 * artifacts to configure the resources identified by the ConfigurationUnits ResultingChange definition.
	 * 
	 * @param targetResourceReference Reference to the resource that can process the ConfigurationUnits artifacts.
	 * @throws InvalidInputException if the parameter is null
	 */
	void setTargetResourceReference(String targetResourceReference);

	
	/**
	 * Gets the configuration artifact.
	 * 
	 * When the ConfigurationUnit is a singleton defined outside of a CompositeInstallable, it MUST define at least one artifact 
	 * element. The inclusion of an artifact element in a singleton ConfigurationUnit implies support for the associated operation.<br><br>
	 * When the ConfigurationUnit is defined within a CompositeInstallable, it MUST define exactly one artifact. The artifact 
	 * defined MUST be a ConfigArtifact and it MUST support the single top level operation defined by the CompositeInstallable. 

	 * @return The artifact associated with the ConfigurationUnit.
	 */
	Artifact getConfigurationArtifact();
	
	/**
	 * Gets the completion element.
	 * 
	 * A Completion element MUST be included if the artifact being processed requires a system operation such as a reboot or logoff to 
	 * occur to function successfully after deployment or if the artifact executes a system operation to complete deployment of the 
	 * contents of the artifact.<br><br>
	 * 
	 * There MUST be an artifact associated with the operation defined by a Completion element. 
	 * <ul>For example, if there is a Completion element for the configure operation, the ConfigurationUnit must define a ConfigArtifact.</ul> 
     *
	 * @return A collection of Completion objects that describe completion actions such as restart and the conditions under which the action is applied.
	 */
	Collection<Completion> getCompletion();
	
	/**
	 * Gets the requirements.
	 * 
	 * Requirements specified in a ConfigurationUnit identify requirements that MUST be met prior to successful processing of the ConfigurationUnits artifacts.
	 *  
	 * @return Requirements that must be met prior to successful processing of the ConfigurationUnits artifacts.
	 */
	Collection<Requirement> getRequirements();
	
	/**
	 * Gets the resulting changes.
	 * 
	 * Configuration changes made when the configuration artifact is processed SHOULD be declared here. This information may 
	 * be necessary when the SDD is aggregated into another SDD and the resulting change satisfies a constraint in the aggregation. 
	 * The information declared here can be compared with resource constraints to determine if application of the ConfigurationUnit 
	 * will satisfy the constraint.
	 * 
	 * @return A collection of ResultingChange objects that define changes made to a resource that is configured by processing the ConfigurationUnits ConfigArtifact.
	 */
	Collection<ResultingChange> getResultingChanges();
	
	/**
	 * Sets the completion elements.
	 * 
	 * A Completion element MUST be included if the artifact being processed requires a system operation such as a reboot or logoff to 
	 * occur to function successfully after deployment or if the artifact executes a system operation to complete deployment of the 
	 * contents of the artifact.<br><br>
	 * 
	 * There MUST be an artifact associated with the operation defined by a Completion element. 
	 * <ul>For example, if there is a Completion element for the configure operation, the ConfigurationUnit must define a ConfigArtifact.</ul> 
     *
	 * @param configurationUnitCompletion  A collection of Completion objects that describe completion actions such as restart and the conditions under which the action is applied.
	 */
	void setCompletion(Collection<Completion> configurationUnitCompletion);
	
	/**
	 * Sets the requirements.
	 * 
	 * Requirements specified in a ConfigurationUnit identify requirements that MUST be met prior to successful processing of the ConfigurationUnits artifacts.
	 *  
	 * @param requirements Requirements that must be met prior to successful processing of the ConfigurationUnits artifacts.
	 */
	void setRequirements(Collection<Requirement> requirements);
	
	/**
	 * Sets the resulting changes.
	 * 
	 * Configuration changes made when the configuration artifact is processed SHOULD be declared here. This information may 
	 * be necessary when the SDD is aggregated into another SDD and the resulting change satisfies a constraint in the aggregation. 
	 * The information declared here can be compared with resource constraints to determine if application of the ConfigurationUnit 
	 * will satisfy the constraint.
	 * 
	 * @param resultingChanges  A collection of ResultingChange objects that define changes made to a resource that is configured by processing the ConfigurationUnits ConfigArtifact.
	 */
	void setResultingChanges(Collection<ResultingChange> resultingChanges);

	/**
	 * Sets the configuration artifact.
	 * 
	 * When the ConfigurationUnit is a singleton defined outside of a CompositeInstallable, it MUST define at least one artifact 
	 * element. The inclusion of an artifact element in a singleton ConfigurationUnit implies support for the associated operation.<br><br>
	 * When the ConfigurationUnit is defined within a CompositeInstallable, it MUST define exactly one artifact. The artifact 
	 * defined MUST be a ConfigArtifact and it MUST support the single top level operation defined by the CompositeInstallable.
	 *  
	 * @param configurationArtifact  The artifact associated with the ConfigurationUnit.
	 * @throws InvalidInputException if the parameter is null
	 */
	void setConfigurationArtifact(Artifact configurationArtifact);
}
