/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.rm.internal.smlif.export.ui;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.eclipse.core.resources.IFile;
import org.eclipse.cosmos.rm.internal.repository.SMLRepositoryUtil;
import org.eclipse.cosmos.rm.internal.smlif.common.SMLCommonUtil;
import org.eclipse.cosmos.rm.internal.smlif.common.SMLConstants;
import org.eclipse.cosmos.rm.internal.smlif.common.SMLMessages;
import org.eclipse.cosmos.rm.internal.validation.artifacts.RuleBinding;
import org.eclipse.cosmos.rm.internal.validation.common.ISMLConstants;
import org.eclipse.cosmos.rm.provisional.repository.resource.ISMLDocument;
import org.eclipse.cosmos.rm.provisional.repository.resource.ISMLMetadata;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.ColumnPixelData;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ControlAdapter;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.events.ControlListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.ScrollBar;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.xml.sax.SAXException;
import org.xml.sax.SAXNotRecognizedException;
import org.xml.sax.SAXNotSupportedException;

/**
 * Third page of export wizard, containing rule binding mappings
 * 
 * @author David Whiteman
 */
public class ExportToSMLIFWizardPage3 extends WizardPage {

	protected static final String DOCUMENT_COLUMN = "DOCUMENT_COLUMN"; //$NON-NLS-1$
	protected static final String RULE_COLUMN = "RULE_COLUMN"; //$NON-NLS-1$
	protected static final String ALIAS_COLUMN = "ALIAS_COLUMN"; //$NON-NLS-1$

	//protected ComboBoxCellEditor documentChoicesCombo;
	protected TableViewer ruleBindingsTableViewer;
	protected EditableComboBoxCellEditor aliasChoicesCombo;
	protected EditableComboBoxCellEditor ruleChoicesCombo;
	protected List<RuleBinding> ruleBindings;
	
	
	/**
	 * List of ModelUnitFile instances the RuleBindingTableCellModifier uses for looking
	 * up a document by its index in the combo box editor
	 */
	protected List<ISMLDocument> documents;

	public ExportToSMLIFWizardPage3(String pageName,
			IStructuredSelection selection) {
		super(pageName);
		setTitle(pageName);
	}

	public void createControl(Composite parent) {
		Composite topComponent = new Composite(parent, SWT.NONE);
		GridData topComponentData = new GridData(GridData.GRAB_HORIZONTAL
				| GridData.FILL_HORIZONTAL);
		topComponent.setLayoutData(topComponentData);

		GridLayout layout = new GridLayout();
		layout.numColumns = 2;
		layout.makeColumnsEqualWidth = false;
		layout.marginHeight = 0;
		layout.marginWidth = 0;
		topComponent.setLayout(layout);

		createRuleBindingsLabel(topComponent);
		createRuleBindingsTable(topComponent);
		createAddRemoveButtons(topComponent);

		setControl(topComponent);

		setPageComplete(true);
		Dialog.applyDialogFont(topComponent);
	}

	protected void createAddRemoveButtons(Composite parent) {
		Composite buttonComposite = ExportToSMLIFWizard.createVerticalButtonPanel(parent);
		Button addButton = ExportToSMLIFWizard.createVerticalButtonPanelButton(buttonComposite,
				SMLMessages.exportWizardAddButtonLabelNonPrompting);
		
		addButton.addSelectionListener(new SelectionAdapter() {
			@SuppressWarnings("unchecked")
			public void widgetSelected(SelectionEvent arg0) {
				RuleBinding newRuleBinding = new RuleBinding();
				((Collection<RuleBinding>) ruleBindingsTableViewer.getInput()).add(newRuleBinding);
				ruleBindingsTableViewer.refresh();
				// https://bugs.eclipse.org/bugs/show_bug.cgi?id=186487
				// Put editor in first typeable cell
				ruleBindingsTableViewer.editElement(newRuleBinding, 0);
			}
		});
		Button removeButton = ExportToSMLIFWizard.createVerticalButtonPanelButton(buttonComposite,
				SMLMessages.exportWizardRemoveButtonText);
		removeButton.addSelectionListener(new SelectionAdapter() {
			@SuppressWarnings("unchecked")
			public void widgetSelected(SelectionEvent arg0) {
				IStructuredSelection selection = (IStructuredSelection) ruleBindingsTableViewer.getSelection();
				((Collection<RuleBinding>) ruleBindingsTableViewer.getInput()).removeAll(selection.toList());
				ruleBindingsTableViewer.refresh();
			}
		});
	}

	protected void createRuleBindingsTable(Composite parent) {
		ruleBindingsTableViewer = new TableViewer(parent, SWT.MULTI
				| SWT.FULL_SELECTION);
		setTableProperties();
		ControlListener listener = createTableControlListener(ruleBindingsTableViewer
				.getTable());
		parent.addControlListener(listener);
	}
	
	protected void setTableProperties() {
		TableLayout layout = new TableLayout();
		Table ruleBindingsTable = ruleBindingsTableViewer.getTable();
		ruleBindingsTable.setLayout(layout);
		ruleBindingsTable.setLinesVisible(true);
		ruleBindingsTable.setLayoutData(new GridData(SWT.FILL, SWT.DEFAULT, true, false));
		ruleBindingsTable.setHeaderVisible(true);				
		GridData gridData = new GridData(SWT.FILL, SWT.FILL, true, true);
		gridData.heightHint = 50;
		ruleBindingsTable.setLayoutData(gridData);
				
		String[] columnNames = {/*SMLMessages.exportWizardRuleTableDocumentColumn,*/ SMLMessages.exportWizardRuleTableAliasColumn, SMLMessages.exportWizardRuleTableRuleColumn};
		String[] columnProperties = {/*DOCUMENT_COLUMN,*/ ALIAS_COLUMN, RULE_COLUMN};
		ColumnPixelData columnLayout = new ColumnPixelData(180, true);
		TableColumn column = null;

		for (int i = 0; i < columnNames.length; i++) {
			column = new TableColumn(ruleBindingsTable, SWT.LEFT);
			column.setResizable(columnLayout.resizable);
			column.setText(columnNames[i]);	
			layout.addColumnData(columnLayout);
		}

		//documentChoicesCombo = new ComboBoxCellEditor(ruleBindingsTable, new String[] {}, SWT.DROP_DOWN | SWT.READ_ONLY);
		aliasChoicesCombo = new EditableComboBoxCellEditor(ruleBindingsTable, new String[] {}, SWT.DROP_DOWN);
		ruleChoicesCombo = new EditableComboBoxCellEditor(ruleBindingsTable, new String[] {}, SWT.DROP_DOWN);
		/* Add the cell editors */
		ruleBindingsTableViewer.setCellEditors(new CellEditor[]{
				//documentChoicesCombo,
				aliasChoicesCombo,
				ruleChoicesCombo});
		
		ruleBindingsTableViewer.setCellModifier(new RuleBindingTableCellModifier(Arrays.asList(columnProperties), this, getExportWizard().getPage2().getAliases()));
		ruleBindingsTableViewer.setColumnProperties(columnProperties);
		
		updateComboChoices();
		
		ruleBindingsTableViewer.setLabelProvider(new RuleBindingTableLabelProvider());
		ruleBindingsTableViewer.setContentProvider(new ArrayContentProvider());
		ruleBindingsTableViewer.setInput(ruleBindings);
		ruleBindingsTable.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent arg0) {
				IStructuredSelection tableSelection = (IStructuredSelection) ruleBindingsTableViewer.getSelection();
				RuleBinding selectedBinding = (RuleBinding) tableSelection.getFirstElement();

				if (selectedBinding == null)
					return;

				aliasChoicesCombo.setValue(selectedBinding.getAlias());
				ruleChoicesCombo.setValue(selectedBinding.getRule());
			}
		});
	}

	@SuppressWarnings("unchecked")
	protected void updateComboChoices() {
		if (aliasChoicesCombo == null) {
			return;
		}
		List<String> allAliases = new ArrayList<String>();
		allAliases.add(SMLConstants.ANY_DOCUMENT_ALIAS);
		documents = getDocuments();
		for (Iterator iterator = documents.iterator(); iterator
				.hasNext();) {
			ISMLDocument smlDoc = (ISMLDocument) iterator.next();
			String[] smlDocAliases = smlDoc.getMetadata().getAliases();
			for (int i = 0; i < smlDocAliases.length; i++) {
				allAliases.add(smlDocAliases[i]);
			}
		}
		
		Collections.sort(allAliases);
		String[] aliases = allAliases.toArray(new String[0]);
		aliasChoicesCombo.setItems(aliases);

		ruleChoicesCombo.setItems(getRuleListChoices());
	}

	protected final int calculateRemainingHorizontalSpace(Table table) {
		Rectangle bounds = table.getBounds();
		boolean visible = bounds.width != 0 || bounds.height != 0;
		if (visible == false)
			return 0;

		int width = bounds.width;
		int tableBorderWidth = table.getBorderWidth();
		width -= tableBorderWidth * 2;

		ScrollBar scrollbar = table.getVerticalBar();
		Point scrollbarSize = scrollbar.getSize();
		width -= scrollbarSize.x;

		TableColumn[] columns = table.getColumns();
		int count = columns.length;
		TableColumn column;
		int columnWidth;
		int cumulative = 0;

		for (int i = 1; i < count; i++) {
			column = columns[i];
			columnWidth = column.getWidth();
			cumulative += columnWidth;
		}

		int remainder = width - cumulative;
		return remainder;
	}
	
	protected ControlListener createTableControlListener(final Table table) {
		return new ControlAdapter() {
			public void controlResized(ControlEvent event) {
				ExportToSMLIFWizardPage3.this.handleTableResized(table);
			}
		};
	}
	
	protected void handleTableResized(Table table) {
		int width = calculateRemainingHorizontalSpace(table);
		if (width == 0)
			return;
		TableColumn[] columns = table.getColumns();
		TableColumn column = columns[0];
		column.setWidth(width);
	}
	
	protected void createRuleBindingsLabel(Composite parent) {
		Label label = new Label(parent, SWT.NONE);
		label.setText(SMLMessages.exportWizardRuleBindingsSectionLabel);
		GridData gridData = new GridData(
				GridData.GRAB_HORIZONTAL | GridData.FILL_HORIZONTAL);
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = 2;
		label.setLayoutData(gridData);
	}

	protected Collection<ISMLDocument> getModel() {
		return getExportWizard().getSelectedFiles().values();
	}

	protected ExportToSMLIFWizard getExportWizard() {
		return (ExportToSMLIFWizard) getWizard();
	}
	
	protected String[] getRuleListChoices() 
	{
		List<String> result = new ArrayList<String>();
		Collection<ISMLDocument> selectedFiles = getModel();
		for (Iterator iterator = selectedFiles.iterator(); iterator.hasNext();) {
			ISMLDocument file = (ISMLDocument) iterator.next();
			ISMLMetadata metadata = file.getMetadata();
			if (metadata.getDocumentType() == ISMLMetadata.DOCUMENT_TYPE_INSTANCE) 
			{
				continue;
			}
			
			String[] aliases = SMLCommonUtil.retrieveAliases(getExportWizard().getPage2().getAliases(), file);
			if (isRule(file)) 
			{
				if (aliases.length > 0)
				{
					// Just use the first alias to identify the rule
					result.add(aliases[0]);
				}
				else
				{					
					result.add(SMLRepositoryUtil.getDocumentName(file));
				}
			}
		}
		return result.toArray(new String[] {});
	}

	private boolean isRule(ISMLDocument file)
	{
		RuleFindingFileHandler handler = new RuleFindingFileHandler();
		SAXParserFactory saxParserFactory = SAXParserFactory.newInstance();
		saxParserFactory.setNamespaceAware(true);

		try 
		{
			saxParserFactory.setFeature(ISMLConstants.SAX_PARSER_FEATURE_NAMESPACES, true);
			saxParserFactory.setFeature(ISMLConstants.SAX_PARSER_FEATURE_NAMESPACE_PREFIXES, true);
			SAXParser parser = saxParserFactory.newSAXParser();
			parser.parse(file.getMetadata().getId(), handler);
		} 
		catch (SAXNotRecognizedException e) 
		{
			e.printStackTrace();
		} 
		catch (SAXNotSupportedException e) 
		{
			e.printStackTrace();
		} catch (ParserConfigurationException e) {
			e.printStackTrace();
		} catch (SAXException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}

		return handler.hasRule();
	}

	/* Answer the definition document represented by file has a schematron rule*/
	protected boolean definitionHasRule(IFile file) {
		RuleFindingFileHandler handler = new RuleFindingFileHandler();
		SAXParserFactory saxParserFactory = SAXParserFactory.newInstance();
		saxParserFactory.setNamespaceAware(true);

		try {
			saxParserFactory.setFeature(
					ISMLConstants.SAX_PARSER_FEATURE_NAMESPACES, true);
			saxParserFactory.setFeature(
					ISMLConstants.SAX_PARSER_FEATURE_NAMESPACE_PREFIXES, true);
			SAXParser parser = saxParserFactory.newSAXParser();
			parser.parse(file.getLocation().toString(), handler);
		} catch (SAXNotRecognizedException e) {
			e.printStackTrace();
		} catch (SAXNotSupportedException e) {
			e.printStackTrace();
		} catch (ParserConfigurationException e) {
			e.printStackTrace();
		} catch (SAXException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}

		return handler.hasRule();
	}

	public void setVisible(boolean visible) {
		super.setVisible(visible);
		// fix column widths
		handleTableResized(ruleBindingsTableViewer.getTable());
		
		if (visible)
		{
			ruleBindingsTableViewer.setInput(getRuleBindings());
			ruleBindingsTableViewer.refresh();
		}
	}

	public List<RuleBinding> getRuleBindings() {
		if (ruleBindings == null) {
			initRuleBindings();
		}
		return ruleBindings;
	}
	
	private void initRuleBindings()
	{
		ruleBindings = new ArrayList<RuleBinding>();
		
		// Copy the rule bindings to the rule bindings map
		// For every document		
		Collection<ISMLDocument> documents = getModel();
		for (Iterator iterator = documents.iterator(); iterator.hasNext();) {
			ISMLDocument currentDocument = (ISMLDocument)iterator.next(); 
			ISMLMetadata metadata = currentDocument.getMetadata();			
			Map<?, ?> boundRules = metadata.getBoundRules();
			
			if (boundRules == null)
				continue;
			
			for (Iterator<?> rules = boundRules.keySet().iterator(); rules.hasNext();)
			{
				String alias = (String) rules.next();				
				List<?> documentRules = (List<?>)boundRules.get(alias);
				
				for (int j = 0, documentRulesCount = documentRules.size(); j < documentRulesCount; j++)
				{
					RuleBinding ruleBinding = new RuleBinding();
//					ruleBinding.setDocument(currentDocument);
					ruleBinding.setAlias(alias);										
					ruleBinding.setRule((String)documentRules.get(j));
					ruleBindings.add(ruleBinding);
				}					
			}		
		}
	}

	public List<ISMLDocument> getDocuments() {
		return new ArrayList(getModel());
	}

}
