/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.databuilders;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

/**
 * This class provides two registries:
 * <ul>
 *  <li> Top level registry - Used to initially parse through the SML-IF document</li>
 *  <li> Instance level registry - Used to parse through instance documents </li>
 * </ul>
 * The methods getTopLevelRegistry() and getInstanceLevelRegistry() can be used
 * to retrieve each registry instance.
 * 
 * @author Sheldon Lee-Loy
 * @author Ali Mehregani
 */
public class DataBuilderRegistry
{
	/**
	 * Used to indicate two different modes: top level
	 * and instance level
	 */
	public static final int TOP_LEVEL_MODE = 0x00;
	public static final int INSTANCE_LEVEL_MODE = 0x01;
	
	/**
	 * Stores the data structure builders 
	 */
	protected Map<String, IDataBuilder<?>> dataStructureBuilders = new HashMap<String, IDataBuilder<?>>();

	/**
	 * The instance representing top level registry
	 */
	private static DataBuilderRegistry topLevelRegistry;
	
	/**
	 * The instance representing instance level registry
	 */
	private static DataBuilderRegistry instanceLevelRegistry;

	
	/**
	 * Retrieves the instance representing the top level registry
	 * 
	 * @return The top level registry
	 */
	public static DataBuilderRegistry getTopLevelRegistry()
	{
		if (topLevelRegistry == null)
		{
			topLevelRegistry = new DataBuilderRegistry();
		}
		return topLevelRegistry;

	}
	
	
	/**
	 * Retrieves the instance representing the instance level registry
	 * 
	 * @return The instance level registry
	 */
	public static DataBuilderRegistry getInstanceLevelRegistry()
	{
		if (instanceLevelRegistry == null)
		{
			instanceLevelRegistry = new DataBuilderRegistry();
		}
		return instanceLevelRegistry;

	}

	
	/**
	 * Register data structure builder, 'dataBuilder', with id: 'id'.
	 *  
	 * @param id The id of the data builder
	 * @param dataBuilder The data builder
	 */
	public void registerDataStructureBuilder(String id, IDataBuilder<?> dataBuilder)
	{
		dataStructureBuilders.put(id, dataBuilder);
	}

	
	/**
	 * Remove the data structure with the passed in id
	 * 
	 * @param id The id of the data structure builder to remove
	 */
	public void unRegisterDataStructureBuilder(String id)
	{
		dataStructureBuilders.remove(id);
	}

	
	/**
	 * Returns the data structure builders registered
	 * 
	 * @return The registered data structure builders
	 */
	public Collection<IDataBuilder<?>> getDataStructureBuilders()
	{
		return dataStructureBuilders.values();
	}

	
	/**
	 * Returns the data structure builder with the passed in id
	 * 
	 * @param id The id of a data structure builder
	 * @return The data structure builder corresponding to 'id'
	 */
	public IDataBuilder<?> getDataStructureBuilder(String id)
	{
		return (IDataBuilder<?>) dataStructureBuilders.get(id);
	}
	
	
	/**
	 * Clears the registry 
	 */
	public void clear()
	{
		dataStructureBuilders.clear();
	}
}
