/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.databuilders;

import java.net.URI;
import java.net.URISyntaxException;

import org.eclipse.cosmos.rm.internal.validation.SMLActivator;
import org.eclipse.cosmos.rm.internal.validation.common.ISMLConstants;
import org.eclipse.cosmos.rm.internal.validation.common.SMLIFIdentity;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidationMessages;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidatorUtil;
import org.eclipse.cosmos.rm.internal.validation.common.AbstractValidationOutput.ValidationMessageFactory;
import org.eclipse.cosmos.rm.internal.validation.core.IFoundationBuilder;
import org.eclipse.osgi.util.NLS;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;

/**
 * This databuilder constructs an SMLIFIdentity object which will include
 * the content that is included in the identity section of an SML-IF 
 * document.
 * 
 * @author Ali Mehregani
 */
public class IdentityDataBuilder extends AbstractDataBuilder<SMLIFIdentity>
{
	/**
	 * The ID of this builder
	 */
	public static final String ID = SMLActivator.PLUGIN_ID + ".IdentityDataBuilder";
	
	
	/**
	 * The sml-if identity object
	 */
	private SMLIFIdentity smlIfIdentity;
	
	/**
	 * Flags used to indicate the start of an element
	 */
	private boolean identityElementHit,					// Identity element hit
					nameElementHit,						// Name element hit
					descriptionElementHit,				// Description element hit
					baseURIHit,							// Base URI element hit
					xmlBaseHit;							// xml:base attribute hit
	
	/**
	 * The name stored in the identity section
	 */
	private String name;

	/**
	 * The description stored in the identity section
	 */
	private String description;

	/**
	 * The base URI stored in the identity section
	 */
	private String baseURI;
	
	
	/**
	 * Constructor
	 */
	public IdentityDataBuilder()
	{
		smlIfIdentity = new SMLIFIdentity();		
		name = "";
		description = "";
		baseURI = "";
		super.addEvent(IFoundationBuilder.EVENT_CHARACTER);
	}
	
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.AbstractDataBuilder#startElement(java.lang.String, java.lang.String, java.lang.String, org.xml.sax.Attributes)
	 */
	public void startElement(String uri, String localName, String qName, Attributes attributes) throws SAXException
	{
		elementNotification (uri, localName, attributes, true);
	}
	
	
	/**
	 * @see org.xml.sax.helpers.DefaultHandler#characters(char[], int, int)
	 */
	public void characters(char[] ch, int start, int length) throws SAXException
	{
		if (nameElementHit)
		{
			name += new String(ch, start, length);
		}
		else if (descriptionElementHit)
		{
			description += new String(ch, start, length);
		}	
		else if (baseURIHit && !xmlBaseHit)
		{
			baseURI += new String(ch, start, length);
		}
	}
	
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.AbstractDataBuilder#endElement(java.lang.String, java.lang.String, java.lang.String)
	 */
	public void endElement(String uri, String localName, String qName) throws SAXException
	{
		elementNotification (uri, localName, null, false);		
	}

	private void validateBaseURI() {
		try
		{						
			baseURI = baseURI.trim();
			URI base = new URI(baseURI);
			
			// The base uri should be absolute
			if (!base.isAbsolute())
			{
				getMessageOutputter().reportMessage(ValidationMessageFactory.createErrorMessage(
						getLocator().getLineNumber(), NLS.bind(SMLValidationMessages.errorBaseURIRelativeRef, baseURI)));
			}
			// The base uri should not have a fragment
			else if (base.getFragment() != null)
			{
				getMessageOutputter().reportMessage(ValidationMessageFactory.createErrorMessage(
						getLocator().getLineNumber(), SMLValidationMessages.errorBadSyntaxURI));
			}
			else
			{
				smlIfIdentity.setBaseURI(baseURI);
			}
			baseURI = "";
		} 
		catch (URISyntaxException e)
		{
			// The URI is not valid
			getMessageOutputter().reportMessage(ValidationMessageFactory.createErrorMessage(
					getLocator().getLineNumber(), SMLValidationMessages.errorBadBaseURI));
			
		}
	}

	
	private void elementNotification (String uri, String localName, Attributes attributes, boolean start)
	{
		if (!ISMLConstants.SMLIF_URI.equals(uri))
			return;

		if (ISMLConstants.MODEL_ELEMENT.equals(localName))
		{
			if (start) {
				String xmlBase = attributes.getValue(ISMLConstants.XML_BASE_ATTRIBUTE);
				if (xmlBase != null) {
					baseURI = xmlBase;
					xmlBaseHit = true;
				}
			}
		} else if (ISMLConstants.IDENTITY_ELEMENT.equals(localName)) {
			identityElementHit = start;
			if (!start && !("".equals(baseURI))) {
				validateBaseURI();
			}
		}
		else if (identityElementHit)
		{
			if (ISMLConstants.NAME_ELEMENT.equals(localName))
			{
				nameElementHit = start;
				if (!start)
					smlIfIdentity.setName(name);
			}
			else if (ISMLConstants.DESCRIPTION_ELEMENT.equals(localName))
			{
				descriptionElementHit = start;
				if (!start)
					smlIfIdentity.setDescription(description);
			}
			else if (ISMLConstants.BASE_URI_ELEMENT.equals(localName))
			{
				baseURIHit = start;
			}
		}

	}
	
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.IDataBuilder#getDataStructure()
	 */
	public SMLIFIdentity getDataStructure()
	{
		return smlIfIdentity;
	}
	
	/**
	 * Retrieves and returns the base URI
	 * @return the base URI
	 */
	public static String retrieveBaseURI()
	{		
		SMLIFIdentity smlIdentity = (SMLIFIdentity)SMLValidatorUtil.retrieveDataStructure(DataBuilderRegistry.TOP_LEVEL_MODE, ID);
		return smlIdentity == null ? null : smlIdentity.getBaseURI();
	}
}
