/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.cosmos.dc.internal.cmdbf.services.transform.artifacts;

import java.io.StringWriter;
import java.net.URI;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.eclipse.cosmos.dc.internal.cmdbf.services.CMDBfInternalUtility.XMLWritableString;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.common.CMDBfServicesUtil;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.common.ICMDBfServicesConstants;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.common.IXMLWritable;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.transform.artifacts.IGraphElement;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.transform.artifacts.IRecord;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.transform.artifacts.IRecordMetadata;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.transform.artifacts.IRecordSerializer;

/**
 * @author David Whiteman
 */
public class Record implements IRecord, IXMLWritable 
{
	private IXMLWritable value;
	private IRecordMetadata recordMetadata;
	private Map<String,String> namespaces = new HashMap<String, String>();
		
	/**
	 * The parent result entity that owns this record
	 */
	private IGraphElement parent;
	
	public Record (IGraphElement parent)
	{
		this.parent = parent;
	}
	
	public IXMLWritable getValue() {
		return value;
	}

	public void setValue(IXMLWritable value) {
		this.value = value;
	}

	public void addNamespace(String key, String value) {
		getNamespaces().put(key, value);
	}

	public IRecordMetadata getRecordMetadata() {
		return recordMetadata;
	}

	public void setRecordMetadata(IRecordMetadata recordMetadata) {
		this.recordMetadata = recordMetadata;
	}

	public Map<String, String> getNamespaces() {
		return namespaces;
	}

	public void toXML(StringWriter writer, int indent) {
		CMDBfServicesUtil.addIndent(writer, indent);
		writer.write(ICMDBfServicesConstants.OPEN_ANGLE_BRACKET+ICMDBfServicesConstants.CMDBF_PREFIX +ICMDBfServicesConstants.RECORD_ELEMENT);
		
		int count = 0;
		for (Iterator<String> iterator = getNamespaces().keySet().iterator(); iterator.hasNext();) {
			String key = iterator.next();
			if (count == 0) {
				writer.write(ICMDBfServicesConstants.SINGLE_SPACE);
			} else {
				CMDBfServicesUtil.addIndent(writer, indent+1);
			}
			count++;
			
			String attributeName = key.length() <= 0 ? ICMDBfServicesConstants.NAMESPACE_PREFIX : ICMDBfServicesConstants.NAMESPACE_PREFIX+ICMDBfServicesConstants.COLON+key;
			CMDBfServicesUtil.writeAttribute(writer, attributeName, (String) getNamespaces().get(key));
			// avoid extra space before the final '>'
			if (count < getNamespaces().size()) {
				writer.write(ICMDBfServicesConstants.SINGLE_SPACE);
			}
		}
		writer.write(ICMDBfServicesConstants.CLOSE_ANGLE_BRACKET+ICMDBfServicesConstants.nl);

		getValue().toXML(writer, indent+1);
		getRecordMetadata().toXML(writer, indent+1);
		
		CMDBfServicesUtil.addIndent(writer, indent);
		writer.write(CMDBfServicesUtil.endTagFor(ICMDBfServicesConstants.CMDBF_PREFIX +ICMDBfServicesConstants.RECORD_ELEMENT)+ICMDBfServicesConstants.nl);
	}


	public void setValueFromString(String stringRepresentationOfValue) {
		IRecordSerializer serializer = CMDBfServicesUtil.getRecordSerializer();
		IXMLWritable writable = null;
		if (serializer != null) {
			writable = serializer.deserialize(stringRepresentationOfValue);
		} else {
			writable = new XMLWritableString(stringRepresentationOfValue);
		}
		setValue(writable);
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result
				+ ((namespaces == null) ? 0 : namespaces.hashCode());
		result = prime * result
				+ ((getRecordMetadata().getRecordId() == null) ? 0 : getRecordMetadata().getRecordId().hashCode());
		result = prime * result
				+ ((recordMetadata == null) ? 0 : recordMetadata.hashCode());
		result = prime * result + ((value == null) ? 0 : value.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		final Record other = (Record) obj;
		if (namespaces == null) {
			if (other.namespaces != null)
				return false;
		} else if (!namespaces.equals(other.namespaces))
			return false;
		if (parent == null) {
			if (other.parent != null)
				return false;
		} 
		
		URI recordId = getRecordMetadata().getRecordId();
		URI otherRecordId = other.getRecordMetadata().getRecordId();
		if (recordId == null) {
			if (otherRecordId != null)
				return false;
		} else if (!recordId.equals(otherRecordId))
			return false;
		if (recordMetadata == null) {
			if (other.recordMetadata != null)
				return false;
		} else if (!recordMetadata.equals(other.recordMetadata))
			return false;
		if (value == null) {
			if (other.value != null)
				return false;
		} else if (!value.equals(other.value))
			return false;
		return true;
	}

}
